﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <shared_mutex>
#include <nn/ns/ns_Result.h>
#include <nn/ns/srv/ns_BlackListManager.h>
#include <nn/fs/fs_SystemDataUpdateEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include "ns_DebugUtil.h"

namespace nn { namespace ns { namespace srv {

namespace {

    typedef std::lock_guard<os::ReaderWriterLock> WriterLock;
    typedef std::shared_lock<os::ReaderWriterLock> ReaderLock;

} // ~namespace nn::ns::srv::<anonymous>


BlackListManager::BlackListManager() NN_NOEXCEPT :
    m_IsCached(false)
{
}

Result BlackListManager::Initialize() NN_NOEXCEPT
{
    NN_RESULT_DO(fs::OpenSystemDataUpdateEventNotifier(&m_Notifier));
    NN_RESULT_DO(m_Notifier->BindEvent(&m_UpdateEvent, os::EventClearMode_AutoClear));

    NN_RESULT_SUCCESS;
}

bool BlackListManager::IsApplicationOnBlackList(ncm::ApplicationId id, uint32_t version) NN_NOEXCEPT
{
    if (!IsCached())
    {
        ReadBlackList();
    }
    return IsRefusedForDebug(id, version) || IsRefusedByBlackList(id, version);
}

bool BlackListManager::IsCached() const NN_NOEXCEPT
{
    ReaderLock lock(m_CacheLock);
    return m_IsCached && !os::TryWaitSystemEvent(&m_UpdateEvent);
}

void BlackListManager::ReadBlackList() NN_NOEXCEPT
{
    WriterLock lock(m_CacheLock);
    BlackListReader reader;
    reader.Mount();
    reader.Read(&m_Cache);
    m_IsCached = true;
}

bool BlackListManager::IsRefusedForDebug(ncm::ApplicationId id, uint32_t version) const NN_NOEXCEPT
{
    ncm::ApplicationId debugId;
    uint32_t debugVersion;
    GetApplicationInfoOnBlackListForDebug(&debugId, &debugVersion);
    return (debugId == id) && (debugVersion >= version);
}

bool BlackListManager::IsRefusedByBlackList(ncm::ApplicationId id, uint32_t version) const NN_NOEXCEPT
{
    ReaderLock lock(m_CacheLock);
    const BlackListRecord* begin = m_Cache.records;
    const BlackListRecord* end = begin + m_Cache.count;
    auto record = std::lower_bound(begin, end, id,
        [](const BlackListRecord& lhs, const ncm::ApplicationId& rhs) NN_NOEXCEPT
        {
            return lhs.ToValue64() < rhs.value;
        });
    return (record != end && record->ToValue64() == id.value) ? record->version >= version : false;
}

}}} // ~namespace nn::ns::srv
