﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/fs/fs_Result.h>

#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentStorage.h>

#include <nn/ns/ns_AsyncProgress.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_ApplicationRecordDatabase.h>

#include <nn/os/os_Thread.h>
#include <nn/os/os_TransferMemory.h>

#include <nn/util/util_ScopeExit.h>

#include "ns_AsyncVerifyContentsImpl.h"
#include "ns_AsyncThreadAllocator.h"

namespace nn { namespace ns { namespace srv {
    namespace {
        ncm::StorageContentMetaKey g_KeyList[MaxContentMetaCountPerApplication];
        NonRecursiveMutex g_KeyLock;

        bool IsInstalledStorage(ncm::StorageId storageId) NN_NOEXCEPT
        {
            return storageId != ncm::StorageId::None && storageId != ncm::StorageId::Any && storageId != ncm::StorageId::Card;
        }

        template <class FuncT>
        Result TraverseApplicationRecordList(
            const ncm::StorageContentMetaKey* list,
            int listLength,
            const IntegratedContentManager* pIntegrated,
            const ncm::StorageId* pStorageIds,
            int numStorage,
            os::Event* pEvent,
            FuncT func) NN_NOEXCEPT
        {
            for (int i = 0; i < listLength; i++)
            {
                const auto& key = list[i].key;

                for (int j = 0; j < numStorage; j++)
                {
                    if (IsInstalledStorage(pStorageIds[j]))
                    {
                        bool hasContentMeta = false;
                        NN_RESULT_DO(pIntegrated->HasContentMetaKey(&hasContentMeta, key, pStorageIds[j]));
                        if (hasContentMeta)
                        {
                            NN_RESULT_DO(func(key, pStorageIds[j]));
                            NN_RESULT_THROW_UNLESS(!pEvent->TryWait(), ResultCanceled());
                        }
                    }
                }
            }

            NN_RESULT_SUCCESS;
        }

        template <class FuncT>
        Result TraverseContentInfo(const ncm::ContentMetaKey& key, const IntegratedContentManager* pIntegrated, ncm::StorageId storageId, os::Event* pEvent, FuncT func) NN_NOEXCEPT
        {
            ncm::ContentInfo contentInfo[16];
            const int NumInfo = static_cast<int>(NN_ARRAY_SIZE(contentInfo));
            int outCount = NumInfo;

            for (int fileCount = 0; outCount == NumInfo; fileCount += outCount)
            {
                NN_RESULT_DO(pIntegrated->ListContentInfo(&outCount, contentInfo, NumInfo, key, fileCount, storageId));

                for (int i = 0; i < outCount; i++)
                {
                    NN_RESULT_DO(func(contentInfo[i]));
                    NN_RESULT_THROW_UNLESS(!pEvent->TryWait(), ResultCanceled());
                }
            }

            NN_RESULT_SUCCESS;
        }

        Result ConvertToResultSdCardDataCorrupted(Result result, ncm::StorageId storageId) NN_NOEXCEPT
        {
            if (storageId == ncm::StorageId::SdCard &&
                (fs::ResultDataCorrupted::Includes(result) || fs::ResultUnexpected::Includes(result)))
            {
                NN_RESULT_THROW(ResultSdCardDataCorrupted());
            }

            NN_RESULT_THROW(result);
        }

        Result ConvertToResultApplicationVerificationFailed(Result result) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(result.IsSuccess() || ResultContentCorrupted::Includes(result) || ResultSdCardDataCorrupted::Includes(result), ResultVerifyContentUnexpected());
            NN_RESULT_THROW(result);
        }

        Result ConvertToResultAddOnContentsVerificationFailed(Result result) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(result.IsSuccess() || ResultApplicationLaunchRightsNotFound::Includes(result), ResultVerifyAddOnContentsRightsUnexpected());
            NN_RESULT_THROW(result);
        }

    } // namespace

    AsyncVerifyApplicationImpl::~AsyncVerifyApplicationImpl() NN_NOEXCEPT
    {
        if (m_ThreadInfo)
        {
            CancelImpl();
            os::WaitThread(m_ThreadInfo->thread);
            os::DestroyThread(m_ThreadInfo->thread);
            GetAsyncThreadAllocator()->Free(*m_ThreadInfo);
        }

        if (m_Transfer)
        {
            m_Transfer->Unmap();
        }
    }

    Result AsyncVerifyApplicationImpl::Initialize(ncm::ApplicationId appId, VerifyContentFlag flags, ApplicationRecordDatabase* pRecordDb, const IntegratedContentManager* pIntegrated, RequestServer::ManagedStop&& stopper, sf::NativeHandle transferHandle, size_t size) NN_NOEXCEPT
    {
        m_CurrentOffset = 0;
        m_SumSize = 0;
        m_AppId = appId;
        m_Flags = flags;
        m_pRecordDb = pRecordDb;
        m_pIntegrated = pIntegrated;
        m_Stopper = std::move(stopper);
        m_BufferSize = size;

        m_Transfer.emplace();
        m_Transfer->Attach(size, transferHandle.GetOsHandle(), transferHandle.IsManaged());
        transferHandle.Detach();
        NN_RESULT_DO(m_Transfer->Map(&m_Buffer, os::MemoryPermission_None));

        NN_RESULT_SUCCESS;
    }

    Result AsyncVerifyApplicationImpl::Run() NN_NOEXCEPT
    {
        ThreadInfo info;
        NN_RESULT_DO(GetAsyncThreadAllocator()->Allocate(&info));
        bool isSuccess = false;
        NN_UTIL_SCOPE_EXIT
        {
            if (!isSuccess)
            {
                GetAsyncThreadAllocator()->Free(info);
            }
        };
        info.priority = NN_SYSTEM_THREAD_PRIORITY(nssrv, CheckApplicationCorruptionTask);
        NN_RESULT_DO(os::CreateThread(info.thread, [](void* p)
            {
                auto t = reinterpret_cast<AsyncVerifyApplicationImpl*>(p);
                t->m_DetailResult = t->Execute();
                t->m_Result = ConvertToResultApplicationVerificationFailed(t->m_DetailResult);
                t->m_Event.Signal();
            }, this, info.stack, info.stackSize, info.priority));
        os::SetThreadNamePointer(info.thread, NN_SYSTEM_THREAD_NAME(nssrv, CheckApplicationCorruptionTask));
        os::StartThread(info.thread);
        m_ThreadInfo = info;
        isSuccess = true;

        NN_RESULT_SUCCESS;
    }

    // アプリケーションレコードから取得したコンテンツメタから検索できるコンテンツをすべてチェックする
    // ユーザーがインストール可能でかつ現在利用可能なストレージはすべて検索する
    Result AsyncVerifyApplicationImpl::Execute() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(g_KeyLock);

        int listCount = 0;
        {
            ApplicationRecordAccessor list;
            NN_RESULT_DO(m_pRecordDb->Open(&list, m_AppId));
            NN_SDK_ASSERT_LESS_EQUAL(list.Count(), static_cast<int>(NN_ARRAY_SIZE(g_KeyList)));

            for (auto i = 0; i < list.Count(); i++)
            {
                if((list[i].key.type == ncm::ContentMetaType::Application && m_Flags.Test<VerifyContentFlag_Application>())
                    || (list[i].key.type == ncm::ContentMetaType::AddOnContent && m_Flags.Test<VerifyContentFlag_AddOnContent>())
                    || (list[i].key.type == ncm::ContentMetaType::Patch && m_Flags.Test<VerifyContentFlag_Patch>()))
                {
                    g_KeyList[listCount] = list[i];
                    listCount++;
                }
            }
        }

        ncm::StorageId enabledStorageIds[IntegratedContentManager::MaxStorageCount];
        auto numStorage = m_pIntegrated->GetRegisteredStorages(enabledStorageIds, IntegratedContentManager::MaxStorageCount);

        // TORIAEZU: 進捗を表現するためにコンテンツファイルの総容量を計算する
        //           ストレージごとに検査するため、ファイルが重複していることもある
        int64_t sumSize = 0;
        NN_RESULT_DO(TraverseApplicationRecordList(g_KeyList, listCount, m_pIntegrated, enabledStorageIds, numStorage, &m_EndEvent,
            [this, &sumSize](const ncm::ContentMetaKey& key, ncm::StorageId storageId) -> Result
            {
                NN_RESULT_DO(ConvertToResultSdCardDataCorrupted(TraverseContentInfo(key, m_pIntegrated, storageId, &m_EndEvent,
                    [&sumSize](ncm::ContentInfo info) -> Result
                    {
                        sumSize += info.GetSize();
                        NN_RESULT_SUCCESS;
                    }), storageId));
                NN_RESULT_SUCCESS;
            }));

        SetSumSize(sumSize);

        // 各コンテンツのハッシュが meta ファイルに載っているハッシュと等しいかを各ストレージごとに調べる
        NN_RESULT_DO(TraverseApplicationRecordList(g_KeyList, listCount, m_pIntegrated, enabledStorageIds, numStorage, &m_EndEvent,
            [this](const ncm::ContentMetaKey& key, ncm::StorageId storageId) -> Result
            {
                ncm::ContentId metaId;
                NN_RESULT_DO(m_pIntegrated->GetContentIdByStorageId(&metaId, key, ncm::ContentType::Meta, 0, storageId));

                NN_RESULT_DO(ConvertToResultSdCardDataCorrupted(TraverseContentInfo(key, m_pIntegrated, storageId, &m_EndEvent,
                    [this, &metaId, &storageId](ncm::ContentInfo info) -> Result
                    {
                        if (info.GetType() == ncm::ContentType::Meta)
                        {
                            AddCurrentOffset(info.GetSize());
                            NN_RESULT_SUCCESS;
                        }

                        auto contentId = info.GetId();
                        ncm::Hash fileHash;
                        NN_RESULT_DO(CalcContentFileHash(&fileHash, contentId, storageId));

                        ncm::Hash metaHash;
                        NN_RESULT_DO(m_pIntegrated->GetContentIdFileHash(&metaHash, contentId, metaId, storageId));

                        NN_RESULT_THROW_UNLESS(std::memcmp(fileHash.data, metaHash.data, sizeof(fileHash.data)) == 0, ResultContentCorrupted());

                        NN_RESULT_SUCCESS;
                    }), storageId));

                NN_RESULT_SUCCESS;
            }));

        NN_RESULT_SUCCESS;
    }

    Result AsyncVerifyApplicationImpl::GetImpl() NN_NOEXCEPT
    {
        return m_Result;
    }

    void AsyncVerifyApplicationImpl::CancelImpl() NN_NOEXCEPT
    {
        m_EndEvent.Signal();
    }

    Result AsyncVerifyApplicationImpl::GetProgressImpl(const nn::sf::OutBuffer& outBuffer) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_StateLock);

        AsyncVerifyApplicationProgress progress;
        NN_RESULT_THROW_UNLESS(outBuffer.GetSize() >= sizeof(progress), ResultBufferNotEnough());
        progress.currentSize = m_CurrentOffset;
        progress.totalSize = m_SumSize;

        std::memcpy(outBuffer.GetPointerUnsafe(), &progress, sizeof(progress));

        NN_RESULT_SUCCESS;
    }

    Result AsyncVerifyApplicationImpl::GetDetailResultImpl() NN_NOEXCEPT
    {
        return m_DetailResult;
    }

    void AsyncVerifyApplicationImpl::SetSumSize(int64_t size) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_StateLock);
        m_SumSize = size;
    }

    void AsyncVerifyApplicationImpl::AddCurrentOffset(int64_t offset) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_StateLock);
        m_CurrentOffset += offset;
    }

    Result AsyncVerifyApplicationImpl::CalcContentFileHash(ncm::Hash* outValue, const nn::ncm::ContentId& contentId, ncm::StorageId storageId) NN_NOEXCEPT
    {
        int64_t fileSize;
        NN_RESULT_DO(m_pIntegrated->GetContentIdFileSize(&fileSize, contentId, storageId));

        crypto::Sha256Generator sha256;
        sha256.Initialize();

        for (int64_t offset = 0; offset < fileSize;)
        {
            NN_RESULT_THROW_UNLESS(!m_EndEvent.TryWait(), ResultCanceled());
            size_t readSize = m_BufferSize;
            // int64_t を超えるようなサイズが来ることはないはずなので、チェックを省く
            if (static_cast<int64_t>(offset + readSize) > fileSize)
            {
                readSize = static_cast<size_t>(fileSize - offset);
            }

            NN_RESULT_DO(m_pIntegrated->ReadContentIdFile(m_Buffer, readSize, contentId, offset, storageId));
            sha256.Update(m_Buffer, readSize);
            offset += readSize;

            // ほとんどがファイル読み込みによるオーバーヘッドなので、
            // 進捗はファイルの読み込み具合として計算する
            AddCurrentOffset(readSize);
        }

        sha256.GetHash(outValue, sizeof(*outValue));

        NN_RESULT_SUCCESS;
    }

    AsyncVerifyAddOnContentsImpl::~AsyncVerifyAddOnContentsImpl() NN_NOEXCEPT
    {
        if (m_ThreadInfo)
        {
            CancelImpl();
            os::WaitThread(m_ThreadInfo->thread);
            os::DestroyThread(m_ThreadInfo->thread);
            GetAsyncThreadAllocator()->Free(*m_ThreadInfo);
        }
    }

    Result AsyncVerifyAddOnContentsImpl::Initialize(ncm::ApplicationId appId, IntegratedContentManager* pIntegrated, ApplicationRecordDatabase* pRecordDb, TicketManager* pTicketManager) NN_NOEXCEPT
    {
        m_Current = 0;
        m_Total = 0;
        m_AppId = appId;
        m_pIntegrated = pIntegrated;
        m_pRecordDb = pRecordDb;
        m_pTicketManager = pTicketManager;

        NN_RESULT_SUCCESS;
    }

    Result AsyncVerifyAddOnContentsImpl::Run() NN_NOEXCEPT
    {
        ThreadInfo info;
        NN_RESULT_DO(GetAsyncThreadAllocator()->Allocate(&info));
        bool isSuccess = false;
        NN_UTIL_SCOPE_EXIT
        {
            if (!isSuccess)
            {
                GetAsyncThreadAllocator()->Free(info);
            }
        };
        info.priority = NN_SYSTEM_THREAD_PRIORITY(nssrv, VerifyAddOnContentsRightsTask);
        NN_RESULT_DO(os::CreateThread(info.thread, [](void* p)
            {
                auto t = reinterpret_cast<AsyncVerifyAddOnContentsImpl*>(p);
                t->m_DetailResult = t->Execute();
                t->m_Result = ConvertToResultAddOnContentsVerificationFailed(t->m_DetailResult);
                t->m_Event.Signal();
            }, this, info.stack, info.stackSize, info.priority));
        os::SetThreadNamePointer(info.thread, NN_SYSTEM_THREAD_NAME(nssrv, VerifyAddOnContentsRightsTask));
        os::StartThread(info.thread);
        m_ThreadInfo = info;
        isSuccess = true;

        NN_RESULT_SUCCESS;
    }

    Result AsyncVerifyAddOnContentsImpl::Execute() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(g_KeyLock);
        int listCount;
        m_Total = 0;
        {
            std::lock_guard<os::Mutex> stateLock(m_StateLock);
            ApplicationRecordAccessor list;
            NN_RESULT_DO(m_pRecordDb->Open(&list, m_AppId));
            listCount = list.Count();
            NN_SDK_ASSERT_LESS_EQUAL(listCount, static_cast<int>(NN_ARRAY_SIZE(g_KeyList)));

            for (auto i = 0; i < listCount; i++)
            {
                if (list.HasInstalled(list[i].key) && list[i].key.type == ncm::ContentMetaType::AddOnContent)
                {
                    bool hasKey;
                    NN_RESULT_DO(m_pIntegrated->HasContentMetaKey(&hasKey, list[i].key));
                    if (hasKey)
                    {
                        g_KeyList[m_Total++] = list[i];
                    }
                }
            }
        }

        for (m_Current = 0; m_Current < m_Total;)
        {
            NN_RESULT_THROW_UNLESS(!m_EndEvent.TryWait(), ResultCanceled());

            uint8_t programIndex;
            NN_RESULT_DO(m_pIntegrated->GetMinimumProgramIndex(&programIndex, g_KeyList[m_Current].key));
            NN_RESULT_TRY(m_pTicketManager->VerifyContentRights(g_KeyList[m_Current].key, ncm::ContentType::Data, programIndex))
                NN_RESULT_CATCH(fs::ResultDataCorrupted)
                {
                    m_pRecordDb->SetApplicationTerminateResult(m_AppId, NN_RESULT_CURRENT_RESULT);
                }
            NN_RESULT_END_TRY;

            {
                std::lock_guard<os::Mutex> stateLock(m_StateLock);
                m_Current++;
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result AsyncVerifyAddOnContentsImpl::GetImpl() NN_NOEXCEPT
    {
        return m_Result;
    }

    void AsyncVerifyAddOnContentsImpl::CancelImpl() NN_NOEXCEPT
    {
        m_EndEvent.Signal();
    }

    Result AsyncVerifyAddOnContentsImpl::GetProgressImpl(const nn::sf::OutBuffer& outBuffer) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_StateLock);

        AsyncVerifyAddOnContentsProgress progress;
        NN_RESULT_THROW_UNLESS(outBuffer.GetSize() >= sizeof(progress), ResultBufferNotEnough());
        progress.current = m_Current;
        progress.total = m_Total;

        std::memcpy(outBuffer.GetPointerUnsafe(), &progress, sizeof(progress));

        NN_RESULT_SUCCESS;
    }

    Result AsyncVerifyAddOnContentsImpl::GetDetailResultImpl() NN_NOEXCEPT
    {
        return m_DetailResult;
    }
}}} // namespace nn::ns::srv
