﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bgtc.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/srv/ns_AsyncTaskManager.h>
#include <nn/ns/srv/ns_OsUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/os/os_Random.h>
#include <nn/util/util_TinyMt.h>

#include "ns_NotificationToken.h"
#include "ns_SystemUpdateTopic.h"

namespace nn { namespace ns { namespace srv {
    namespace
    {
#if !defined(NN_SDK_BUILD_RELEASE)
        const char NotificationTypeName[][30] = {
            "SystemUpdate",
            "DownloadTaskList",
            "VersionList",
            "AutoUpdate",
            "ETicketAvailable",
            "SendRightsUsageStatusRequest",
            "SyncELicenseRequest",
        };
        NN_STATIC_ASSERT(sizeof(NotificationTypeName) == sizeof(NotificationTypeName[0]) * static_cast<int>(AsyncTask::AsyncTaskType::NumberOfAsyncTaskType));

        const char* GetNotificationTypeName(AsyncTask::AsyncTaskType type) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(static_cast<int>(type) >= 0 && static_cast<int>(type) < static_cast<int>(AsyncTask::AsyncTaskType::NumberOfAsyncTaskType));
            return NotificationTypeName[static_cast<int>(type)];
        }
#endif

        int64_t CalcScheduleTime(int64_t limit, int64_t registeredTime, os::Tick currentTick) NN_NOEXCEPT
        {
            os::Tick registeredTick(registeredTime);
            auto elapsed = (currentTick - registeredTick).ToTimeSpan().GetSeconds();
            return elapsed > limit ? 0 : limit - elapsed;
        }

        AsyncTaskOpt* GetForemostTask(AsyncTaskManager::AsyncTaskList& infoList, os::Tick currentTick) NN_NOEXCEPT
        {
            auto begin = infoList.begin();
            auto end = infoList.end();
            auto foremost = std::min_element(begin, end, [&](AsyncTaskOpt& a, AsyncTaskOpt& b) NN_NOEXCEPT -> bool
            {
                // a == nullopt なら最小値として扱わない
                if (!a)
                {
                    return false;
                }

                // a != nullopt && b == nullopt なら a を最小値として扱う
                if (!b)
                {
                    return true;
                }

                // a != nullopt && b != nullopt なら a, b からスケジュールタイム sa, sb を計算し、sa < sb であれば、a を最小値として扱う
                auto scheduleTimeA = CalcScheduleTime(a->waitingLimit, a->registeredTime, currentTick);
                auto scheduleTimeB = CalcScheduleTime(b->waitingLimit, b->registeredTime, currentTick);

                return scheduleTimeA < scheduleTimeB;
            });

            // 全部 nullopt の場合 infoList[0] (nullopt) を返す
            if (foremost == end)
            {
                return &(*begin);
            }

            return &(*foremost);
        }

        Result PerformTask(AsyncTaskManager::NotificationHandlerHolder& handlerHolderRoot, const NotificationInfo& info, AsyncTask::AsyncTaskType taskType) NN_NOEXCEPT
        {
            for (auto handlerHolder = handlerHolderRoot.next; handlerHolder != nullptr; handlerHolder = handlerHolder->next)
            {
                auto handlerType = handlerHolder->notificationType;
                if (handlerType == taskType)
                {
                    NN_NM_TRACE("Call notification handler (for %s)\n", GetNotificationTypeName(taskType));
                    auto result = handlerHolder->handler(info);
                    if (result.IsFailure())
                    {
                        NN_NM_TRACE("Notification handler result: %08x\n", result.GetInnerValueForDebug());
                    }
                }
            }

            NN_RESULT_SUCCESS;
        }

        bool HasDuplicatedInfo(AsyncTaskManager::AsyncTaskList& infoList, const NotificationInfo& newInfo) NN_NOEXCEPT
        {
            auto begin = infoList.begin();
            auto end = infoList.end();

            auto found = std::any_of(begin, end, [&](AsyncTaskOpt& a) NN_NOEXCEPT -> bool
            {
                if (!a)
                {
                    return false;
                }

                return a->info.type == newInfo.type && a->info.eTag == newInfo.eTag;
            });

            return found;
        }

        AsyncTaskOpt* FindTaskByType(AsyncTaskManager::AsyncTaskList& infoList, AsyncTask::AsyncTaskType type) NN_NOEXCEPT
        {
            auto begin = infoList.begin();
            auto end = infoList.end();
            auto task = std::find_if(begin, end, [&](const AsyncTaskOpt& task) NN_NOEXCEPT -> bool
            {
                if (!task)
                {
                    return false;
                }
                return task->taskType == type;
            });

            return task != end ? &(*task) : nullptr;
        }

        AsyncTaskOpt* FindEmptyTask(AsyncTaskManager::AsyncTaskList& infoList) NN_NOEXCEPT
        {
            auto begin = infoList.begin();
            auto end = infoList.end();
            auto task = std::find_if(begin, end, [&](const AsyncTaskOpt& task) NN_NOEXCEPT -> bool
            {
                return !task;
            });

            return task != end ? &(*task) : nullptr;
        }

        AsyncTask::AsyncTaskType GetAsyncTaskFromNotificationInfo(const NotificationInfo& info) NN_NOEXCEPT
        {
            switch(info.type)
            {
                case NotificationInfo::NotificationType::SystemUpdate:
                    return AsyncTask::AsyncTaskType::SystemUpdate;
                case NotificationInfo::NotificationType::DownloadTaskList:
                    return AsyncTask::AsyncTaskType::DownloadTaskList;
                case NotificationInfo::NotificationType::VersionList:
                    return AsyncTask::AsyncTaskType::VersionList;
                case NotificationInfo::NotificationType::ETicketAvailable:
                    return AsyncTask::AsyncTaskType::ETicketAvailable;
                case NotificationInfo::NotificationType::SendRightsUsageStatusRequest:
                    return AsyncTask::AsyncTaskType::SendRightsUsageStatusRequest;
                case NotificationInfo::NotificationType::SyncELicenseRequest:
                    return AsyncTask::AsyncTaskType::SyncELicenseRequest;

                default:
                    NN_UNEXPECTED_DEFAULT;
            }
        }

    } // namespace

    AsyncTaskManager::Config AsyncTaskManager::MakeDefaultConfig() NN_NOEXCEPT
    {
        Config config = {};
        const char* Name = "ns.notification";

        nn::settings::fwdbg::GetSettingsItemValue(
            &config.isNetworkUpdateEnabled, sizeof(config.isNetworkUpdateEnabled), Name, "enable_network_update");

        nn::settings::fwdbg::GetSettingsItemValue(
            &config.isDownloadTaskListEnabled, sizeof(config.isDownloadTaskListEnabled), Name, "enable_download_task_list");

        nn::settings::fwdbg::GetSettingsItemValue(
            &config.isVersionListEnabled, sizeof(config.isVersionListEnabled), Name, "enable_version_list");

        nn::settings::fwdbg::GetSettingsItemValue(
            &config.isETicketAvailableEnabled, sizeof(config.isETicketAvailableEnabled), Name, "enable_download_ticket");

        nn::settings::fwdbg::GetSettingsItemValue(
            &config.isSendRightsUsageStatusRequestEnabled, sizeof(config.isSendRightsUsageStatusRequestEnabled), Name, "enable_send_rights_usage_status_request");

        nn::settings::fwdbg::GetSettingsItemValue(
            &config.isSyncELicenseRequestEnabled, sizeof(config.isSyncELicenseRequestEnabled), Name, "enable_sync_elicense_request");

        nn::settings::fwdbg::GetSettingsItemValue(
            &config.isRandomWaitEnabled, sizeof(config.isRandomWaitEnabled), Name, "enable_random_wait");

        nn::settings::fwdbg::GetSettingsItemValue(
            &config.debugWaitingLimit, sizeof(config.debugWaitingLimit), Name, "debug_waiting_limit");

        nn::settings::fwdbg::GetSettingsItemValue(
            &config.versionListWaitingLimit, sizeof(config.versionListWaitingLimit), Name, "version_list_waiting_limit");

        config.useVersionListWaitPolicyForTest = false;

        return config;
    }

    Result AsyncTaskManager::Initialize(ApplicationInstallRequestList* requestList, RequestServer* requestServer, ApplicationVersionManager* applicationVersionManager) NN_NOEXCEPT
    {
        auto config = MakeDefaultConfig();

        const int MaxHandlerCount = static_cast<int>(AsyncTask::AsyncTaskType::NumberOfAsyncTaskType);

#if defined(NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2013)
        static NotificationHandlerHolder enableHandlers[MaxHandlerCount];
#else
        NN_FUNCTION_LOCAL_STATIC(NotificationHandlerHolder, enableHandlers[MaxHandlerCount]);
#endif

        int enableHandlerCount = 0;
        if (config.isNetworkUpdateEnabled)
        {
            enableHandlers[enableHandlerCount] =
            {
                [](const NotificationInfo& info) NN_NOEXCEPT -> Result
                {
                    NN_NM_TRACE("Process nup notification.\n");
                    NN_RESULT_DO(ProcessNetworkUpdateNotification(info));
                    NN_RESULT_SUCCESS;
                }, AsyncTask::AsyncTaskType::SystemUpdate
            };
            enableHandlerCount++;
        }

        if (config.isDownloadTaskListEnabled)
        {
            enableHandlers[enableHandlerCount] =
            {
                [this](const NotificationInfo& info) NN_NOEXCEPT -> Result
                {
                    NN_NM_TRACE("Process down load task notification.\n");
                    NN_RESULT_DO(this->RequestDownloadTaskList(info.eTag, true));
                    NN_RESULT_SUCCESS;
                }, AsyncTask::AsyncTaskType::DownloadTaskList
            };
            enableHandlerCount++;
        }

        if (config.isVersionListEnabled)
        {
            enableHandlers[enableHandlerCount] =
            {
                [this](const NotificationInfo& info) NN_NOEXCEPT -> Result
                {
                    NN_NM_TRACE("Process version list notification.\n");
                    this->UpdateAutoUpdateTask(info);
                    NN_RESULT_DO(this->RequestVersionList(info.eTag));
                    NN_RESULT_SUCCESS;
                }, AsyncTask::AsyncTaskType::VersionList
            };
            enableHandlerCount++;
        }

        {
            enableHandlers[enableHandlerCount] =
            {
                [this](const NotificationInfo& info) NN_NOEXCEPT -> Result
                {
                    NN_UNUSED(info);
                    NN_NM_TRACE("Process auto update.\n");
                    auto stop = m_RequestServer->Stop();
                    NN_RESULT_DO(m_ApplicationVersionManager->PerformAutoUpdate());
                    NN_RESULT_SUCCESS;
                }, AsyncTask::AsyncTaskType::AutoUpdate
            };
            enableHandlerCount++;
        }

        if (config.isETicketAvailableEnabled)
        {
            enableHandlers[enableHandlerCount] =
            {
                [this](const NotificationInfo& info)->Result
                {
                    NN_NM_TRACE("Process eticket available notification.\n");
                    NN_RESULT_DO(this->RequestDownloadTicket(info.eTicketAvailableInfo.titleId));
                    NN_RESULT_SUCCESS;
                }, AsyncTask::AsyncTaskType::ETicketAvailable
            };
            enableHandlerCount++;
        }

        if (config.isSendRightsUsageStatusRequestEnabled)
        {
            enableHandlers[enableHandlerCount] =
            {
                [](const NotificationInfo& info) NN_NOEXCEPT -> Result
                {
                    NN_NM_TRACE("Send rights usage status request notification.\n");

                    // TODO: 処理を実装する
                    NN_UNUSED(info);

                    NN_RESULT_SUCCESS;
                }, AsyncTask::AsyncTaskType::SendRightsUsageStatusRequest
            };
            enableHandlerCount++;
        }

        if (config.isSyncELicenseRequestEnabled)
        {
            enableHandlers[enableHandlerCount] =
            {
                [](const NotificationInfo& info) NN_NOEXCEPT -> Result
                {
                    NN_NM_TRACE("Send sync elicense request notification.\n");

                    // TODO: 処理を実装する
                    NN_UNUSED(info);

                    NN_RESULT_SUCCESS;
                }, AsyncTask::AsyncTaskType::SyncELicenseRequest
            };
            enableHandlerCount++;
        }

        NN_SDK_ASSERT(enableHandlerCount <= MaxHandlerCount);

        return Initialize(enableHandlers, enableHandlerCount, config, requestList, requestServer, applicationVersionManager);
    } // NOLINT(impl/function_size)

    Result AsyncTaskManager::Initialize(
        NotificationHandlerHolder* holders,
        int holderCount,
        const Config& config,
        ApplicationInstallRequestList* requestList,
        RequestServer* requestServer,
        ApplicationVersionManager* applicationVersionManager) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(holders);

        m_RequestList = requestList;
        m_RequestServer = requestServer;
        m_ApplicationVersionManager = applicationVersionManager;

        {
            uint32_t mtSeed;
            os::GenerateRandomBytes(&mtSeed, sizeof(mtSeed));
            NN_NM_TRACE("TinyMT seed: %d\n", mtSeed);
            m_Mt.Initialize(mtSeed);
        }

        for (int i = 0; i < holderCount; ++i)
        {
            AddHandler(&holders[i]);
        }

        m_Config = config;
        m_BgTask.Initialize();

        Start();

        NN_RESULT_SUCCESS;
    }

    void AsyncTaskManager::TaskThreadFunc() NN_NOEXCEPT
    {
        NN_NM_TRACE("Task thread start\n");

        while (NN_STATIC_CONDITION(true))
        {
            MultiWaitSystemEvent multiWait;

            util::optional<int> scheduleEventIndex = util::nullopt;
            // 直近で一番先に起動すべきタスクを取得
            auto foremost = GetForemostTask(m_AsyncTaskList, os::GetSystemTick());
            if (*foremost)
            {
                NN_NM_TRACE("FormostTask: %s\n", GetNotificationTypeName((*foremost)->taskType));

                // 待ち時間が 0 なら即座に実行する。
                auto& task = *foremost;
                auto scheduleTime = CalcScheduleTime(task->waitingLimit, task->registeredTime, os::GetSystemTick());
                if (scheduleTime <= 0)
                {
                    NN_NM_TRACE("Perform task immediately\n");
                    PerformTask(m_HandlerHolderRoot, task->info, task->taskType);
                    task = util::nullopt;
                    continue;
                }

                // 待ち時間が 1 以上なら bgtc をスケジューリングしつつ待ち。
                auto& scheduleEvent = m_BgTask.GetScheduleEvent();
                m_BgTask.ScheduleUnsafe(static_cast<bgtc::Interval>(scheduleTime));
                scheduleEventIndex.emplace(multiWait.Link(&scheduleEvent));
            }

            // キャンセルイベント
            auto stopEventIndex = multiWait.Link(&m_StopEvent);

            // バージョンリストインポート完了イベント
            auto versionListImportedEvent = m_RequestList->GetVersionListImportedEvent();
            auto versionListImportedEventIndex = multiWait.Link(versionListImportedEvent);

            auto signaled = multiWait.WaitAny();

            if (signaled == stopEventIndex)
            {
                NN_NM_TRACE("Task thread stopped\n");
                m_BgTask.Unschedule();
                return;
            }
            else if (signaled == versionListImportedEventIndex)
            {
                versionListImportedEvent->Clear();
                NN_NM_TRACE("Version list imported event signaled.\n");

                RegisterTaskImpl(m_AutoUpdateTask);
                continue;
            }
            else if (scheduleEventIndex && signaled == (*scheduleEventIndex))
            {
                NN_NM_TRACE("Task signaled\n");
                PerformTask(m_HandlerHolderRoot, (*foremost)->info, (*foremost)->taskType);
                *foremost = util::nullopt;
                continue;
            }
            else
            {
                NN_ABORT("Not come here. index = %d\n", signaled);
            }
        }
    }

    void AsyncTaskManager::RegisterTaskImpl(const AsyncTask& task) NN_NOEXCEPT
    {
        NN_NM_TRACE("--- Registered notification task ---\n");
        NN_NM_TRACE("Type         : %s\n", GetNotificationTypeName(task.taskType));
        NN_NM_TRACE("WaitingPolicy: %d\n", task.info.waitingPolicy.type);
        NN_NM_TRACE("WaitingLimit : %d\n", task.info.waitingPolicy.waitingLimit);
        NN_NM_TRACE("ActualLimit  : %lld\n", task.waitingLimit);
        NN_NM_TRACE("------------------------------------\n");

        // これ以降タスクリストへの書き込みが発生する。
        // デバッグ用 API からタスクリストがアクセスされる可能性があるのでロックする。
        std::lock_guard<os::Mutex> lock(m_TaskListLock);

        // 同じタイプのタスクを上書き
        auto sameTypeTask = FindTaskByType(m_AsyncTaskList, task.taskType);
        if (sameTypeTask)
        {
            NN_NM_TRACE("--- Notification task has overwritten ---\n");
            NN_NM_TRACE("Type         : %s\n", GetNotificationTypeName(task.taskType));
            NN_NM_TRACE("------------------------------------\n");

            // WaitPolicy が一致している場合は、待ち時間をリセットしないために
            // registeredTime と limit を更新せず lastUpdated と info を更新する。
            if ((*sameTypeTask)->info.waitingPolicy == task.info.waitingPolicy)
            {
                NN_NM_TRACE("Inherited waiting time.\n");
                (*sameTypeTask)->info = task.info;
                (*sameTypeTask)->lastUpdatedTime = task.registeredTime;
            }
            else
            {
                *sameTypeTask = task;
            }
        }
        else
        {
            // 同じタイプのタスクがなければ空いている箇所に追加
            auto emptyTask = FindEmptyTask(m_AsyncTaskList);
            if (emptyTask)
            {
                emptyTask->emplace(task);
            }
            else
            {
                NN_ABORT("Not come here. Notification task list has broken.\n");
            }
        }
    }

    void AsyncTaskManager::Start() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_TaskThreadLock);

        if (!m_TaskThreadRunning)
        {
            m_StopEvent.Clear();
            os::CreateThread(&m_TaskThread, [](void* arg) NN_NOEXCEPT
            {
                reinterpret_cast<AsyncTaskManager*>(arg)->TaskThreadFunc();
            }, this, m_TaskThreadStack, m_TaskThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(nssrv, AsyncTaskManager));
            os::SetThreadNamePointer(&m_TaskThread, NN_SYSTEM_THREAD_NAME(nssrv, AsyncTaskManager));
            os::StartThread(&m_TaskThread);
            m_TaskThreadRunning = true;
        }
    }

    void AsyncTaskManager::Stop() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_TaskThreadLock);

        if (m_TaskThreadRunning)
        {
            m_StopEvent.Signal();
            os::WaitThread(&m_TaskThread);
            os::DestroyThread(&m_TaskThread);
            m_TaskThreadRunning = false;
        }
    }

    int64_t AsyncTaskManager::CalcVersionListWaitingLimit() NN_NOEXCEPT
    {
        if (m_Config.versionListWaitingLimit > 0)
        {
            return m_Mt.GenerateRandomU32() % m_Config.versionListWaitingLimit;
        }
        return 0;
    }

    int64_t AsyncTaskManager::CalcWaitingLimitFromWaitingPolicy(const NotificationInfo::WaitingPolicy& policy) NN_NOEXCEPT
    {
        if (m_Config.debugWaitingLimit > 0 && policy.waitingLimit != 0)
        {
            NN_NM_TRACE("Ingore waiting policy: %d\n", policy.type);
            NN_NM_TRACE("Set waitingLimit to %d\n", m_Config.debugWaitingLimit);
            return m_Config.debugWaitingLimit;
        }
        else if (policy.type == NotificationInfo::WaitingPolicy::WaitingPolicyType::RandomWaiting)
        {
            if (m_Config.isRandomWaitEnabled && policy.waitingLimit != 0)
            {
                return m_Mt.GenerateRandomU32() % policy.waitingLimit;
            }
            else
            {
                return policy.waitingLimit;
            }
        }
        else
        {
            NN_NM_TRACE("Unexpected waiting policy: %d\n", policy.type);
            NN_NM_TRACE("Set waitingLimit to 0\n");
            return 0;
        }
    }

    void AsyncTaskManager::MakeAsyncTask(AsyncTask* out, const NotificationInfo& info) NN_NOEXCEPT
    {
        out->info = info;
        out->taskType = GetAsyncTaskFromNotificationInfo(info);
        out->registeredTime = os::GetSystemTick().GetInt64Value();
        out->lastUpdatedTime = out->registeredTime;

        // バージョンリスト通知の場合は、通常は true 側を通る。
        //
        // テスト時のみ、useVersionListWaitPolicyForTest が true になることがあるので、
        // バージョンリスト通知の場合でも false 側を通る可能性がある。
        // その場合は fwdbg ではなくて、通知に書かれた値を基にランダムウェイト値が決定される。
        if (info.type == NotificationInfo::NotificationType::VersionList && !m_Config.useVersionListWaitPolicyForTest)
        {
            out->waitingLimit = CalcVersionListWaitingLimit();
        }
        else
        {
            out->waitingLimit = CalcWaitingLimitFromWaitingPolicy(info.waitingPolicy);
        }
    }

    Result AsyncTaskManager::Register(const NotificationInfo& info) NN_NOEXCEPT
    {
        // タスクスレッドを止める
        // 登録処理後にタスクスレッドを再起動する
        Stop();
        NN_UTIL_SCOPE_EXIT{ Start(); };

        // TODO: サーバ側の対応が入ったら外側の if は削除すること。(4 NUP 時でよい)
        //       サーバが対応するまでは e_tag が空の通知が来る場合がある。その場合に必ずタスク登録するための処置として外側の if を暫定で入れている。
        if (!info.eTag.IsEmpty())
        {
            // 重複する通知の場合はタスク登録せずに終了
            if (HasDuplicatedInfo(m_AsyncTaskList, info))
            {
                NN_NM_TRACE("--- Skipped duplicated task ---\n");
                NN_NM_TRACE("Type         : %d\n", info.type);
                NN_NM_TRACE("eTag         : %s\n", info.eTag);
                NN_NM_TRACE("--------------------------------\n");

                NN_RESULT_SUCCESS;
            }
        }

        // 新しいタスクを作成
        AsyncTask newTask = {};
        MakeAsyncTask(&newTask, info);

        RegisterTaskImpl(newTask);

        NN_RESULT_SUCCESS;
    }

    Result AsyncTaskManager::RequestDownloadTaskList(const nim::ETag& eTag, bool notifiesRequiredSystemUpdate) NN_NOEXCEPT
    {
        auto stop = m_RequestServer->Stop();
        m_RequestList->RequestDownloadTaskList(eTag, notifiesRequiredSystemUpdate);
        NN_RESULT_SUCCESS;
    }

    Result AsyncTaskManager::RequestVersionList(const nim::ETag& eTag) NN_NOEXCEPT
    {
        auto stop = m_RequestServer->Stop();
        m_RequestList->RequestVersionList(eTag);
        NN_RESULT_SUCCESS;
    }

    Result AsyncTaskManager::RequestDownloadTicket(uint64_t titleId) NN_NOEXCEPT
    {
        auto stop = m_RequestServer->Stop();
        // 4NUP 時点では ETicketAvailable の title_ids に書かれる値は ApplicationId のみなので、ApplicationId として扱う。
        // 追加コンテンツのあらかじめダウンロードがサポートされた場合、ApplicationId 以外の値が書かれることがありえるので、実装の変更が必要となる。
        m_RequestList->RequestDownloadTicket({ titleId });
        NN_RESULT_SUCCESS;
    }

    void AsyncTaskManager::AddHandler(NotificationHandlerHolder* holder) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(holder);
        NN_SDK_ASSERT_NOT_NULL(holder->handler);
        NN_SDK_ASSERT_EQUAL(holder->next, nullptr);

        auto parent = &m_HandlerHolderRoot;
        for (; parent->next != nullptr; parent = parent->next)
        {
        }
        parent->next = holder;
    }

    Result AsyncTaskManager::ListNotificationTask(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::ns::AsyncTask>& outList) NN_NOEXCEPT
    {
        *outCount = 0;

        // タスクリストへの書き込みが発生する可能性があるのでロックする
        std::lock_guard<os::Mutex> lock(m_TaskListLock);

        for (int i = 0; i < AsyncTaskListSize && static_cast<size_t>(*outCount) < outList.GetLength(); ++i)
        {
            AsyncTaskOpt& task = m_AsyncTaskList[i];
            if (task)
            {
                outList[*outCount] = *task;
                (*outCount)++;
            }
        }
        NN_RESULT_SUCCESS;
    }

    void AsyncTaskManager::UpdateAutoUpdateTask(const NotificationInfo& info) NN_NOEXCEPT
    {
        // VersionList の待ちタスク登録時の通知ではなくて、最後に受信した VersionList 通知の info が使用されることに注意
        //
        // 例
        // 1. VersionList 通知A 受信
        // 2. VersionList 待ち開始
        // 3. VersionList 通知B 受信
        // 4. 待ち継続
        // 5. VersionList 待ち完了
        // 6. 通知B を使って自動更新待ちを開始

        m_AutoUpdateTask.taskType = AsyncTask::AsyncTaskType::AutoUpdate;
        m_AutoUpdateTask.waitingLimit = CalcWaitingLimitFromWaitingPolicy(info.versionListInfo.autoUpdateWaitingPolicy);
        m_AutoUpdateTask.registeredTime = os::GetSystemTick().GetInt64Value();
        m_AutoUpdateTask.lastUpdatedTime = m_AutoUpdateTask.registeredTime;
        m_AutoUpdateTask.info = info;

        // ここに入ってくる info は、VersionList 通知のものである。
        // VersionList 通知の info を使って、AutoUpdate 用の AsyncTask を作るために下記の代入が必要になる。
        m_AutoUpdateTask.info.waitingPolicy = m_AutoUpdateTask.info.versionListInfo.autoUpdateWaitingPolicy;
    }
}}}
