﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ec/system/ec_Async.h>
#include <nn/ec/system/ec_DeviceAccountApi.h>
#include <nn/ec/system/ec_DeviceLinkApi.h>
#include <nn/ec/system/ec_TicketSystemApi.h>
#include <nn/err/err_ErrorContext.h>
#include <nn/util/util_Optional.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_Types.h>
#include <nn/ncm/ncm_InstallTaskBase.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_RetailInteractiveDisplayApi.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_GameCardRegistrationApi.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/ns/srv/ns_ApplicationRecordDatabase.h>
#include <nn/ns/srv/ns_GameCardInfo.h>
#include <nn/ns/srv/ns_IntegratedContentManager.h>
#include <nn/ns/srv/ns_OsUtil.h>
#include <nn/ns/srv/ns_RequestServer.h>
#include <nn/ns/srv/ns_TicketManager.h>
#include "ns_Config.h"
#include "ns_PrepurchasedContent.h"
#include "ns_ThreadAllocator.h"

namespace nn { namespace ns { namespace srv {

    class ErrorContextHolder
    {
    public:
        virtual ~ErrorContextHolder() NN_NOEXCEPT {}
        const err::ErrorContext& GetErrorContextImpl() NN_NOEXCEPT
        {
            return m_ErrorContext;
        }

        template<typename T>
        Result GetAndSaveErrorContext(T& async) NN_NOEXCEPT
        {
            NN_RESULT_DO(SaveErrorContextIfFailed(async, async.Get()));
            NN_RESULT_SUCCESS;
        }

        template<typename T>
        Result SaveErrorContextIfFailed(T& async, Result result) NN_NOEXCEPT
        {
            NN_RESULT_TRY(result)
                NN_RESULT_CATCH_ALL
                {
                    async.GetErrorContext(&m_ErrorContext);
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }

        template<typename T>
        Result SaveNuiShellTaskErrorContextIfFailed(T& async, Result result) NN_NOEXCEPT
        {
            NN_RESULT_TRY(result)
                NN_RESULT_CATCH_ALL
                {
                    async.CreateErrorContext(&m_ErrorContext, result);
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }

    private:
        err::ErrorContext m_ErrorContext {};
    };

    class AsyncBase
    {
    public:
        virtual ~AsyncBase() NN_NOEXCEPT {}
        Result Cancel() NN_NOEXCEPT
        {
            CancelImpl();
            NN_RESULT_SUCCESS;
        }

        static Result ToAsyncResult(Result result) NN_NOEXCEPT;

        virtual Result GetErrorContext(sf::Out<err::ErrorContext> outValue) NN_NOEXCEPT
        {
            *outValue = {};
            NN_RESULT_SUCCESS;
        }

    private:
        virtual void CancelImpl() NN_NOEXCEPT = 0;
    };

    class AsyncResultBase : public AsyncBase
    {
    public:
        virtual ~AsyncResultBase() {}
        Result Get() NN_NOEXCEPT
        {
            return ToAsyncResult(GetImpl());
        }

    private:
        virtual Result GetImpl() NN_NOEXCEPT = 0;

    };

    class AsyncValueBase : public AsyncBase
    {
    public:
        virtual ~AsyncValueBase() {}
        Result GetSize(nn::sf::Out<std::uint64_t> outValue) NN_NOEXCEPT
        {
            *outValue = GetSizeImpl();
            NN_RESULT_SUCCESS;
        }

        Result Get(const nn::sf::OutBuffer& buffer) NN_NOEXCEPT
        {
            return ToAsyncResult(GetImpl(buffer));
        }

    private:
        virtual size_t GetSizeImpl() NN_NOEXCEPT = 0;
        virtual Result GetImpl(const nn::sf::OutBuffer& buffer) NN_NOEXCEPT = 0;
    };

    class ProgressAsyncResultBase : public AsyncBase
    {
    public:
        virtual ~ProgressAsyncResultBase() {}
        Result Get() NN_NOEXCEPT
        {
            return ToAsyncResult(GetImpl());
        }

        Result GetProgress(const nn::sf::OutBuffer& buffer) NN_NOEXCEPT
        {
            return ToAsyncResult(GetProgressImpl(buffer));
        }

        Result GetDetailResult() NN_NOEXCEPT
        {
            return GetDetailResultImpl();
        }

    private:
        virtual Result GetImpl() NN_NOEXCEPT = 0;
        virtual Result GetProgressImpl(const nn::sf::OutBuffer& buffer) NN_NOEXCEPT = 0;
        virtual Result GetDetailResultImpl() NN_NOEXCEPT = 0;
    };

    class NullAsyncResult : public AsyncResultBase
    {
    public:
        NullAsyncResult() NN_NOEXCEPT
        {
            m_Event.Signal();
        }
        os::SystemEvent& GetEvent() NN_NOEXCEPT
        {
            return m_Event;
        }

    private:
        virtual Result GetImpl() NN_NOEXCEPT NN_OVERRIDE
        {
            return ResultSuccess();
        }
        virtual void CancelImpl() NN_NOEXCEPT NN_OVERRIDE{}
        ManualClearSystemEvent m_Event;
    };

    class AsyncLatestSystemUpdateImpl : public AsyncValueBase, private ErrorContextHolder
    {
    public:
        AsyncLatestSystemUpdateImpl(RequestServer::ManagedStop&& stop, bool requiresExFatDriver) NN_NOEXCEPT : m_Stop(std::move(stop)), m_IsCanceled(false), m_RequiresExFatDriver(requiresExFatDriver) {}
        virtual ~AsyncLatestSystemUpdateImpl() NN_NOEXCEPT;
        Result Run() NN_NOEXCEPT;

        os::SystemEvent& GetEvent() NN_NOEXCEPT
        {
            return m_Event;
        }

        virtual Result GetErrorContext(sf::Out<err::ErrorContext> outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = ErrorContextHolder::GetErrorContextImpl();
            NN_RESULT_SUCCESS;
        }

    private:
        virtual void CancelImpl() NN_NOEXCEPT NN_OVERRIDE;
        virtual size_t GetSizeImpl() NN_NOEXCEPT NN_OVERRIDE
        {
            return sizeof(LatestSystemUpdate);
        }
        Result Execute() NN_NOEXCEPT;
        virtual Result GetImpl(const nn::sf::OutBuffer& buffer) NN_NOEXCEPT NN_OVERRIDE;
        Result GetImpl(LatestSystemUpdate* outValue) NN_NOEXCEPT;

        Result GetLatestSystemUpdateState(LatestSystemUpdate* outValue, nim::SystemUpdateTaskId id) NN_NOEXCEPT;

        Result m_Result;
        LatestSystemUpdate m_LatestSystemUpdate;
        ManualClearSystemEvent m_Event;
        util::optional<nim::AsyncContentMetaKey> m_AsyncKey;
        util::optional<nim::AsyncResult> m_AsyncResult;
        NonRecursiveMutex m_CancelMutex;
        RequestServer::ManagedStop m_Stop;
        bool m_IsCanceled;
        bool m_RequiresExFatDriver;
        util::optional<ThreadInfo> m_ThreadInfo;
    };

    class AsyncDownloadLatestUpdateImpl : public AsyncResultBase, private ErrorContextHolder
    {
    public:
        AsyncDownloadLatestUpdateImpl(RequestServer::ManagedStop&& stop, bool requiresExFatDriver) NN_NOEXCEPT : m_Stop(std::move(stop)), m_IsCanceled(false), m_RequiresExFatDriver(requiresExFatDriver) {}
        virtual ~AsyncDownloadLatestUpdateImpl() NN_NOEXCEPT;

        Result Run() NN_NOEXCEPT;

        os::SystemEvent& GetEvent() NN_NOEXCEPT
        {
            return m_Event;
        }

        virtual Result GetErrorContext(sf::Out<err::ErrorContext> outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = ErrorContextHolder::GetErrorContextImpl();
            NN_RESULT_SUCCESS;
        }

    private:
        Result Execute() NN_NOEXCEPT;
        virtual void CancelImpl() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result GetImpl() NN_NOEXCEPT NN_OVERRIDE;

        Result m_Result;
        ManualClearSystemEvent m_Event;
        util::optional<nim::AsyncContentMetaKey> m_AsyncKey;
        util::optional<nim::AsyncResult> m_AsyncResult;
        NonRecursiveMutex m_CancelMutex;
        RequestServer::ManagedStop m_Stop;
        bool m_IsCanceled;
        bool m_RequiresExFatDriver;
        util::optional<ThreadInfo> m_ThreadInfo;
    };

    class AsyncPrepareCardUpdateImpl : public AsyncResultBase, private ErrorContextHolder
    {
    public:
        AsyncPrepareCardUpdateImpl(ncm::InstallTaskBase* task, RequestServer::ManagedStop&& stop) NN_NOEXCEPT : m_Task(task), m_Stop(std::move(stop)){}
        virtual ~AsyncPrepareCardUpdateImpl() NN_NOEXCEPT;

        Result Run() NN_NOEXCEPT;

        os::SystemEvent& GetEvent() NN_NOEXCEPT
        {
            return m_Event;
        }

        virtual Result GetErrorContext(sf::Out<err::ErrorContext> outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = ErrorContextHolder::GetErrorContextImpl();
            NN_RESULT_SUCCESS;
        }

    private:
        virtual void CancelImpl() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result GetImpl() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_Result;
        }

        Result Execute() NN_NOEXCEPT;
        Result m_Result;
        ManualClearSystemEvent m_Event;
        util::optional<ThreadInfo> m_ThreadInfo;
        ncm::InstallTaskBase* m_Task;
        RequestServer::ManagedStop m_Stop;
    };

    class AsyncApplicationUpdateInfoImpl : public AsyncValueBase, private ErrorContextHolder
    {
    public:
        Result Initialize(const ncm::ApplicationId applicationId, ncm::ContentMetaKey keyList[], int count) NN_NOEXCEPT;

        Result Immediate(ApplicationUpdateInfo immediate) NN_NOEXCEPT;

        os::SystemEvent& GetEvent() NN_NOEXCEPT
        {
            return m_Immediate ? m_ImmediateEvent : m_AsyncLatestKeyList.GetEvent();
        }

        virtual Result GetErrorContext(sf::Out<err::ErrorContext> outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = ErrorContextHolder::GetErrorContextImpl();
            NN_RESULT_SUCCESS;
        }

    private:
        virtual void CancelImpl() NN_NOEXCEPT NN_OVERRIDE
        {
            if (!m_Immediate)
            {
                m_AsyncLatestKeyList.Cancel();
            }
        }
        virtual size_t GetSizeImpl() NN_NOEXCEPT NN_OVERRIDE
        {
            return sizeof(ApplicationUpdateInfo);
        }
        virtual Result GetImpl(const nn::sf::OutBuffer& buffer) NN_NOEXCEPT NN_OVERRIDE;

        util::optional<ApplicationUpdateInfo> m_Immediate;
        ManualClearSystemEvent m_ImmediateEvent;

        ncm::ContentMetaKey m_InstalledContentMetaKeyList[MaxContentMetaCountPerApplication];
        int m_InstalledContentMetaKeyCount;
        ncm::ContentMetaKey m_LatestKeyList[MaxContentMetaCountPerApplication];
        nim::AsyncLatestKeyList m_AsyncLatestKeyList;
    };

    enum class DownloadApplicationMode
    {
        Update,
        DownloadApplication,
        DownloadAddOnContent,
    };

    class AsyncDownloadApplicationImpl : public AsyncResultBase, private ErrorContextHolder
    {
    public:
        explicit AsyncDownloadApplicationImpl(ncm::ApplicationId appId, DownloadApplicationMode mode, RequestServer::ManagedStop&& stop, ncm::StorageId storageId = ncm::StorageId::Any, bool forceDirectUpdate = false) NN_NOEXCEPT
            : m_AppId(appId), m_Mode(mode), m_Stop(std::move(stop)), m_StorageId(storageId), m_ForceDirectUpdate(forceDirectUpdate), m_InstalledContentMetaKeyCount(), m_ForceUpdateList(&m_ForceUpdateBitmap, sizeof(m_ForceUpdateBitmap), MaxContentMetaCountPerApplication) {}

        Result Add(const ncm::ContentMetaKey keyList[], int count) NN_NOEXCEPT;
        Result Add(const ncm::ContentMetaKey keyList[], int count, bool forceUpdate) NN_NOEXCEPT;

        Result Run() NN_NOEXCEPT;

        os::SystemEvent& GetEvent() NN_NOEXCEPT
        {
            return m_AsyncLatestKeyList.GetEvent();
        }

        virtual Result GetErrorContext(sf::Out<err::ErrorContext> outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = ErrorContextHolder::GetErrorContextImpl();
            NN_RESULT_SUCCESS;
        }

    private:
        virtual void CancelImpl() NN_NOEXCEPT NN_OVERRIDE
        {
            m_AsyncLatestKeyList.Cancel();
        }
        virtual Result GetImpl() NN_NOEXCEPT NN_OVERRIDE;

        ncm::ApplicationId m_AppId;
        DownloadApplicationMode m_Mode;
        RequestServer::ManagedStop m_Stop;
        ncm::StorageId m_StorageId;
        bool m_ForceDirectUpdate;
        int m_InstalledContentMetaKeyCount;

        ncm::ContentMetaKey m_RequestedContentMetaKeyList[MaxContentMetaCountPerApplication];
        Bit64 m_ForceUpdateBitmap[MaxContentMetaCountPerApplication / NN_BITSIZEOF(Bit64)];
        util::BitArray m_ForceUpdateList;
        ncm::ContentMetaKey m_LatestKeyList[MaxContentMetaCountPerApplication];
        nim::AsyncLatestKeyList m_AsyncLatestKeyList;
    };

    class GameCardManager;

    template<class AsyncT>
    class AsyncGameCardBase : public ErrorContextHolder
    {
    public:
        ~AsyncGameCardBase() NN_NOEXCEPT;
        Result Initialize(const account::Uid& uid, ncm::ApplicationId appId, GameCardManager* gameCardManager) NN_NOEXCEPT;

        Result Run() NN_NOEXCEPT;

        Result GetResult() NN_NOEXCEPT
        {
            return m_Result;
        }

        util::optional<AsyncT>& GetAsync() NN_NOEXCEPT
        {
            return m_Async;
        }

        os::SystemEvent& GetEvent() NN_NOEXCEPT
        {
            return m_Event;
        }

        void CancelBase() NN_NOEXCEPT;

    protected:
        int GetApplicationIdListCount() const NN_NOEXCEPT;
        const ncm::ApplicationId* GetApplicationIdList() const NN_NOEXCEPT;

    private:
        Result Execute() NN_NOEXCEPT;
        Result GetToken() NN_NOEXCEPT;

        account::Uid m_Uid;
        ncm::ApplicationId m_AppId;
        GameCardInfo m_GameCardInfo;
        char m_IdToken[account::NintendoAccountIdTokenLengthMax + 1];

        virtual Result RequestAsync(AsyncT* outValue, const char* token, const void* cert, size_t certSize) NN_NOEXCEPT = 0;

        util::optional<account::NintendoAccountAuthorizationRequestContext> m_AuthRequest;
        util::optional<AsyncT> m_Async;

        ManualClearSystemEvent m_Event;
        NonRecursiveMutex m_CancelMutex;

        Result m_Result;
        util::optional<ThreadInfo> m_ThreadInfo;
    };

    class AsyncCheckGameCardRegistrationImpl : public AsyncResultBase, private AsyncGameCardBase<nim::AsyncGameCardRegistrationStatus>
    {
    public:
        ~AsyncCheckGameCardRegistrationImpl() NN_NOEXCEPT;
        Result Initialize(ncm::ApplicationId appId, GameCardManager* gameCardManager) NN_NOEXCEPT;
        Result Run() NN_NOEXCEPT;
        using AsyncGameCardBase<nim::AsyncGameCardRegistrationStatus>::GetEvent;

        virtual Result GetErrorContext(sf::Out<err::ErrorContext> outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = ErrorContextHolder::GetErrorContextImpl();
            NN_RESULT_SUCCESS;
        }

    private:
        virtual Result GetImpl() NN_NOEXCEPT NN_OVERRIDE;
        virtual void CancelImpl() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result RequestAsync(nim::AsyncGameCardRegistrationStatus* outValue, const char* token, const void* cert, size_t certSize) NN_NOEXCEPT NN_OVERRIDE;
    };

    class AsyncGameCardRegistrationGoldPointImpl : public AsyncValueBase, private AsyncGameCardBase<nim::AsyncGameCardRegistrationStatus>
    {
    public:
        ~AsyncGameCardRegistrationGoldPointImpl() NN_NOEXCEPT;
        Result Initialize(const account::Uid& uid, ncm::ApplicationId appId, GameCardManager* gameCardManager) NN_NOEXCEPT;
        Result Run() NN_NOEXCEPT;
        using AsyncGameCardBase<nim::AsyncGameCardRegistrationStatus>::GetEvent;

        virtual Result GetErrorContext(sf::Out<err::ErrorContext> outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = ErrorContextHolder::GetErrorContextImpl();
            NN_RESULT_SUCCESS;
        }

    private:
        virtual size_t GetSizeImpl() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result GetImpl(const nn::sf::OutBuffer& buffer) NN_NOEXCEPT NN_OVERRIDE;
        virtual void CancelImpl() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result RequestAsync(nim::AsyncGameCardRegistrationStatus* outValue, const char* token, const void* cert, size_t certSize) NN_NOEXCEPT NN_OVERRIDE;
    };

    class AsyncRegisterGameCardImpl : public AsyncResultBase, private AsyncGameCardBase<nim::AsyncResult>
    {
    public:
        ~AsyncRegisterGameCardImpl() NN_NOEXCEPT;
        Result Initialize(const account::Uid& uid, ncm::ApplicationId appId, int goldPoint, GameCardManager* gameCardManager) NN_NOEXCEPT;
        Result Run() NN_NOEXCEPT;
        using AsyncGameCardBase<nim::AsyncResult>::GetEvent;

        virtual Result GetErrorContext(sf::Out<err::ErrorContext> outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = ErrorContextHolder::GetErrorContextImpl();
            NN_RESULT_SUCCESS;
        }

    private:
        virtual Result GetImpl() NN_NOEXCEPT NN_OVERRIDE;
        virtual void CancelImpl() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result RequestAsync(nim::AsyncResult* outValue, const char* token, const void* cert, size_t certSize) NN_NOEXCEPT NN_OVERRIDE;

        int m_GoldPoint;
    };

    class AsyncDeviceLinkBase : public ErrorContextHolder
    {
    public:
        explicit AsyncDeviceLinkBase(const account::Uid& uid) NN_NOEXCEPT : m_Uid(uid), m_IsCanceled(false) {}
        virtual ~AsyncDeviceLinkBase() NN_NOEXCEPT;

        Result Run() NN_NOEXCEPT;

        Result GetResult() NN_NOEXCEPT
        {
            return m_Result;
        }

        os::SystemEvent& GetEvent() NN_NOEXCEPT
        {
            return m_Event;
        }

        void CancelBase() NN_NOEXCEPT;

    private:
        Result Execute() NN_NOEXCEPT;
        Result ExecuteImpl() NN_NOEXCEPT;

        bool IsCanceled() NN_NOEXCEPT;

        account::Uid m_Uid;

        util::optional<ec::system::AsyncResult> m_Async;
        util::optional<ec::system::AsyncDeviceRegistrationInfo> m_AsyncInfo;
        bool m_IsCanceled;

        ManualClearSystemEvent m_Event;
        NonRecursiveMutex m_CancelMutex;

        Result m_Result;
        util::optional<ThreadInfo> m_ThreadInfo;
    };

    class AsyncDeviceLinkImpl : public AsyncResultBase, private AsyncDeviceLinkBase
    {
    public:
        explicit AsyncDeviceLinkImpl(const account::Uid& uid) NN_NOEXCEPT : AsyncDeviceLinkBase(uid) {}
        virtual ~AsyncDeviceLinkImpl() NN_NOEXCEPT;

        Result Run() NN_NOEXCEPT;

        os::SystemEvent& GetEvent() NN_NOEXCEPT
        {
            return AsyncDeviceLinkBase::GetEvent();
        }

        virtual Result GetErrorContext(sf::Out<err::ErrorContext> outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = ErrorContextHolder::GetErrorContextImpl();
            NN_RESULT_SUCCESS;
        }

    private:
        virtual void CancelImpl() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result GetImpl() NN_NOEXCEPT NN_OVERRIDE;
    };

    class AsyncDownloadApplicationPrepurchasedRightsBase : public ErrorContextHolder
    {
    public:
        AsyncDownloadApplicationPrepurchasedRightsBase(TicketManager* pTicketManager, RequestServer* requestServer) NN_NOEXCEPT
            : m_pTicketManager(pTicketManager), m_RequestServer(requestServer), m_IsCanceled(false) {}

        virtual ~AsyncDownloadApplicationPrepurchasedRightsBase() NN_NOEXCEPT;

        Result Initialize(es::RightsIdIncludingKeyId* rightsIds, int rightsIdsCount) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(rightsIdsCount <= MaxPrepurchasedContentsCount, ResultTooManyPrepurchasedContent());

            for (int i = 0; i < rightsIdsCount; i++)
            {
                m_RightsIds[i] = rightsIds[i];
            }

            m_RightsIdCount = rightsIdsCount;

            NN_RESULT_SUCCESS;
        }

        Result Run() NN_NOEXCEPT;

        Result GetResult() NN_NOEXCEPT
        {
            return m_Result;
        }

        util::optional<ec::system::AsyncTicketDownloadStatusForPrepurchasedContents>& GetAsync() NN_NOEXCEPT
        {
            return m_Async;
        }

        os::SystemEvent& GetEvent() NN_NOEXCEPT
        {
            return m_Event;
        }

        void CancelBase() NN_NOEXCEPT;

    private:
        Result Execute() NN_NOEXCEPT;
        Result ExecuteImpl() NN_NOEXCEPT;
        bool IsCanceled() NN_NOEXCEPT;

        es::RightsIdIncludingKeyId m_RightsIds[MaxPrepurchasedContentsCount];
        int m_RightsIdCount;
        TicketManager* m_pTicketManager;
        RequestServer* m_RequestServer;

        util::optional<ec::system::AsyncTicketDownloadStatusForPrepurchasedContents> m_Async;
        bool m_IsCanceled;

        ManualClearSystemEvent m_Event;
        NonRecursiveMutex m_CancelMutex;

        Result m_Result;
        util::optional<ThreadInfo> m_ThreadInfo;
    };

    class AsyncDownloadApplicationPrepurchasedRightsImpl : public AsyncResultBase, private AsyncDownloadApplicationPrepurchasedRightsBase
    {
    public:
        AsyncDownloadApplicationPrepurchasedRightsImpl(TicketManager* pTicketManager, RequestServer* requestServer) NN_NOEXCEPT
            : AsyncDownloadApplicationPrepurchasedRightsBase(pTicketManager, requestServer) {}
        ~AsyncDownloadApplicationPrepurchasedRightsImpl() NN_NOEXCEPT;

        Result Initialize(es::RightsIdIncludingKeyId* rightsIds, int rightsIdsCount) NN_NOEXCEPT
        {
            return AsyncDownloadApplicationPrepurchasedRightsBase::Initialize(rightsIds, rightsIdsCount);
        }

        Result Run() NN_NOEXCEPT;

        os::SystemEvent& GetEvent() NN_NOEXCEPT
        {
            return AsyncDownloadApplicationPrepurchasedRightsBase::GetEvent();
        }

        virtual Result GetErrorContext(sf::Out<err::ErrorContext> outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = ErrorContextHolder::GetErrorContextImpl();
            NN_RESULT_SUCCESS;
        }

    private:
        virtual void CancelImpl() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result GetImpl() NN_NOEXCEPT NN_OVERRIDE;
    };

    class AsyncSyncRightsBase : public ErrorContextHolder
    {
    public:
        explicit AsyncSyncRightsBase(RequestServer::ManagedStop&& stop) NN_NOEXCEPT
            : m_Stop(std::move(stop)), m_IsCanceled(false) {}

        virtual ~AsyncSyncRightsBase() NN_NOEXCEPT;

        Result Run() NN_NOEXCEPT;

        Result GetResult() NN_NOEXCEPT
        {
            return m_Result;
        }

        os::SystemEvent& GetEvent() NN_NOEXCEPT
        {
            return m_Event;
        }

        void CancelBase() NN_NOEXCEPT;

    private:
        Result Execute() NN_NOEXCEPT;
        Result ExecuteImpl() NN_NOEXCEPT;
        bool IsCanceled() NN_NOEXCEPT;

        RequestServer::ManagedStop m_Stop;

        util::optional<nim::AsyncResult> m_Async;
        util::optional<nim::AsyncAssignedELicenses> m_AsyncAssignedELicenses;
        util::optional<ec::system::AsyncProgressResult> m_AsyncProgress;
        bool m_IsCanceled;

        ManualClearSystemEvent m_Event;
        NonRecursiveMutex m_CancelMutex;

        Result m_Result;
        util::optional<ThreadInfo> m_ThreadInfo;
    };

    class AsyncSyncRightsImpl : public AsyncResultBase, private AsyncSyncRightsBase
    {
    public:
        explicit AsyncSyncRightsImpl(RequestServer::ManagedStop&& stop) NN_NOEXCEPT
            : AsyncSyncRightsBase(std::move(stop)) {}
        ~AsyncSyncRightsImpl() NN_NOEXCEPT;

        Result Run() NN_NOEXCEPT;

        os::SystemEvent& GetEvent() NN_NOEXCEPT
        {
            return AsyncSyncRightsBase::GetEvent();
        }

        virtual Result GetErrorContext(sf::Out<err::ErrorContext> outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = ErrorContextHolder::GetErrorContextImpl();
            NN_RESULT_SUCCESS;
        }

    private:
        virtual void CancelImpl() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result GetImpl() NN_NOEXCEPT NN_OVERRIDE;
    };
}}}

