﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ns/detail/ns_Log.h>
#include <nn/util/util_Optional.h>
#include <nn/ns/srv/ns_ApplicationShellEventObserver.h>

namespace nn { namespace ns { namespace srv {

    namespace {
        util::optional<ApplicationShellEventInfo> MakeShellEventInfo(const pm::ProcessEventInfo& info) NN_NOEXCEPT
        {
            ApplicationShellEvent appEvent;
            switch (info.event)
            {
            case pm::ProcessEvent_Exit: appEvent = ApplicationShellEvent::Exit; break;
            case pm::ProcessEvent_DebugRunning: appEvent = ApplicationShellEvent::DebugRunning; break;
            case pm::ProcessEvent_DebugBreaked: appEvent = ApplicationShellEvent::DebugBreaked; break;
            case pm::ProcessEvent_Exception: appEvent = ApplicationShellEvent::Exception; break;
            default: return util::nullopt;
            }

            ApplicationShellEventInfo appInfo = {};
            appInfo.event = appEvent;
            appInfo.processId = info.processId;

            return appInfo;
        }
    }

    ApplicationShellEventObserver::ApplicationShellEventObserver() NN_NOEXCEPT :
        m_MessageQueue(m_QueueBuffer, MaxQueueCount), m_Event(os::EventClearMode_AutoClear, true)
    {
        m_HeapHandle = lmem::CreateUnitHeap(m_EventInfoData, sizeof(m_EventInfoData), sizeof(m_EventInfoData[0]), lmem::CreationOption_ThreadSafe, 8, &m_HeapHead);
    }

    int ApplicationShellEventObserver::PopApplicationShellEventInfo(ApplicationShellEventInfo outList[], int count) NN_NOEXCEPT
    {
        int outCount = 0;
        uintptr_t message;
        while (outCount < count && m_MessageQueue.TryReceive(&message))
        {
            auto info = reinterpret_cast<ApplicationShellEventInfo*>(message);
            outList[outCount] = *info;
            lmem::FreeToUnitHeap(m_HeapHandle, info);

            outCount++;
        }

        return outCount;
    }

    void ApplicationShellEventObserver::NotifyImpl(const ApplicationShellEventInfo& appInfo) NN_NOEXCEPT
    {
        auto allocated = reinterpret_cast<ApplicationShellEventInfo*>(lmem::AllocateFromUnitHeap(m_HeapHandle));
        if(!allocated)
        {
            NN_DETAIL_NS_TRACE("[ApplicationShellEventObserver] event dropped. event %u, processId 0x%016llx\n", appInfo.event, appInfo.processId);
            return;
        }
        *allocated = appInfo;

        if(!m_MessageQueue.TrySend(reinterpret_cast<uintptr_t>(allocated)))
        {
            NN_DETAIL_NS_TRACE("[ApplicationShellEventObserver] event dropped. event %u, processId 0x%016llx\n", appInfo.event, appInfo.processId);
            lmem::FreeToUnitHeap(m_HeapHandle, allocated);
            return;
        }

        m_Event.Signal();
    }

    void ApplicationShellEventObserver::Notify(const pm::ProcessEventInfo& info) NN_NOEXCEPT
    {
        auto appInfo = MakeShellEventInfo(info);
        if (!appInfo)
        {
            return;
        }

        NotifyImpl(*appInfo);
    }

    void ApplicationShellEventObserver::RequestReboot() NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[ApplicationShellEventObserver] RequestReboot\n");
        ApplicationShellEvent appEvent = ApplicationShellEvent::RebootRequest;
        ApplicationShellEventInfo appInfo = {};
        appInfo.event = appEvent;
        appInfo.processId = os::ProcessId::GetInvalidId();

        NotifyImpl(appInfo);
    }

    void ApplicationShellEventObserver::NotifyStartCardUpdate() NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[ApplicationShellEventObserver] NotifyStartCardUpdate\n");
        ApplicationShellEvent appEvent = ApplicationShellEvent::NotifyStartCardUpdate;
        ApplicationShellEventInfo appInfo = {};
        appInfo.event = appEvent;
        appInfo.processId = os::ProcessId::GetInvalidId();

        NotifyImpl(appInfo);
    }

    void ApplicationShellEventObserver::RequestShowCardUpdateProcessing() NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[ApplicationShellEventObserver] RequestShowCardUpdateProcessing\n");
        ApplicationShellEvent appEvent = ApplicationShellEvent::ShowCardUpdateProcessingRequest;
        ApplicationShellEventInfo appInfo = {};
        appInfo.event = appEvent;
        appInfo.processId = os::ProcessId::GetInvalidId();

        NotifyImpl(appInfo);
    }

    void ApplicationShellEventObserver::NotifyEndCardUpdate() NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[ApplicationShellEventObserver] NotifyEndCardUpdate\n");
        ApplicationShellEvent appEvent = ApplicationShellEvent::NotifyEndCardUpdate;
        ApplicationShellEventInfo appInfo = {};
        appInfo.event = appEvent;
        appInfo.processId = os::ProcessId::GetInvalidId();

        NotifyImpl(appInfo);
    }
}}}  // namespace nn::ns::srv

