﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <array>
#include <nn/nn_ErrorResult.h>
#include <nn/ec/ec_Result.h>
#include <nn/erpt/erpt_Context.h>
#include <nn/err/err_SystemApi.h>
#include <nn/es.h>
#include <nn/es/es_RightsTypes.h>
#include <nn/es/es_RightsApi.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_RightsId.h>
#include <nn/lr/lr_Service.h>
#include <nn/lr/lr_Result.h>
#include <nn/lr/lr_LocationResolver.h>
#include <nn/lr/lr_RegisteredLocationResolver.h>
#include <nn/pm/pm_ShellApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include <nn/util/util_Optional.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ProgramLocation.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/ns/ns_ApplicationRightsSystemApi.h>
#include <nn/ns/srv/ns_ApplicationContentMetaStatusUtil.h>
#include <nn/ns/srv/ns_ApplicationLaunchManager.h>
#include <nn/ns/srv/ns_DevelopInterfaceServerFactory.h>
#include <nn/ns/srv/ns_Shell.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/os/os_Types.h>

#include "ns_AccountUtil.h"
#include "ns_ApplicationControlFileUtil.h"
#include "ns_DebugUtil.h"
#include "ns_ProgramIndexUtil.h"
#include "ns_RightsUtil.h"
#include "ns_ShellUtil.h"

namespace nn { namespace ns { namespace srv {
namespace {
    Result ReportError(Result result) NN_NOEXCEPT
    {
        auto errorCode = err::ConvertResultToErrorCode(result);
        char errorString[nn::err::ErrorCode::StringLengthMax];
        err::GetErrorCodeString(errorString, sizeof(errorString), errorCode);
        auto errorStringLength = static_cast<uint32_t>(strnlen(errorString, sizeof(errorString)));

        erpt::Context context(erpt::ErrorInfo);
        NN_RESULT_DO(context.Add(erpt::FieldId::ErrorCode, errorString, errorStringLength));
        NN_RESULT_DO(context.AddCurrentThreadName());
        NN_RESULT_DO(context.CreateReport(erpt::ReportType_Invisible));
        NN_RESULT_SUCCESS;
    }

    ApplicationRightsOnClientFlag GetApplicationRightsOnClientFlag(const es::RightsStatus& status) NN_NOEXCEPT
    {
        ApplicationRightsOnClientFlag outValue = {};
        if (status.status == es::RightsStatus::Status::Available)
        {
            outValue.Set<ApplicationRightsOnClientFlag_HasAvailableRights>();
        }
        else if (status.status == es::RightsStatus::Status::AvailableSinceAccountRestricted)
        {
            outValue.Set<ApplicationRightsOnClientFlag_HasAvailableRights>();
            outValue.Set<ApplicationRightsOnClientFlag_HasAccountRestrictedRights>();
            // TODO: LongRun の時は Falseになるような対応をする。
            outValue.Set<ApplicationRightsOnClientFlag_RecommendInquireServer>();
        }
        // TODO: 予約状態 の判定
        // 権利がない、もしくは、権利はあるけど有効期限切れとかの場合はサーバーに問い合わせる
        else
        {
            outValue.Set<ApplicationRightsOnClientFlag_HasUnavailableRights>();
            outValue.Set<ApplicationRightsOnClientFlag_RecommendInquireServer>();
        }

        if (status.rights.IsRecommendsServerInteraction())
        {
            outValue.Set<ApplicationRightsOnClientFlag_RecommendInquireServer>();
        }

        return outValue;
    }

    Result MakeApplicationRightsOnClientFlag(ApplicationRightsOnClientFlag* outValue, es::RightsId* pList, int listCount, const account::Uid& uid) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        constexpr int StatusListCount = 16;
        NN_ABORT_UNLESS(listCount <= StatusListCount);
        es::RightsStatus statusList[StatusListCount];
        if (uid != account::InvalidUid)
        {
            account::NintendoAccountId naId;
            NN_RESULT_DO(GetNintendoAccountId(&naId, uid));
            NN_RESULT_DO(es::CheckRightsStatus(statusList, listCount, pList, listCount, &naId, 1));
        }
        else
        {
            NN_RESULT_DO(es::CheckRightsStatus(statusList, listCount, pList, listCount));
        }

        ApplicationRightsOnClientFlag resultFlag = {};
        for (const auto& status : util::MakeSpan(statusList, statusList + listCount))
        {
            resultFlag |= GetApplicationRightsOnClientFlag(status);
        }

#else
        NN_UNUSED(pList);
        NN_UNUSED(listCount);
        NN_UNUSED(uid);
        ApplicationRightsOnClientFlag resultFlag = {};
        resultFlag.Set<ApplicationRightsOnClientFlag_HasAvailableRights>();
#endif

        *outValue = resultFlag;
        NN_RESULT_SUCCESS;
    }
}

void ApplicationLaunchManager::Initialize(ApplicationRecordDatabase* recordDb, IntegratedContentManager* integrated, ApplicationVersionManager* version, GameCardManager* gameCard, ApplicationControlDataManager* control, TicketManager* ticket, CommitManager* commit) NN_NOEXCEPT
{
    m_pRecordDb = recordDb;
    m_pIntegrated = integrated;
    m_pVersionManager = version;
    m_pGameCardManager = gameCard;
    m_pControlManager = control;
    m_pTicketManager = ticket;
    m_pCommitManager = commit;
}

Result ApplicationLaunchManager::PrepareApplicationControlProperty(const ApplicationLaunchInfo& info, uint8_t programIndex) NN_NOEXCEPT
{
    std::unique_ptr<ApplicationControlProperty> property(new ApplicationControlProperty());
    NN_RESULT_THROW_UNLESS(property, ResultBufferNotEnough());
    NN_RESULT_DO(m_pControlManager->GetApplicationControlProperty(property.get(), info, programIndex));

    if (property->addOnContentRegistrationType == AddOnContentRegistrationType::AllOnLaunch)
    {
        NN_RESULT_DO(UnregisterAllAddOnContentLocation());
        NN_RESULT_DO(RegisterAllAddOnContentLocationAndExternalKey(info.id));
    }
    m_pCommitManager->RegisterRuntimeInstallPolicy(info.id, property->runtimeAddOnContentInstall);

    // クラッシュレポートに関するアプリのポリシーを記憶しておく
    nn::ns::srv::RegisterApplicationCrashReportPolicy(property->crashReport);

    if ((property->attributeFlag & static_cast<Bit32>(AttributeFlag::RetailInteractiveDisplay)) == static_cast<Bit32>(AttributeFlag::RetailInteractiveDisplay))
    {
        NN_RESULT_THROW_UNLESS(settings::system::GetQuestFlag(), ResultRetailInteractiveDisplayApplicationNotPermitted());
    }
    if (((property->repairFlag & static_cast<Bit8>(RepairFlag::SuppressGameCardAccess)) == static_cast<Bit8>(RepairFlag::SuppressGameCardAccess))
        && (info.applicationStorageId == ncm::StorageId::Card))
    {
        m_pGameCardManager->DisableSignalEventOnRemove();
    }
    NN_RESULT_THROW_UNLESS(property->programIndex == programIndex, ResultUnexpectedProgramIndex());

    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::GetControlPath(ncm::Path* outValue, const ncm::ContentMetaKey& key, uint8_t programIndex) NN_NOEXCEPT
{
    ncm::ContentId contentId;
    NN_RESULT_TRY(m_pIntegrated->GetContentId(&contentId, key, ncm::ContentType::Control, programIndex))
        NN_RESULT_CATCH(ncm::ResultContentMetaNotFound){ NN_RESULT_THROW(ResultApplicationControlDataNotFound()); }
    NN_RESULT_END_TRY

    NN_RESULT_DO(m_pIntegrated->GetContentPath(outValue, contentId));

    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::RedirectApplicationProgram(const ncm::ContentMetaKey& key, ncm::ApplicationId id, uint8_t programIndex, ncm::StorageId storageId) NN_NOEXCEPT
{
    lr::LocationResolver locationResolver;
    NN_RESULT_DO(lr::OpenLocationResolver(&locationResolver, storageId));

    lr::Path path;
    NN_RESULT_DO(GetContentPathByStorageId(&path, key, ncm::ContentType::Program, programIndex, storageId));
    locationResolver.RedirectApplicationProgramPath(GetProgramId(id, programIndex), path);

    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::GetContentPathByStorageId(lr::Path* outValue, const ncm::ContentMetaKey& key, ncm::ContentType type, uint8_t programIndex, ncm::StorageId targetStorage) NN_NOEXCEPT
{
    ncm::ContentId contentId;
    ncm::Path path;

    NN_RESULT_TRY(m_pIntegrated->GetContentIdByStorageId(&contentId, key, type, programIndex, targetStorage))
        NN_RESULT_CATCH(ncm::ResultContentNotFound) { NN_RESULT_THROW(ResultApplicationContentNotFound()); }
    NN_RESULT_END_TRY
    NN_RESULT_DO(m_pIntegrated->GetContentPathByStorageId(&path, contentId, targetStorage));

    // ncm と lr のパスは同じなのだがコピーが要るのが残念
    std::memcpy(outValue->string, path.string, sizeof(lr::Path));

    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::RegisterProgramPatch(const ncm::ContentMetaKey& key, ncm::ApplicationId id, uint8_t programIndex, ncm::StorageId storageId) NN_NOEXCEPT
{
    // 毎回取得せずに、メンバ変数として持ってもいいが、ひとまず
    lr::RegisteredLocationResolver locationResolver;
    NN_RESULT_DO(lr::OpenRegisteredLocationResolver(&locationResolver));

    // lr へパッチ情報の登録
    if (key.type == ncm::ContentMetaType::Patch)
    {
        lr::Path lrPath;
        NN_DETAIL_NS_TRACE("Patch: %016llx, %u, %d, %d\n", key.id, key.version, key.installType, storageId);
        NN_RESULT_DO(GetContentPathByStorageId(&lrPath, key, ncm::ContentType::Program, programIndex, storageId));
        NN_RESULT_DO(locationResolver.RegisterProgramPath(GetProgramId(id, programIndex), lrPath));
        NN_DETAIL_NS_TRACE("Register Patch of %s\n", lrPath.string);
    }
    else
    {
        NN_RESULT_DO(locationResolver.UnregisterProgramPath(id));
    }
    NN_RESULT_SUCCESS;
}

// DeveloperInterface 向け
Result ApplicationLaunchManager::AfterLaunchProgram(os::ProcessId processId, const ApplicationLaunchInfo& info, uint8_t programIndex) NN_NOEXCEPT
{
    NN_RESULT_DO(PrepareApplicationControlProperty(info, programIndex));

    NN_RESULT_DO(m_LaunchPropertyManager.Update(info, programIndex, processId));

    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::LaunchApplication(os::ProcessId* outValue, const ApplicationLaunchInfo& info, uint8_t programIndex) NN_NOEXCEPT
{
    return LaunchApplicationImpl(outValue, info, programIndex);
}

Result ApplicationLaunchManager::LaunchApplicationForDevelop(os::ProcessId* outValue, const ApplicationLaunchInfo& info) NN_NOEXCEPT
{
    util::optional<ncm::ContentMetaKey> found;
    NN_RESULT_DO(m_pRecordDb->FindLaunchableMain(&found, info.id));
    NN_RESULT_THROW_UNLESS(found, ResultApplicationContentNotFound());

    uint8_t programIndex;
    NN_RESULT_DO(m_pIntegrated->GetMinimumProgramIndex(&programIndex, *found));

    NN_RESULT_DO(LaunchApplicationImpl(outValue, info, programIndex));
    auto isSuccess = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!isSuccess)
        {
            srv::TerminateProcess(*outValue);
        }
    };

    NN_RESULT_DO(m_LaunchPropertyManager.Update(info, programIndex, *outValue));

    isSuccess = true;
    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::LaunchApplicationImpl(os::ProcessId* outValue, const ApplicationLaunchInfo& info, uint8_t programIndex) NN_NOEXCEPT
{
    const auto id = info.id;
    NN_RESULT_DO(CheckLaunchVersionImpl(id, CheckLaunchVersionResultHandlingPolicy::IgnoreResultApplicationUpdateRecommended));

    util::optional<ncm::ContentMetaKey> appKey;
    util::optional<ncm::ContentMetaKey> patchKey;
    NN_RESULT_DO(
        m_pRecordDb->FindLaunchableApplicationAndPatch(
            &appKey, &patchKey, id, info.applicationStorageId, info.patchStorageId));
    NN_RESULT_THROW_UNLESS(appKey, ResultMainApplicationNotFound());
    NN_RESULT_THROW_UNLESS(
        !ncm::IsUniqueStorage(info.patchStorageId) || patchKey,
        ResultApplicationContentNotFound());

    // 権利チェック
    NN_RESULT_DO(CheckApplicationLaunchRightsImpl(appKey, patchKey));

    {
        lr::LocationResolver locationResolver;
        NN_RESULT_DO(lr::OpenLocationResolver(&locationResolver, info.applicationStorageId));
        locationResolver.ClearApplicationRedirection();
    }

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    es::UnregisterAllTitleKey();
#endif
    NN_RESULT_DO(RegisterContentExternalKeyImpl(*appKey, programIndex, ncm::ContentType::Program, info.applicationStorageId));
    NN_RESULT_DO(RegisterApplicationExternalKey(id));

    auto mainContentKey = DecideMainContentMetaKey(appKey, patchKey);
    auto mainStorageId = mainContentKey->type == ncm::ContentMetaType::Application ? info.applicationStorageId : info.patchStorageId;

    NN_RESULT_DO(RedirectApplicationProgram(*appKey, id, programIndex, info.applicationStorageId));
    NN_RESULT_DO(RegisterProgramPatch(*mainContentKey, id, programIndex, mainStorageId));
    auto version = mainContentKey->version;
    NN_RESULT_THROW_UNLESS(version == info.version, ResultUnexpectedApplicationVersion());

    // 自動更新の登録
    {
        ncm::PatchId patchId;
        NN_RESULT_DO(m_pIntegrated->GetPatchId(&patchId, *appKey));

        AutoUpdateInfo updateInfo = { patchId, version };
        NN_RESULT_DO(m_pRecordDb->EnableAutoUpdate(id, updateInfo));
    }

    // TerminateResult のクリア
    NN_RESULT_DO(m_pRecordDb->ClearApplicationTerminateResult(id));

    // pm に対しては、Application の情報で launchProgram する
    // pm に渡す情報は、ContentMetaKey と storageId
    os::ProcessId processId;
    NN_RESULT_DO(CallLaunchProgramForApplication(&processId, info, programIndex));
    auto isSuccess = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!isSuccess)
        {
            srv::TerminateProcess(processId);
        }
    };

    NN_RESULT_DO(m_pVersionManager->UpgradeLaunchRequiredVersion(id, version));

    NN_RESULT_DO(m_pRecordDb->UpdateEvent(id, ApplicationEvent::Launched));

    isSuccess = true;
    *outValue = processId;
    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::LaunchApplicationForAutoBoot(
    os::ProcessId* outValue,
    ncm::ApplicationId id) NN_NOEXCEPT
{
    const ncm::StorageId storageId = ncm::StorageId::Card;
    ncm::StorageId patchStorageId;
    ncm::ContentMetaKey mainContentKey;

    ncm::ContentMetaDatabase db;
    NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, storageId));

    // OnCardPatch 対応
    {
        // 自動起動時にアプリケーション記録に OnCardPatch の情報はないので、コンテンツメタデータベースを見に行く
        ncm::ContentMetaKey appKey;
        auto count = db.ListContentMeta(&appKey, 1, ncm::ContentMetaType::Application);
        NN_RESULT_THROW_UNLESS(count.total == 1, ResultApplicationContentNotFound());

        ncm::ContentMetaKey patchKey;
        count = db.ListContentMeta(&patchKey, 1, ncm::ContentMetaType::Patch);
        bool hasPatch = count.total == 1;

        mainContentKey = hasPatch ? patchKey : appKey;
        patchStorageId = hasPatch ? storageId : ncm::StorageId::None;

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        es::UnregisterAllTitleKey();
#endif
        // Program は内部鍵なので、鍵の登録は必要ない
        // OnCardPatch のみ 鍵の登録が必要
        if (hasPatch)
        {
            NN_RESULT_DO(RegisterContentsExternalKey(mainContentKey, storageId));
        }

        // 自動起動の場合はマルチプログラムアプリケーションをサポートしない
        NN_RESULT_DO(RegisterProgramPatch(mainContentKey, id, 0, storageId));
    }

    // 起動直後なので、host の Resolver はクリアしなくてもよい

    const int32_t LaunchFlags = LaunchProgramFlags_NotifyDebug | LaunchProgramFlags_NotifyExit;
    auto isApplication = mainContentKey.type == ncm::ContentMetaType::Application || mainContentKey.type == ncm::ContentMetaType::Patch;

    os::ProcessId processId;
    NN_RESULT_DO(srv::LaunchProgram(&processId, ncm::MakeProgramLocation(storageId, id), LaunchFlags));
    auto isSuccess = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!isSuccess)
        {
            srv::TerminateProcess(processId);
        }
    };

    if (isApplication)
    {
        const uint8_t ProgramIndex = 0; // AutoBoot のアプリはサブプログラムを入れる想定はない
        const Bit8 LaunchInfoFlags = static_cast<Bit8>(ApplicationLaunchInfoFlag::AutoBootByGameCard);
        ApplicationLaunchInfo info = { id, mainContentKey.version, LaunchFlags, storageId, patchStorageId, LaunchInfoFlags };

        NN_RESULT_DO(m_LaunchPropertyManager.Update(info, ProgramIndex, processId));

        NN_RESULT_DO(PrepareApplicationControlProperty(info, ProgramIndex));
    }

    isSuccess = true;
    *outValue = processId;
    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::CallLaunchProgramForApplication(os::ProcessId* outValue, const ApplicationLaunchInfo& info, uint8_t programIndex) NN_NOEXCEPT
{
    // 前回 Host 起動で残った Redirection を削除しておく
    lr::LocationResolver hostLocationResolver;
    NN_RESULT_DO(lr::OpenLocationResolver(&hostLocationResolver, ncm::StorageId::Host));
    NN_RESULT_DO(hostLocationResolver.ClearApplicationRedirection());

    NN_RESULT_DO(PrepareApplicationControlProperty(info, programIndex));

    os::ProcessId processId;
    NN_RESULT_DO(srv::LaunchProgram(&processId, ncm::MakeProgramLocation(info.applicationStorageId, GetProgramId(info.id, programIndex)), info.launchFlags));
    auto isSuccess = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!isSuccess)
        {
            srv::TerminateProcess(processId);
        }
    };

    isSuccess = true;
    *outValue = processId;

    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::CheckApplicationLaunchVersion(nn::ncm::ApplicationId id) NN_NOEXCEPT
{
    return CheckLaunchVersionImpl(id, CheckLaunchVersionResultHandlingPolicy::ThrowAll);
}

Result ApplicationLaunchManager::CheckLaunchVersionImpl(ncm::ApplicationId id, CheckLaunchVersionResultHandlingPolicy policy) NN_NOEXCEPT
{
    util::optional<ncm::ContentMetaKey> mainMeta;
    NN_RESULT_DO(m_pRecordDb->FindLaunchableMain(&mainMeta, id));
    NN_RESULT_THROW_UNLESS(mainMeta, ResultMainApplicationNotFound());

    ncm::PatchId patchId = {};
    if (mainMeta->type == ncm::ContentMetaType::Application)
    {
        NN_RESULT_DO(m_pIntegrated->GetPatchId(&patchId, *mainMeta));
    }
    else if (mainMeta->type == ncm::ContentMetaType::Patch)
    {
        patchId.value = mainMeta->id;
    }
    else
    {
        NN_ABORT("Must not come here\n");
    }

    // INFO: 必須バージョンを取得
    uint32_t requiredSystemVersion;
    NN_RESULT_DO(m_pIntegrated->GetRequiredSystemVersion(&requiredSystemVersion, *mainMeta));

    uint32_t requiredApplicationVersion;
    NN_RESULT_DO(GetRequiredApplicationVersion(&requiredApplicationVersion, id));

    NN_RESULT_TRY(m_pVersionManager->CheckApplicationLaunchVersion(id, patchId, mainMeta->version, requiredSystemVersion, requiredApplicationVersion))
        NN_RESULT_CATCH(ResultApplicationUpdateRecommended)
        {
            if (policy != CheckLaunchVersionResultHandlingPolicy::IgnoreResultApplicationUpdateRecommended)
            {
                NN_RESULT_RETHROW;
            }
        }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::CheckApplicationLaunchRights(nn::ncm::ApplicationId id) NN_NOEXCEPT
{
    util::optional<ncm::ContentMetaKey> appMetaKey;
    util::optional<ncm::ContentMetaKey> patchMetaKey;
    NN_RESULT_DO(m_pRecordDb->FindLaunchableApplicationAndPatch(&appMetaKey, &patchMetaKey, id));
    NN_RESULT_THROW_UNLESS(appMetaKey, ResultApplicationContentNotFound());

    auto result = CheckApplicationLaunchRightsImpl(appMetaKey, patchMetaKey);

    // 予約購入状態以外で権利チェックに失敗したときはエラーレポートを送信する
    if (result.IsFailure() && !ResultApplicationPrepurchased::Includes(result))
    {
        // 戻り値を無視する
        ReportError(result);
    }
    NN_RESULT_DO(result);

    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::CheckApplicationLaunchRightsImpl(const util::optional<ncm::ContentMetaKey>& appMetaKey, const util::optional<ncm::ContentMetaKey>& patchMetaKey) NN_NOEXCEPT
{
    // Program の権限だけチェックする
    NN_RESULT_DO(CheckProgramContentRights(*appMetaKey));
    if (patchMetaKey)
    {
        NN_RESULT_DO(CheckProgramContentRights(*patchMetaKey));
    }

    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::CheckApplicationResumeRights(nn::ncm::ApplicationId id) NN_NOEXCEPT
{
    // INFO: アプリケーション記録から、アプリとパッチの ContentMetaKey 取得
    util::optional<ncm::ContentMetaKey> appMetaKey;
    util::optional<ncm::ContentMetaKey> patchMetaKey;
    NN_RESULT_DO(m_pRecordDb->FindLaunchableApplicationAndPatch(&appMetaKey, &patchMetaKey, id));
    NN_RESULT_THROW_UNLESS(appMetaKey, ResultApplicationContentNotFound());

    // アプリケーションの Program だけ権限チェックを行う
    NN_RESULT_DO(CheckProgramContentRights(*appMetaKey));

    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::UpdateApplicationRightsOnClientWorkingInfo(ApplicationRightsOnClientWorkingInfo* pRefValue, const ncm::ContentMetaKey& key, const account::Uid& uid) NN_NOEXCEPT
{
    bool hasContentMetaKey;
    NN_RESULT_DO(m_pIntegrated->HasContentMetaKey(&hasContentMetaKey, key));
    if (!hasContentMetaKey)
    {
        // 特に更新は必要ない
        NN_RESULT_SUCCESS;
    }

    // TORIAEZU: 記録のストレージ情報はあてにならないため取り直す
    ncm::StorageId storageId;
    NN_RESULT_DO(m_pIntegrated->GetContentMetaStorage(&storageId, key));

    if (storageId == ncm::StorageId::GameCard)
    {
        pRefValue->flags.Set<ApplicationRightsOnClientFlag_HasAvailableRights>();
    }
    else
    {
        NN_ABORT_UNLESS_LESS(pRefValue->currentIndex, pRefValue->listCount);
        int offset = 0;
        while (NN_STATIC_CONDITION(true))
        {
            int outCount;
            auto remainListCount = pRefValue->listCount - pRefValue->currentIndex;
            if (key.type == ncm::ContentMetaType::AddOnContent)
            {
                NN_RESULT_DO(GetRightsIdListLightly(&outCount, &pRefValue->pWorkingList[pRefValue->currentIndex], remainListCount, key, m_pIntegrated, offset));
            }
            else
            {
                NN_RESULT_DO(GetRightsIdList(&outCount, &pRefValue->pWorkingList[pRefValue->currentIndex], remainListCount, key, m_pIntegrated, offset));
                // 内部鍵対策
                if (outCount == 0)
                {
                    bool hasKey;
                    NN_RESULT_DO(m_pIntegrated->HasContentMetaKey(&hasKey, key));
                    if (hasKey)
                    {
                        pRefValue->flags.Set<ApplicationRightsOnClientFlag_HasAvailableRights>();
                    }
                }
            }
            pRefValue->currentIndex += outCount;
            offset += outCount;
            if (pRefValue->currentIndex == pRefValue->listCount)
            {
                ApplicationRightsOnClientFlag flags;
                NN_RESULT_DO(MakeApplicationRightsOnClientFlag(&flags, pRefValue->pWorkingList, pRefValue->currentIndex, uid));
                pRefValue->flags |= flags;
                pRefValue->currentIndex = 0;
            }
            if (outCount != remainListCount)
            {
                break;
            }
        }
    }
    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::GetApplicationRightsOnClient(int* outCount, ApplicationRightsOnClient* outList, int listCount, ncm::ApplicationId appId, const account::Uid& uid, ApplicationRightsOnClientQueryFlag flags) NN_NOEXCEPT
{
    NN_RESULT_TRY(GetApplicationRightsOnClientImpl(outCount, outList, listCount, appId, uid, flags))
        NN_RESULT_CATCH_CONVERT(fs::ResultDataCorrupted, ResultCorruptedContentFound())
    NN_RESULT_END_TRY
    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::GetApplicationRightsOnClientImpl(int* outCount, ApplicationRightsOnClient* outList, int listCount, ncm::ApplicationId appId, const account::Uid& uid, ApplicationRightsOnClientQueryFlag flags) NN_NOEXCEPT
{
    ApplicationRecordAccessor list;
    NN_RESULT_DO(m_pRecordDb->Open(&list, appId));

    constexpr int AppListCount = 2;
    es::RightsId appRightsIdList[AppListCount];
    constexpr int AocListCount = 16;
    es::RightsId aocRightsIdList[AocListCount];
    ApplicationRightsOnClientWorkingInfo infoList[2];
    int infoCount = 0;

    // NA が連携されていないアカウントだった場合、機器認証のみを問い合わせるようにする
    account::Uid queryUid = IsAvailableNaIdToken(uid) ? uid : account::InvalidUid;

    ApplicationRightsOnClientWorkingInfo* pAppInfo = nullptr;
    if (flags.Test<ApplicationRightsOnClientQueryFlag_Application>())
    {
        infoList[infoCount] = { appId, queryUid, ApplicationContentType::Application, appRightsIdList, AppListCount };
        pAppInfo = &infoList[infoCount];
        infoCount++;
    }

    ApplicationRightsOnClientWorkingInfo* pAocInfo = nullptr;
    if (flags.Test<ApplicationRightsOnClientQueryFlag_AddOnContent>())
    {
        infoList[infoCount] = { appId, queryUid, ApplicationContentType::AddOnContent, aocRightsIdList, AocListCount };
        pAocInfo = &infoList[infoCount];
        infoCount++;
    }

    for (const auto& storageKey : list.GetSpan())
    {
        const auto& key = storageKey.key;
        if (key.type == ncm::ContentMetaType::Application && pAppInfo)
        {
            NN_RESULT_DO(UpdateApplicationRightsOnClientWorkingInfo(pAppInfo, key, queryUid));
        }
        else if (key.type == ncm::ContentMetaType::Patch && pAppInfo)
        {
            NN_RESULT_DO(UpdateApplicationRightsOnClientWorkingInfo(pAppInfo, key, queryUid));
        }
        else if (key.type == ncm::ContentMetaType::AddOnContent && pAocInfo)
        {
            NN_RESULT_DO(UpdateApplicationRightsOnClientWorkingInfo(pAocInfo, key, queryUid));
        }
        else
        {
            continue;
        }
    }

    int index = 0;
    if (pAppInfo)
    {
        NN_RESULT_THROW_UNLESS(index < listCount, ResultBufferNotEnough());
        ApplicationRightsOnClient clientRights;
        NN_RESULT_DO(MakeApplicationRightsOnClient(&clientRights, *pAppInfo));
        if (clientRights.HasUnavailableRights())
        {
            // アプリかパッチの権利のどちらかがあるかもしれないが、
            // 1つでも権利がなかったら起動出来ないので、権利がないという状態とする
            clientRights._flags.Set<ApplicationRightsOnClientFlag_HasAvailableRights>(false);
        }
        if (clientRights._flags.IsAnyOn())
        {
            outList[index++] = clientRights;
        }
    }
    if (pAocInfo)
    {
        NN_RESULT_THROW_UNLESS(index < listCount, ResultBufferNotEnough());
        ApplicationRightsOnClient clientRights;
        NN_RESULT_DO(MakeApplicationRightsOnClient(&clientRights, *pAocInfo));
        if (clientRights._flags.IsAnyOn())
        {
            outList[index++] = clientRights;
        }
    }

    *outCount = index;

    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::BoostSystemMemoryResourceLimit(int64_t boostSize) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    // boostSize のチェックも pm に任せる
    NN_RESULT_DO(pm::BoostSystemMemoryResourceLimit(boostSize));
    GetSharedDevelopInterfaceServerImpl().SetCurrentBoostedSystemMemoryResourceLimitValue(boostSize);
    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(boostSize);
    NN_RESULT_SUCCESS;
#endif
}

Result ApplicationLaunchManager::RegisterContentExternalKeyImpl(
    const ncm::ContentMetaKey& key,
    uint8_t programIndex,
    ncm::ContentType contentType,
    ncm::StorageId storageId) const NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    fs::RightsId fsRightsId;
    uint8_t keyGeneration;
    NN_RESULT_DO(m_pIntegrated->GetContentRightsId(&fsRightsId, &keyGeneration, key, contentType, programIndex, storageId));

    es::RightsIdIncludingKeyId rightsId = es::RightsIdIncludingKeyId::Construct(fsRightsId);
    if (!rightsId.IsExternalKey())
    {
        NN_RESULT_SUCCESS;
    }

    if (key.type == ncm::ContentMetaType::Patch && storageId == ncm::StorageId::GameCard)
    {
        account::NintendoAccountId naIdList[account::UserCountMax];
        int count;
        bool hasRestrictedAccount = es::ListAccountRestrictedRightsUser(&count, naIdList, NN_ARRAY_SIZE(naIdList));

        es::RightsStatus status;
        if (hasRestrictedAccount)
        {
            NN_SDK_ASSERT(count > 0);
            NN_RESULT_DO(es::CheckRightsStatus(&status, 1, &rightsId, 1, naIdList, count));
        }
        else
        {
            NN_RESULT_DO(es::CheckRightsStatus(&status, 1, &rightsId, 1));
        }
        if (status.status == es::RightsStatus::NotAvailable)
        {
            NN_RESULT_DO(m_pGameCardManager->ImportTicket(fsRightsId));
            NN_DETAIL_NS_TRACE("[ApplicationLaunchManager] Import ticket on card\n");
        }
    }

    int keyGenerationToInt = static_cast<int>(keyGeneration);
    NN_RESULT_TRY(es::RegisterTitleKey(&rightsId, &keyGenerationToInt, 1))
        NN_RESULT_CATCH(es::ResultTitleKeyAlreadyRegistered) {}
    NN_RESULT_END_TRY

#else
    NN_UNUSED(key);
    NN_UNUSED(contentType);
    NN_UNUSED(storageId);
    NN_UNUSED(programIndex);

#endif

    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::RegisterApplicationExternalKey(ncm::ApplicationId id) const NN_NOEXCEPT
{
    ApplicationRecordAccessor list;
    NN_RESULT_DO(m_pRecordDb->Open(&list, id));

    const auto count = list.Count();
    for (auto i = 0; i < count; i++)
    {
        if (list[i].key.type == ncm::ContentMetaType::Patch ||
            list[i].key.type == ncm::ContentMetaType::Delta)
        {
            bool hasContentMetaKey;
            NN_RESULT_DO(m_pIntegrated->HasContentMetaKey(&hasContentMetaKey, list[i].key));
            if (hasContentMetaKey)
            {
                NN_RESULT_DO(RegisterContentsExternalKey(list[i].key));
            }
        }
    }

    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::RegisterContentsExternalKey(const ncm::ContentMetaKey& key, ncm::StorageId storageId) const NN_NOEXCEPT
{
    ncm::ContentInfo buffer[16];
    const int Count = static_cast<int>(NN_ARRAY_SIZE(buffer));
    int offset = 0;
    int outCount;

    ncm::StorageId targetStorage;
    if (storageId == ncm::StorageId::Any)
    {
        NN_RESULT_DO(m_pIntegrated->GetContentMetaStorage(&targetStorage, key));
    }
    else
    {
        targetStorage = storageId;
    }

    do
    {
        NN_RESULT_DO(m_pIntegrated->ListContentInfo(&outCount, buffer, Count, key, offset, targetStorage));
        offset += outCount;

        for (int j = 0; j < outCount; j++)
        {
            // TORIAEZU: ゲームカードのMeta はうまく GetPath 出来ない
            //           アプリケーション実行中にコンテンツメタを参照することは無いはずなので、鍵登録の対象から除外する
            if (buffer[j].type != ncm::ContentType::Meta)
            {
                NN_RESULT_DO(RegisterContentExternalKeyImpl(key, GetProgramIndex(buffer[j]), buffer[j].type, targetStorage));
            }
        }
    } while (outCount == Count);

    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::CheckProgramContentRights(const ncm::ContentMetaKey& key) NN_NOEXCEPT
{
    ncm::StorageId storageId;
    NN_RESULT_DO(m_pIntegrated->GetContentMetaStorage(&storageId, key));

    // オンカードパッチのチケットは、初めて RegisterContentExternalKey が実行されたときにインポートされるので、
    // 権限チェックをスキップする
    if (key.type == ncm::ContentMetaType::Patch && storageId == ncm::StorageId::GameCard)
    {
        NN_RESULT_SUCCESS;
    }

    const ncm::ContentType ProgramType = ncm::ContentType::Program;
    int count = 0;
    int offset = 0;
    const int ListCount = 16;
    uint8_t indexList[ListCount];
    do
    {
        NN_RESULT_DO(m_pIntegrated->ListProgramIndex(&count, indexList, ListCount, key, ProgramType, offset));
        for (int i = 0; i < count; i++)
        {
            NN_RESULT_DO(m_pTicketManager->VerifyContentRights(key, ProgramType, indexList[i], storageId));
        }
        offset += count;
    } while (count == ListCount);

    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::CheckContentsRights(const ncm::ContentMetaKey& key, ncm::StorageId storageId) const NN_NOEXCEPT
{
    ncm::StorageId targetStorage;
    if (storageId == ncm::StorageId::Any)
    {
        NN_RESULT_DO(m_pIntegrated->GetContentMetaStorage(&targetStorage, key));
    }
    else
    {
        targetStorage = storageId;
    }

    // オンカードパッチのチケットは、初めて RegisterContentExternalKey が実行されたときにインポートされるので、
    // 権限チェックをスキップする
    if (key.type == ncm::ContentMetaType::Patch && targetStorage == ncm::StorageId::GameCard)
    {
        NN_RESULT_SUCCESS;
    }

    ncm::ContentInfo buffer[16];
    const int Count = static_cast<int>(NN_ARRAY_SIZE(buffer));
    int offset = 0;
    int outCount;

    do
    {
        NN_RESULT_DO(m_pIntegrated->ListContentInfo(&outCount, buffer, Count, key, offset, targetStorage));
        offset += outCount;

        for (int j = 0; j < outCount; j++)
        {
            // TORIAEZU: ゲームカードのMeta はうまく GetPath 出来ない
            //           アプリケーション実行中にコンテンツメタを参照することは無いはずなので、鍵登録の対象から除外する
            if (buffer[j].type != ncm::ContentType::Meta)
            {
                NN_RESULT_DO(m_pTicketManager->VerifyContentRights(key, buffer[j].type, GetProgramIndex(buffer[j]), targetStorage));
            }
        }
    } while (outCount == Count);

    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::GetRequiredApplicationVersion(uint32_t* outValue, ncm::ApplicationId id) const NN_NOEXCEPT
{
    ApplicationRecordAccessor list;
    NN_RESULT_DO(m_pRecordDb->Open(&list, id));

    const auto count = list.Count();
    *outValue = 0;
    for (auto i = 0; i < count; i++)
    {
        if (list[i].key.type == ncm::ContentMetaType::AddOnContent)
        {
            bool hasContentMetaKey;
            NN_RESULT_DO(m_pIntegrated->HasContentMetaKey(&hasContentMetaKey, list[i].key));
            if (hasContentMetaKey)
            {
                uint32_t version;
                NN_RESULT_DO(m_pIntegrated->GetRequiredApplicationVersion(&version, list[i].key));
                *outValue = std::max(*outValue, version);
            }
        }
    }

    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::RegisterAllAddOnContentLocationAndExternalKey(ncm::ApplicationId id) NN_NOEXCEPT
{
    const int BufferCount = 2048;
    static ApplicationContentMetaStatus s_ListBuffer[BufferCount];

    static nn::os::Mutex s_Mutex(false);
    std::lock_guard<nn::os::Mutex> lock(s_Mutex);

    int listCount;
    NN_RESULT_TRY(ListAvailableAddOnContentImpl(&listCount, s_ListBuffer, BufferCount, id, 0, *m_pRecordDb, *m_pIntegrated, *this))
        NN_RESULT_CATCH(ResultApplicationRecordNotFound) { listCount = 0; }
    NN_RESULT_END_TRY;

    lr::AddOnContentLocationResolver ar;
    NN_RESULT_DO(lr::OpenAddOnContentLocationResolver(&ar));

    for (int i = 0; i < listCount; ++i)
    {
        auto& status = s_ListBuffer[i];

        NN_RESULT_DO(ar.RegisterAddOnContentStorage({ status.id }, status.installedStorage));

        auto contentMetaKey = ncm::ContentMetaKey::Make(status.id, status.version, ncm::ContentMetaType::AddOnContent);
        NN_RESULT_DO(RegisterContentsExternalKey(contentMetaKey, status.installedStorage));
    }

    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::UnregisterAllAddOnContentLocation() NN_NOEXCEPT
{
    lr::AddOnContentLocationResolver ar;
    NN_RESULT_DO(lr::OpenAddOnContentLocationResolver(&ar));
    ar.UnregisterAllAddOnContentPath();
    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::RegisterLogoDataExternalKey(const ncm::ContentMetaKey& key, uint8_t programIndex) const NN_NOEXCEPT
{
    ncm::StorageId storageId;
    NN_RESULT_DO(m_pIntegrated->GetContentMetaStorage(&storageId, key));
    NN_RESULT_DO(RegisterContentExternalKeyImpl(key, programIndex, ncm::ContentType::Program, storageId));

    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::GetApplicationLaunchInfo(ApplicationLaunchInfo* outValue, ncm::ApplicationId id) NN_NOEXCEPT
{
    util::optional<ncm::ContentMetaKey> appKey;
    util::optional<ncm::ContentMetaKey> patchKey;
    NN_RESULT_DO(m_pRecordDb->FindLaunchableApplicationAndPatch(&appKey, &patchKey, id));
    NN_RESULT_THROW_UNLESS(appKey, ResultMainApplicationNotFound());

    auto getStorageId = [this](ncm::StorageId* outValue, const util::optional<ncm::ContentMetaKey>& key) NN_NOEXCEPT -> Result
    {
        if (!key)
        {
            *outValue = ncm::StorageId::None;
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_DO(m_pIntegrated->GetContentMetaStorage(outValue, *key));
        NN_RESULT_SUCCESS;
    };

    ncm::StorageId appStorageId;
    NN_RESULT_DO(getStorageId(&appStorageId, appKey));
    ncm::StorageId patchStorageId;
    NN_RESULT_DO(getStorageId(&patchStorageId, patchKey));

    auto mainContentKey = DecideMainContentMetaKey(appKey, patchKey);
    auto version = mainContentKey->version;
    constexpr int32_t DefaultFlags = LaunchProgramFlags_NotifyDebug | LaunchProgramFlags_NotifyStart | LaunchProgramFlags_NotifyExit;

    ApplicationLaunchInfo info = { id, version, DefaultFlags , appStorageId, patchStorageId, static_cast<Bit8>(ApplicationLaunchInfoFlag::None), {}, m_LaunchPropertyManager.GenerateApplicationLaunchInfoId() };
    *outValue = info;
    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::AcquireApplicationLaunchInfo(ApplicationLaunchInfo* outValue, os::ProcessId processId) NN_NOEXCEPT
{
    NN_RESULT_DO(m_LaunchPropertyManager.Acquire(outValue, processId));
    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::GetProgramIndexForDevelop(uint8_t* outValue, const ApplicationLaunchInfo& info) const NN_NOEXCEPT
{
    NN_RESULT_DO(m_LaunchPropertyManager.GetProgramIndex(outValue, info));
    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::MakeApplicationRightsOnClient(ApplicationRightsOnClient* outValue, const ApplicationRightsOnClientWorkingInfo& info) NN_NOEXCEPT
{
    ApplicationRightsOnClientFlag flags = info.flags;
    if (info.currentIndex > 0)
    {
        ApplicationRightsOnClientFlag tmpFlags;
        NN_RESULT_DO(MakeApplicationRightsOnClientFlag(&tmpFlags, info.pWorkingList, info.currentIndex, info.uid));
        flags |= tmpFlags;
    }
    ApplicationRightsOnClient rights = { info.appId, info.uid, flags, info.contentType };
    *outValue = rights;
    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchManager::RedirectApplicationProgramForDevelop(ncm::ApplicationId id, uint8_t programIndex) NN_NOEXCEPT
{
    ncm::StorageId storageIdList[IntegratedContentManager::MaxStorageCount];
    auto count = m_pIntegrated->GetRegisteredStorages(storageIdList, static_cast<int>(NN_ARRAY_SIZE(storageIdList)));
    for (auto storageId : util::MakeSpan(storageIdList, count))
    {
        lr::LocationResolver locationResolver;
        NN_RESULT_DO(lr::OpenLocationResolver(&locationResolver, storageId));
        locationResolver.ClearApplicationRedirection();

        if (m_pRecordDb->Has(id))
        {
            util::optional<ncm::ContentMetaKey> appKey;
            util::optional<ncm::ContentMetaKey> patchKey;
            NN_RESULT_DO(
                m_pRecordDb->FindLaunchableApplicationAndPatch(
                &appKey, &patchKey, id, storageId, ncm::StorageId::Any));
            if (appKey)
            {
                lr::Path path;
                NN_RESULT_DO(GetContentPathByStorageId(&path, *appKey, ncm::ContentType::Program, programIndex, storageId));
                locationResolver.RedirectApplicationProgramPath(GetProgramId(id, programIndex), path);
            }
        }
    }

    NN_RESULT_SUCCESS;
}

}}}
