﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <nn/fs.h>
#include <nn/fs/fs_Content.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/ns/ns_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include "ns_ApplicationControlFileUtil.h"
#include "ns_ApplicationLanguage.h"
#include "ns_ProgramIndexUtil.h"

namespace nn { namespace ns { namespace srv {
    namespace {
        typedef kvdb::BoundedString<48> Path;

        Path MakeApplicationControlPropertyPath(const char* mountName) NN_NOEXCEPT
        {
            return Path::MakeFormat("%s:/control.nacp", mountName);
        }

        Path MakeIconDataPath(const char* mountName, detail::ApplicationLanguage language) NN_NOEXCEPT
        {
            return Path::MakeFormat("%s:/icon_%s.dat", mountName, GetApplicationLanguageString(language));
        }

        Path MakeUniqueMountName() NN_NOEXCEPT
        {
            static std::atomic_uint g_Count;
            Path name;
            name.AssignFormat("@nsac%08x", g_Count.fetch_add(1));

            return name;
        }

        Result OpenIconFile(util::optional<fs::FileHandle>* outValue, const char* mountName, settings::LanguageCode language) NN_NOEXCEPT
        {
            auto appLanguage = ToApplicationLanguage(language);
            NN_RESULT_THROW_UNLESS(appLanguage, ResultApplicationControlDataNotFound());

            fs::FileHandle file;
            auto result = fs::OpenFile(&file, MakeIconDataPath(mountName, *appLanguage), fs::OpenMode_Read);
            if (result.IsSuccess())
            {
                *outValue = file;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_THROW_UNLESS(result <= fs::ResultPathNotFound(), result);

            // 優先度順に合わせて適切なアイコンを取得できるようにする
            settings::LanguageCode settingsLanguage;
            settings::GetLanguageCode(&settingsLanguage);
            auto applicationLanguage = ToApplicationLanguage(settingsLanguage);
            NN_RESULT_THROW_UNLESS(applicationLanguage, ResultApplicationLanguageNotFound());

            auto languageList = GetLanguagePriorityList(*applicationLanguage);
            for (int i = 0; i < LanguagePriorityListLength; i++)
            {
                result = fs::OpenFile(&file, MakeIconDataPath(mountName, languageList[i]), fs::OpenMode_Read);
                if (result.IsSuccess())
                {
                    *outValue = file;
                    NN_RESULT_SUCCESS;
                }
                NN_RESULT_THROW_UNLESS(result <= fs::ResultPathNotFound(), result);
            }

            *outValue = util::nullopt;
            NN_RESULT_SUCCESS;
        }
    }

    Result ReadApplicationControlData(size_t* outValue, void* buffer, size_t bufferSize, ncm::ApplicationId applicationId, uint8_t programIndex, const char* path, settings::LanguageCode language) NN_NOEXCEPT
    {
        const size_t propertySize = sizeof(ApplicationControlProperty);
        NN_RESULT_THROW_UNLESS(bufferSize >= propertySize, ResultBufferNotEnough());

        auto mountName = MakeUniqueMountName();
        NN_RESULT_DO(fs::MountContent(mountName, path, GetProgramId(applicationId, programIndex), fs::ContentType_Control));
        NN_UTIL_SCOPE_EXIT{ fs::Unmount(mountName); };

        size_t valueSize = 0;
        {
            fs::FileHandle file;
            NN_RESULT_DO(fs::OpenFile(&file, MakeApplicationControlPropertyPath(mountName), fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };
            NN_RESULT_DO(fs::ReadFile(file, 0, buffer, propertySize));
            valueSize += propertySize;
        }

        util::optional<fs::FileHandle> file;
        NN_RESULT_DO(OpenIconFile(&file, mountName, language));

        if (file)
        {
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(*file); };
            int64_t fileSize;
            NN_RESULT_DO(fs::GetFileSize(&fileSize, *file));
            auto iconDataSize = static_cast<size_t>(fileSize);
            NN_RESULT_THROW_UNLESS(iconDataSize <= bufferSize - propertySize, ResultBufferNotEnough());

            auto bufferPtr = static_cast<char*>(buffer);
            NN_RESULT_DO(fs::ReadFile(*file, 0, &bufferPtr[propertySize], iconDataSize));
            valueSize += iconDataSize;
        }

        *outValue = valueSize;

        NN_RESULT_SUCCESS;
    }

    Result ReadApplicationControlProperty(ApplicationControlProperty* pOut, ncm::ApplicationId applicationId, uint8_t programIndex, const char* path) NN_NOEXCEPT
    {
        auto mountName = MakeUniqueMountName();
        NN_RESULT_DO(fs::MountContent(mountName, path, GetProgramId(applicationId, programIndex), fs::ContentType_Control));
        NN_UTIL_SCOPE_EXIT{ fs::Unmount(mountName); };

        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, MakeApplicationControlPropertyPath(mountName), fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };
        NN_RESULT_DO(fs::ReadFile(file, 0, pOut, sizeof(*pOut)));

        NN_RESULT_SUCCESS;
    }
}}}
