﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdlib>
#include <nn/es/es_Api.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_ApplicationContentMetaStatusUtil.h>
#include <nn/ns/srv/ns_ApplicationLaunchManager.h>
#include <nn/ns/srv/ns_OsUtil.h>
#include <nn/util/util_Optional.h>

#include "ns_ProgramIndexUtil.h"
#include "ns_ShellUtil.h"
#include "ns_StringUtil.h"

namespace nn { namespace ns { namespace srv {

namespace
{
    template<class predicate>
    Result GetStorageIdBy(ncm::StorageId* out, const ncm::ContentMetaKey& key, predicate finder) NN_NOEXCEPT
    {
        NN_RESULT_TRY(finder(out, key))
            NN_RESULT_CATCH(ncm::ResultContentMetaNotFound)
            {
                *out = ncm::StorageId::None;
            }
        NN_RESULT_END_TRY;
        NN_RESULT_SUCCESS;
    }

    Result GetStorageId(ncm::StorageId* out, const ncm::ContentMetaKey& key, const IntegratedContentManager& integrated) NN_NOEXCEPT
    {
        return GetStorageIdBy(out, key, [&integrated](ncm::StorageId* out, const ncm::ContentMetaKey& key) -> Result
        {
            return integrated.GetInstalledContentMetaStorage(out, key);
        });
    }

    Result GetContentsRight(bool* out, const ncm::ContentMetaKey& key, ncm::StorageId installed, ApplicationLaunchManager& launchManager) NN_NOEXCEPT
    {
        NN_RESULT_TRY(launchManager.CheckContentsRights(key, installed))
            NN_RESULT_CATCH(ResultApplicationLaunchRightsNotFound)
            {
                *out = false;
                NN_RESULT_SUCCESS;
            }
            // SIGLO-50432 2.1NUP 向け暫定対応
            // 3.0.0 向けには Result を定義しなおす。
            NN_RESULT_CATCH(ResultInactiveNintendoAccount)
            {
                *out = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        *out = true;
        NN_RESULT_SUCCESS;
    }

    Result CheckContentMetaRights(ContentMetaRightsCheck* outValue, IntegratedContentManager& integrated, const ncm::ContentMetaKey& key, ncm::StorageId storageId) NN_NOEXCEPT
    {
        ncm::ContentInfo buffer[16];
        const int Count = static_cast<int>(NN_ARRAY_SIZE(buffer));
        int offset = 0;
        int outCount;

        auto check = ContentMetaRightsCheck::NotNeeded;
        do
        {
            NN_RESULT_DO(integrated.ListContentInfo(&outCount, buffer, Count, key, offset, storageId));
            offset += outCount;

            for (int j = 0; j < outCount; j++)
            {
                auto contentType = buffer[j].type;
                if (contentType != ncm::ContentType::Meta)
                {
                    fs::RightsId fsRightsId;
                    NN_RESULT_DO(integrated.GetContentRightsId(&fsRightsId, key, contentType, GetProgramIndex(buffer[j]), storageId));

                    es::RightsIdIncludingKeyId rightsId = es::RightsIdIncludingKeyId::Construct(fsRightsId);
                    if (!rightsId.IsExternalKey())
                    {
                        continue;
                    }

#ifdef NN_BUILD_CONFIG_OS_HORIZON
                    // 1つの RightsId に対して最大で持ちえるチケットの数(最大でリンクされる VA の数 + DA + コモンチケット)
                    const int MaxTicketNum = nn::account::UserCountMax + 1 + 1;

                    // 権利を持っているかを確認
                    es::LightTicketInfo ticketInfo[MaxTicketNum];

                    int ticketCount = es::ListLightTicketInfo(ticketInfo, MaxTicketNum, rightsId);

                    // チケットがない場合、権利を持っていない
                    if (ticketCount == 0)
                    {
                        *outValue = ContentMetaRightsCheck::NoRights;
                        NN_RESULT_SUCCESS;
                    }

                    if (ticketInfo[0].accountId == 0)
                    {
                        check = ContentMetaRightsCheck::CommonRights;
                    }
                    else
                    {
                        check = ContentMetaRightsCheck::PersonalizedRights;
                    }
#endif
                }
            }
        } while (outCount == Count);

        *outValue = check;
        NN_RESULT_SUCCESS;
    }
} // namespace

Result  ListApplicationContentMetaStatusImpl(
    int* outCount, ns::ApplicationContentMetaStatus outList[], int maxOutCount, ncm::ApplicationId appId, std::int32_t offset, ApplicationRecordDatabase& recordDb, IntegratedContentManager& integrated, bool checkRights) NN_NOEXCEPT
{
    ApplicationRecordAccessor list;
    NN_RESULT_DO(recordDb.Open(&list, appId));
    const auto readCount = std::min(list.Count() - offset, maxOutCount);

    for (int i = 0; i < readCount; i++)
    {
        auto& key = list[offset + i].key;

        ncm::StorageId installed;
        NN_RESULT_TRY(integrated.GetInstalledContentMetaStorage(&installed, key))
            NN_RESULT_CATCH(ncm::ResultContentMetaNotFound)
            {
                installed = ncm::StorageId::None;
            }
        NN_RESULT_END_TRY

        ApplicationContentMetaStatus status = {};
        status.id = key.id;
        status.version = key.version;
        status.type = key.type;
        status.installedStorage = installed;
        if (installed != ncm::StorageId::None && checkRights)
        {
            ContentMetaRightsCheck check;
            NN_RESULT_DO(CheckContentMetaRights(&check, integrated, key, installed));
            status.rightsCheck = check;
        }
        else
        {
            status.rightsCheck = ContentMetaRightsCheck::NotChecked;
        }
        outList[i] = status;
    }

    *outCount = readCount;
    NN_RESULT_SUCCESS;
}

Result ListAvailableAddOnContentImpl(int* outCount, ns::ApplicationContentMetaStatus outList[], int maxOutCount, ncm::ApplicationId appId, int addOnContentOffset, ApplicationRecordDatabase& recordDb, IntegratedContentManager& integrated, ApplicationLaunchManager& launchManager) NN_NOEXCEPT
{
    ApplicationRecordAccessor list;
    NN_RESULT_DO(recordDb.Open(&list, appId));

    // TODO: アプリケーション以外が追加コンテンツを利用する場合、
    //       取得できない・取得したものが現在実行中のアプリケーションの為、正しくありません
    arp::ApplicationLaunchProperty property;
    bool hasProperty = true;
    NN_RESULT_TRY(GetCurrentApplicationLaunchProperty(&property))
        NN_RESULT_CATCH_ALL
        {
            hasProperty = false;
        }
    NN_RESULT_END_TRY;

    int listedCount = 0;
    int foundCount = 0;
    bool willRecommend = false;
    for (int i = 0; i < list.Count() && listedCount < maxOutCount; i++)
    {
        auto& key = list[i].key;
        if (key.type != ncm::ContentMetaType::AddOnContent)
        {
            continue;
        }

        ncm::StorageId installed;
        NN_RESULT_DO(GetStorageIdBy(&installed, key, [&integrated](ncm::StorageId* out, const ncm::ContentMetaKey& key) -> Result
        {
            return integrated.GetContentMetaStorage(out, key);
        }));
        if (installed == ncm::StorageId::None)
        {
            continue;
        }

        if (hasProperty == true)
        {
            // 起動アプリがAOCの必須アプリバージョンを満たさない場合はリストアップしない
            uint32_t requiredApplicationVersion;
            NN_RESULT_DO(integrated.GetRequiredApplicationVersionByStorageId(&requiredApplicationVersion, key, installed));
            if (property.version < requiredApplicationVersion)
            {
                continue;
            }
        }

        bool right;
        NN_RESULT_DO(GetContentsRight(&right, key, installed, launchManager));
        if (right)
        {
            if (foundCount >= addOnContentOffset)
            {
                ApplicationContentMetaStatus status = {};
                status.id = key.id;
                status.version = key.version;
                status.type = key.type;
                status.installedStorage = installed;
                outList[listedCount] = status;
                listedCount++;
            }
            foundCount++;
        }
        else if (!willRecommend)
        {
            willRecommend = true;
        }
    }

    if (willRecommend)
    {
        NN_RESULT_DO(recordDb.RecommendCleanupAddOnContentsWithNoRights(appId));
    }

    *outCount = listedCount;
    NN_RESULT_SUCCESS;
}

Result CheckRecordAndDatabaseImpl(bool *outHasRecord, bool *outInstalled, ncm::ContentMetaKey key, ncm::ApplicationId id, ApplicationRecordDatabase& recordDb, IntegratedContentManager& integrated) NN_NOEXCEPT
{
    if (!recordDb.Has(id))
    {
        *outHasRecord = false;
    }
    else
    {
        ApplicationRecordAccessor accessor;
        NN_RESULT_DO(recordDb.Open(&accessor, id));
        *outHasRecord = accessor.Has(key);
    }
    ncm::StorageId storageId;
    NN_RESULT_DO(GetStorageId(&storageId, key, integrated));
    *outInstalled = (storageId != ncm::StorageId::None);

    NN_RESULT_SUCCESS;
}

}}}
