﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nsd/nsd_ApiForMenu.h>
#include <nn/nsd/detail/nsd_DetailApiForMenu.h>
#include <nn/nsd/detail/nsd_DetailApi.h>
#include <nn/nsd/detail/nsd_Log.h>

#include <nn/nsd/detail/fs/nsd_Fs.h>
#include <nn/nsd/detail/config/nsd_Config.h>
#include <nn/nsd/detail/device/nsd_Device.h>
#include <nn/nsd/detail/json/nsd_JsonParser.h>
#include <nn/nsd/detail/json/nsd_RapidJsonEventAccepterForClaims.h>
#include <nn/nsd/detail/jwt/nsd_JwtParser.h>
#include <nn/nsd/detail/jwt/nsd_JwtPublicKeyData.h>

#include <nn/nsd/nsd_ResultPrivate.h>
#include <nn/nsd/nsd_TypesPrivate.h>
#include <nn/nsd/detail/nsd_FqdnResolver.h>

#include <nn/nn_SdkAssert.h>
#include <nn/crypto.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_ServiceDiscovery.h>
#include <nn/util/util_ScopeExit.h>

#include <rapidjson/document.h>

namespace nn { namespace nsd { namespace detail {

    namespace
    {

        /**
         * @brief   JWT ドキュメントの解析
         * @tparam  VerifierType                    署名検証を行うクラス(nn::crypto::Rsa2048Pkcs1Sha256Verifier など)
         * @tparam  SignatureBuffersize             署名のデータサイズ
         * @param[out]  pOut                        セーブデータの出力先バッファ
         * @param[in]   pJwtDocument                JWT ドキュメントの先頭ポインタ
         * @param[in]   jwtDocumentSize             JWT ドキュメントの全体サイズ
         * @param[in]   expectAlg                   許容する暗号化アルゴリズム
         * @param[in]   isSignatureVerifyEnabled    署名検証を行うかどうか
         * @param[in]   modulusBuffer               公開鍵の係数を指すバッファ
         * @param[in]   modulusSize                 modulusBuffer サイズ
         * @param[in]   exponentBuffer              公開鍵の指数を指すバッファ
         * @param[in]   exponentSize                exponentBuffer サイズ
         */
        template<typename VerifierType, size_t SignatureBuffersize>
        Result ParseJwtDocument(
            nn::nsd::SaveData* pOut,
            const char* pJwtDocument, size_t jwtDocumentSize,
            const char* expectAlg,
            bool isSignatureVerifyEnabled,
            const uint8_t* modulusBuffer, size_t modulusSize,
            const uint8_t* exponentBuffer, size_t exponentSize) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(pOut);
            NN_SDK_ASSERT_NOT_NULL(pJwtDocument);
            NN_SDK_ASSERT_NOT_NULL(expectAlg);

            jwt::JwtElements jwtElements;
            NN_RESULT_DO( jwt::GetJwtElements(&jwtElements, pJwtDocument, jwtDocumentSize) );

            { // ヘッダ解析
                jwt::JwtHeaderData headerResult;
                NN_RESULT_DO( jwt::ParseHeader(&headerResult, jwtElements) );

                // JWT のみ許容
                NN_RESULT_THROW_UNLESS(
                    util::IsEqualString(headerResult.typ, "JWT"),
                    ResultUnexpectedJwtHeaderTyp());

                // 指定された expectAlg("RS256"とか) のみ許容する
                NN_RESULT_THROW_UNLESS(
                    util::IsEqualString(headerResult.alg, expectAlg),
                    ResultUnexpectedJwtHeaderAlg());
            }

            // 署名検証
            if(isSignatureVerifyEnabled)
            {
                VerifierType verifier;
                NN_RESULT_THROW_UNLESS(
                    verifier.Initialize(modulusBuffer, modulusSize, exponentBuffer, exponentSize),
                    ResultJwtSignatureVerifierInitializeFailed());

                size_t signatureBufferSize;
                char signatureBuffer[SignatureBuffersize] = {};
                NN_RESULT_DO( jwt::DecodeSignatureBuffer(
                    &signatureBufferSize, signatureBuffer, sizeof(signatureBuffer), jwtElements) );

                NN_RESULT_DO( jwt::VerifySignature(
                    verifier, signatureBuffer, signatureBufferSize, jwtElements) );
            }

            // クレーム解析
            {
                json::RapidJsonEventAccepterForClaims handlerForJwtClaims(pOut);
                NN_RESULT_DO( jwt::ParseClaim(&handlerForJwtClaims, jwtElements) );

                if(handlerForJwtClaims.HasServerError())
                {
                    NN_DETAIL_NSD_INFO("[NSD] ServerErrorResponse. ErrorCode:%s, Message:%s",
                        handlerForJwtClaims.GetServerErrorCode(),
                        handlerForJwtClaims.GetServerErrorMessage());
                    NN_RESULT_DO( HandleServerError(handlerForJwtClaims.GetServerErrorCode()) );
                }
            }

            NN_RESULT_SUCCESS;
        }

        /**
         * @brief   インポートモードのハンドリング
         * @param[in]   importMode      インポートモード
         * @param[in]   input           インポートしたいセーブデータ
         * @param[in]   currentSaveData 現在のセーブデータ
         * @details
         *  input で指定するセーブデータが importMode でインポート可能かどうか、
         *  input と currentSaveData を比較してチェックします。
         */
        Result HandleImportMode(
            ImportMode importMode,
            const SaveData& input,
            const SaveData& currentSaveData
            ) NN_NOEXCEPT
        {
            if(importMode == ImportMode_All)
            {
                // 環境識別子書き換え禁止チェック
                if (IsChangeEnvironmentIdentifierDisabled() && input.environmentIdentifier != currentSaveData.environmentIdentifier)
                {
                    NN_RESULT_THROW(nn::nsd::ResultEnvironmentIdentifierNotUpdated());
                }

                // どんな設定でも上書き可
                NN_RESULT_SUCCESS;
            }
            else if(importMode == ImportMode_ApplicationSettings)
            {
                // ImportMode_ApplicationSettings で環境識別は書き換え不可
                NN_RESULT_THROW_UNLESS(
                    input.environmentIdentifier == currentSaveData.environmentIdentifier,
                    nn::nsd::ResultEnvironmentIdentifierNotUpdated());
                NN_RESULT_SUCCESS;
            }

            NN_RESULT_THROW(nn::nsd::ResultUnknownImportMode());
        }

        /**
         * @brief   設定である JWT ドキュメントを解析して nn::nsd::SaveData を抽出
         * @param[out]  pOutSaveData        セーブデータ出力先バッファ
         * @param[in]   pJwtBuffer          JWT ドキュメントの先頭バッファ
         * @param[in]   jwtBufferSize       JWT ドキュメントのサイズ
         */
        nn::Result ParseSettings(
            SaveData* pOutSaveData,
            const void* pJwtBuffer,
            size_t jwtBufferSize) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(pOutSaveData);

            memset(pOutSaveData, 0, sizeof(SaveData));

            NN_RESULT_DO( (ParseJwtDocument<nn::crypto::Rsa2048Pkcs1Sha256Verifier, 256>(
                pOutSaveData,
                static_cast<const char*>( pJwtBuffer ),
                jwtBufferSize,
                jwt::SignAlgorithm,
                config::IsJwtSignatureVerificationEnabled,
                jwt::Modulus, sizeof(jwt::Modulus),
                jwt::Exponent, sizeof(jwt::Exponent))) );

            NN_RESULT_SUCCESS;
        }

    }

    //----------------------------------------------------------------
    // 設定の削除
    //
    // すべての削除か、アプリ設定の削除かを選択できる
    //
    nn::Result DeleteSettings(DeleteMode deleteMode) NN_NOEXCEPT
    {
        switch (deleteMode)
        {
        case nn::nsd::DeleteMode_All:
            {
                // 全設定削除につきセーブデータファイルを削除
                // 元からなかった場合は nn::nsd::ResultNotFound() が返る
                NN_RESULT_DO(detail::fs::DeleteSaveData());

                // 環境識別子変更許可フラグも初期値に戻す
                SetChangeEnvironmentIdentifierDisabled(false);
            }
            break;
        case nn::nsd::DeleteMode_ApplicationSettings:
            {
                // セーブファイルから環境識別子を読む
                EnvironmentIdentifier saveEnv;
                NN_RESULT_DO( detail::fs::GetEnvironmentIdentifier( &saveEnv ) ); // 存在しない場合は nn::nsd::ResultNotFound()

                // 削除後の環境識別子(=環境依存のデフォルト値) と比較
                if ( saveEnv == detail::device::GetEnvironmentIdentifierFromSettings() )
                {
                    // 等しい場合は全設定を削除できるのでそうする
                    NN_RESULT_DO(detail::fs::DeleteSaveData());
                }
                else
                {
                    // 異なる場合はアプリケーション設定のみを削除(有効フラグはそのまま)
                    NN_RESULT_DO(detail::fs::FillEmptyApplicationSetting());
                }
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }

        // FqdnResolver インスタンスのセーブデータキャッシュ更新は、
        // デバイスのハードリセット後のプロセス起動時にのみ反映される仕様

        NN_RESULT_SUCCESS;
    }

    //----------------------------------------------------------------
    // 設定のインポート
    //
    nn::Result ImportSettings(
        const void* pBuffer, size_t bufferSize,
        void* pWorkBuffer, size_t workBufferSize,
        ImportMode importMode) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pBuffer);
        NN_SDK_ASSERT_NOT_NULL(pWorkBuffer);
        NN_SDK_ASSERT_GREATER_EQUAL(workBufferSize, WorkBufferSizeForImportSettings);
        NN_UTIL_SCOPE_EXIT
        {
            // FQDN とかが入ってるバッファなので念のため最後に全部消す。
            memset(pWorkBuffer, 0, workBufferSize);
        };

        SaveData* pImportData = static_cast<SaveData*>(pWorkBuffer); // インポートする設定を入れるバッファ
        SaveData* pCurrentSaveData = pImportData + 1; // 既存の設定を読み込むバッファ

        DeviceId ownDeviceId;
        detail::device::LoadOwnDeviceId(&ownDeviceId);

        {
            // 保存済の nsd::SaveData の呼び出し
            nn::Result result = detail::fs::ReadSaveData(pCurrentSaveData);
            if ( ! result.IsSuccess() )
            {
                NN_RESULT_THROW_UNLESS( nn::nsd::ResultNotFound().Includes(result), result );
            }

            // 存在しなかったか、文字列が空の場合デフォルト設定を読み込む。
            // 存在していた場合は デバイスID チェック
            if(nn::nsd::ResultNotFound().Includes(result) || detail::IsInitialSaveData(*pCurrentSaveData))
            {
                const EnvironmentIdentifier env = detail::device::GetEnvironmentIdentifierFromSettings();
                detail::GetDefaultSettings(pCurrentSaveData, ownDeviceId, env);
            }
            else
            {
                NN_RESULT_DO( detail::VerifySaveData(*pCurrentSaveData, ownDeviceId, config::IsDeviceIdCheckEnabled) );
            }
        }

        // インポートするセーブデータを JWT ドキュメントから抽出
        NN_RESULT_DO( ParseSettings(pImportData, pBuffer, bufferSize) );

        // デバイスID チェック
        NN_RESULT_DO( detail::VerifySaveData(*pImportData, ownDeviceId, config::IsDeviceIdCheckEnabled) );

        // インポートするセーブデータ、既存のセーブデータを比べ、
        // インポートモードで許可された内容以外が変更されてようとしたらエラーに落とす
        NN_RESULT_DO( HandleImportMode(importMode, *pImportData, *pCurrentSaveData) );

        // ここまで来たらインポートしてOKなセーブデータ.
        pImportData->isAvailable = true; // 有効に
        NN_RESULT_DO( detail::fs::WriteSaveData(*pImportData) ); // 永続化

        // FqdnResolver インスタンスのセーブデータキャッシュ更新は、
        // デバイスのハードリセット後のプロセス起動時にのみ反映される仕様

        NN_RESULT_SUCCESS;
    }

    //----------------------------------------------------------------
    // サービスディスカバリ設定名を取得
    // キャッシュから取得する
    //
    nn::Result GetSettingName(SettingName* pOut) NN_NOEXCEPT
    {
        return GetFqdnResolverPtr()->GetSettingName(pOut);
    }

    //----------------------------------------------------------------
    // 環境識別子の取得
    // キャッシュから取得する
    //
    // セーブデータが保存されていない場合、デフォルトの値を返す。
    // デフォルトの値は量産機か開発機かで異なる。
    //
    nn::Result GetEnvironmentIdentifier(EnvironmentIdentifier* pOut) NN_NOEXCEPT
    {
        return GetFqdnResolverPtr()->GetEnvironmentIdentifier(pOut);
    }

    bool IsChangeEnvironmentIdentifierDisabled() NN_NOEXCEPT
    {
        nn::settings::system::ServiceDiscoveryControlSettings value = {};
        nn::settings::system::GetServiceDiscoveryControlSettings(&value);

        return value.flag.Test<nn::settings::system::ServiceDiscoveryControlFlag::IsChangeEnvironmentIdentifierDisabled>();
    }

    void SetChangeEnvironmentIdentifierDisabled(bool isDisabled) NN_NOEXCEPT
    {
        // 他フラグを書き換えないように、取得してから ServiceDiscoveryControlFlag::IsChangeEnvironmentIdentifierDisabled のみ切り替える
        nn::settings::system::ServiceDiscoveryControlSettings value = {};
        nn::settings::system::GetServiceDiscoveryControlSettings(&value);
        value.flag.Set<nn::settings::system::ServiceDiscoveryControlFlag::IsChangeEnvironmentIdentifierDisabled>(isDisabled);
        nn::settings::system::SetServiceDiscoveryControlSettings(value);
    }

}}} // nn::nsd::detail
