﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nsd/detail/jwt/nsd_JwtParser.h>

#include <nn/nsd/nsd_ResultPrivate.h>
#include <nn/nsd/detail/util/nsd_StringUtility.h>
#include <nn/nsd/detail/json/nsd_JsonParser.h>
#include <nn/nsd/detail/json/nsd_RapidJsonEventAccepterForHeader.h>

#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Base64.h>

namespace nn { namespace nsd { namespace detail { namespace jwt {

    Result GetJwtElements(
        JwtElements* pOut,
        const char* pJwtDocument, size_t jwtDocumentSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOut);
        NN_SDK_ASSERT_NOT_NULL(pJwtDocument);

        const char *pDot1 = util::SearchCharWithEndPtr(pJwtDocument, pJwtDocument + jwtDocumentSize, '.');
        NN_RESULT_THROW_UNLESS(pDot1 != nullptr, ResultJwtInvalidDotCount());
        const char *pDot2 = util::SearchCharWithEndPtr(pDot1 + 1, pJwtDocument + jwtDocumentSize, '.');
        NN_RESULT_THROW_UNLESS(pDot2 != nullptr, ResultJwtInvalidDotCount());
        const char *pDot3 = util::SearchCharWithEndPtr(pDot2 + 1, pJwtDocument + jwtDocumentSize, '.');
        NN_RESULT_THROW_UNLESS(pDot3 == nullptr, ResultJwtInvalidDotCount()); // 3つ目あったらダメ

        // . で split した要素のサイズが 1 未満だとダメ
        NN_RESULT_THROW_UNLESS(pJwtDocument < pDot1, ResultJwtInvalidElementSize());
        NN_RESULT_THROW_UNLESS(pDot1 + 1 < pDot2, ResultJwtInvalidElementSize());
        NN_RESULT_THROW_UNLESS(pDot2 + 1 < pJwtDocument + jwtDocumentSize - 1, ResultJwtInvalidElementSize());

        pOut->pHeader = pJwtDocument;
        pOut->headerSize = pDot1 - pJwtDocument;

        pOut->pClaim = pDot1 + 1;
        pOut->claimSize = pDot2 - (pDot1 + 1);

        pOut->pSignature = pDot2 + 1;
        pOut->signatureSize = (pJwtDocument + jwtDocumentSize) - (pDot2 + 1);

        pOut->pSignTarget = pJwtDocument;
        pOut->signTargetSize = pDot2 - pJwtDocument;

        NN_RESULT_SUCCESS;
    }

    Result ParseHeader(JwtHeaderData* pOut, const JwtElements& jwtElements) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOut);
        NN_SDK_ASSERT_NOT_NULL(jwtElements.pHeader);

        // BufferSize は必要なら外から注入できるように
        const size_t StringBufferSize = 32; // Header は文字数少ないから小さめでOK.
        const size_t DocumentBufferSize = 64; // スタック食わないように小さめ. Header 自体小さいしOK.
        json::Base64UrlSafeEncodedJsonInputStream<StringBufferSize, DocumentBufferSize>
            stream(jwtElements.pHeader, jwtElements.headerSize);
        json::RapidJsonEventAccepterForHeader eventHandler(pOut);
        NN_RESULT_DO( json::Parse(&stream, &eventHandler) );

        NN_RESULT_SUCCESS;
    };

    Result ParseClaim(
        json::JsonEventAccepter* rapidJsonEventHandlerForJwtClaims,
        const JwtElements& jwtElements) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(jwtElements.pClaim);

        // BufferSize は必要なら外から注入できるように
        const size_t StringBufferSize = 256; // JSON ドキュメント内の文字列の最大サイズにする必要がある.
        const size_t DocumentBufferSize = 256;
        json::Base64UrlSafeEncodedJsonInputStream<StringBufferSize, DocumentBufferSize>
            stream(jwtElements.pClaim, jwtElements.claimSize);

        // Claims Set パース実行
        NN_RESULT_DO( json::Parse(&stream, rapidJsonEventHandlerForJwtClaims) );

        NN_RESULT_SUCCESS;
    };

    Result DecodeSignatureBuffer(
        size_t* pOutActualSize, void* pOutBuffer, size_t outBufferSize,
        const JwtElements& jwtElements) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutActualSize);
        NN_SDK_ASSERT_NOT_NULL(pOutBuffer);
        NN_SDK_ASSERT_NOT_NULL(jwtElements.pSignature);

        auto ret = nn::util::Base64::FromBase64String(
            pOutActualSize, pOutBuffer, outBufferSize,
            jwtElements.pSignature,
            static_cast<int>(jwtElements.signatureSize),
            nn::util::Base64::Mode_UrlSafe);
        NN_RESULT_DO(json::HandleBase64Result(ret));

        NN_RESULT_SUCCESS;
    }
}}}} // nn::nsd::detail::jwt
