﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/npns.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/npns/npns_Api.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/npns/detail/npns_InternalTypes.h>
#include <nn/npns/detail/npns_HipcPorts.h>
#include <nn/npns/detail/npns_Utility.h>
#include <nn/os.h>
#include <mutex>

#include "npns_LibraryMain.h"
#include "npns_ServiceProxy.h"
#include "npns_Config.h"

#if NN_BUILD_CONFIG_OS_WIN
#include "npns_Instance.h"
#endif

namespace nn { namespace npns {

//----------------------------------------------------------------------------------
namespace
{
    int s_InitializeCount = 0;
    nn::os::Mutex s_InitializeCountMutex(false);
    PortIndex s_InitializedPort = PortIndex_Invalid;
}

Result InitializeImpl(PortIndex portIndex) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(s_InitializeCountMutex);
    Result result;

    if(s_InitializeCount == 0)
    {
#if NN_BUILD_CONFIG_OS_WIN
        g_Daemon.Initialize();
#endif
        // マクロによって、HIPC/DFC を切り替える
#if NN_NPNS_ENABLE_HIPC
        result = CreateHipcProxy(portIndex);
#else
        result = CreateDfcProxy(portIndex);
#endif
        if (result.IsFailure())
        {
            return result;
        }
        s_InitializedPort = portIndex;
    }

    ++s_InitializeCount;
    return ResultSuccess();
}

Result Initialize() NN_NOEXCEPT
{
    return InitializeImpl(PortIndex_User);
}

Result InitializeForSystem() NN_NOEXCEPT
{
    return InitializeImpl(PortIndex_System);
}

void FinalizeImpl () NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(s_InitializeCountMutex);

    NN_SDK_REQUIRES(s_InitializeCount > 0);

    if(--s_InitializeCount == 0)
    {
        DestroyServiceProxy(s_InitializedPort);
        s_InitializedPort = PortIndex_Invalid;
#if NN_BUILD_CONFIG_OS_WIN
        g_Daemon.Finalize();
#endif
    }
}

void Finalize() NN_NOEXCEPT
{
    FinalizeImpl();
}

void FinalizeForSystem() NN_NOEXCEPT
{
    FinalizeImpl();
}

bool IsInitialized() NN_NOEXCEPT
{
    return s_InitializeCount > 0;
}

Result ListenAll() NN_NOEXCEPT
{
    return GetServiceProxyUser().ListenAll();
}

Result ListenTo(ApplicationId toId) NN_NOEXCEPT
{
    return GetServiceProxyUser().ListenTo(toId);
}

Result ListenToMyApplicationId() NN_NOEXCEPT
{
    return GetServiceProxyUser().ListenToMyApplicationId(0);
}

Result ListenUndelivered() NN_NOEXCEPT
{
    return GetServiceProxySystem().ListenUndelivered();
}

Result Receive(NotificationData* pOut) NN_NOEXCEPT
{
    nn::sf::OutBuffer outBuffer(reinterpret_cast<char*>(pOut), sizeof(NotificationData));
    return GetServiceProxyUser().Receive(outBuffer, npns::Version);
}

Result GetReceiveEvent(nn::os::SystemEvent& event) NN_NOEXCEPT
{
    nn::sf::NativeHandle outHandle;
    Result result = GetServiceProxyUser().GetReceiveEvent(&outHandle);
    if (result.IsFailure())
    {
        return result;
    }

    event.AttachReadableHandle(outHandle.GetOsHandle(), outHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    outHandle.Detach();
    return ResultSuccess();
}

Result GetStateChangeEvent(nn::os::SystemEvent& event) NN_NOEXCEPT
{
    nn::sf::NativeHandle outHandle;
    Result result = GetServiceProxyUser().GetStateChangeEvent(&outHandle);
    if (result.IsFailure())
    {
        return result;
    }

    event.AttachReadableHandle(outHandle.GetOsHandle(), outHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    outHandle.Detach();
    return ResultSuccess();
}

Result SubscribeTopic(const char* pName) NN_NOEXCEPT
{
    nn::sf::InArray<char> inArray(pName, std::strlen(pName));
    Result result = GetServiceProxySystem().SubscribeTopic(inArray);
    return result;
}

Result UnsubscribeTopic(const char* pName) NN_NOEXCEPT
{
    nn::sf::InArray<char> inArray(pName, std::strlen(pName));
    Result result = GetServiceProxySystem().UnsubscribeTopic(inArray);
    return result;
}

Result QueryIsTopicExist(bool* pResult, const char* pName) NN_NOEXCEPT
{
    nn::sf::InArray<char> inArray(pName, std::strlen(pName));
    Result result = GetServiceProxySystem().QueryIsTopicExist(pResult, inArray);
    return result;
}

Result CreateToken(NotificationToken* pTokenOut, const nn::account::Uid& uid, ApplicationId applicationId) NN_NOEXCEPT
{
    nn::sf::Out<NotificationToken> out(pTokenOut);
    Result result = GetServiceProxySystem().CreateTokenWithApplicationId(out, uid, applicationId);
    return result;
}

Result CreateToken(NotificationToken* pTokenOut, const nn::account::Uid& uid) NN_NOEXCEPT
{
    nn::sf::Out<NotificationToken> out(pTokenOut);
    Result result = GetServiceProxyUser().CreateToken(out, uid, 0);
    return result;
}

Result DestroyToken(const nn::account::Uid& uid, ApplicationId applicationId) NN_NOEXCEPT
{
    Result result = GetServiceProxySystem().DestroyTokenWithApplicationId(uid, applicationId);
    return result;
}

Result DestroyToken(const nn::account::Uid& uid) NN_NOEXCEPT
{
    Result result = GetServiceProxyUser().DestroyToken(uid, 0);
    return result;
}

Result QueryIsTokenValid(bool*, const NotificationToken&) NN_NOEXCEPT
{
    return ResultNotImplemented();
}

Result Suspend() NN_NOEXCEPT
{
    Result result = GetServiceProxyUser().Suspend();
    return result;
}

Result Resume() NN_NOEXCEPT
{
    Result result = GetServiceProxyUser().Resume();
    return result;
}

nn::npns::State GetState() NN_NOEXCEPT
{
    int32_t state = State_Initial;
    sf::Out<int32_t> outState(&state);
    GetServiceProxyUser().GetState(outState);
    return static_cast<State>(outState.Get());
}

Result CreateJid() NN_NOEXCEPT
{
    Result result = GetServiceProxySystem().CreateJid();
    return result;
}

Result DestroyJid() NN_NOEXCEPT
{
    Result result = GetServiceProxySystem().DestroyJid();
    return result;
}

Result AttachJid(const char* pJid, const char* pPassword) NN_NOEXCEPT
{
    nn::sf::InArray<char> jid(pJid, std::strlen(pJid));
    nn::sf::InArray<char> password(pPassword, std::strlen(pPassword));
    Result result = GetServiceProxySystem().AttachJid(jid, password);
    return result;
}

Result DetachJid(char* pJidOut, size_t jidLength, char* pPasswordOut, size_t passwordLength) NN_NOEXCEPT
{
    nn::sf::OutArray<char> outArrayJid(pJidOut, jidLength - 1);
    nn::sf::OutArray<char> outArrayPassword(pPasswordOut, passwordLength - 1);
    std::memset(pJidOut, 0, jidLength);
    std::memset(pPasswordOut, 0, passwordLength);
    Result result = GetServiceProxySystem().DetachJid(outArrayJid, outArrayPassword);
    return result;
}

nn::Result GetStatistics(nn::npns::Statistics* pOutStatistics) NN_NOEXCEPT
{
    nn::sf::OutBuffer outBuffer(reinterpret_cast<char*>(pOutStatistics), sizeof(*pOutStatistics));
    Result result = GetServiceProxyUser().GetStatistics(outBuffer);
    return result;
}

nn::Result BindPlayReportRequestEvent(nn::os::SystemEventType * pEventType, nn::os::EventClearMode clearMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEventType);

    nn::sf::NativeHandle outHandle;
    Result result = GetServiceProxySystem().GetPlayReportRequestEvent(&outHandle);
    if (result.IsFailure())
    {
        return result;
    }

    nn::os::AttachReadableHandleToSystemEvent(pEventType, outHandle.GetOsHandle(), outHandle.IsManaged(), clearMode);
    outHandle.Detach();

    return ResultSuccess();
}

Result GetJid(char* pJidOut, size_t length) NN_NOEXCEPT
{
    nn::sf::OutArray<char> outArray(pJidOut, length - 1);
    std::memset(pJidOut, 0, length);
    Result result = GetServiceProxyUser().GetJid(outArray);
    return result;
}

Result UploadTokenToBaaS(const nn::account::Uid& uid) NN_NOEXCEPT
{
    Result result = GetServiceProxySystem().UploadTokenToBaaS(uid);
    return result;
}

nn::Result DestroyTokenForBaaS(const nn::account::Uid& uid) NN_NOEXCEPT
{
    Result result = GetServiceProxySystem().DestroyTokenForBaaS(uid);
    return result;
}

// http://spdlybra.nintendo.co.jp/jira/browse/IAAD-1594
// http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-82627
//
// 以下を一括して行う
//  - JID 再発行
//  - BaaS 用 NT の再登録(アカウントごと)
nn::Result RefreshResources(const nn::account::Uid uids[], int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(count, 0, nn::account::UserCountMax);

    auto& proxy = GetServiceProxySystem();

    // 自律動作を抑止
    NN_ABORT_UNLESS_RESULT_SUCCESS(proxy.Suspend());
    NN_UTIL_SCOPE_EXIT { proxy.Resume(); };

    Result result = proxy.CreateJid(); // 既に JID を保持している場合は再発行となる
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    for(int i = 0 ; i < count ; i++)
    {
        // uids は DA を持つユーザーのIDであることを想定
        result = proxy.UploadTokenToBaaS(uids[i]);
        NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
    }

    return result;
}

// for test only.
nn::Result RequestChangeStateForceTimed(nn::npns::State targetState, nn::TimeSpan timeout) NN_NOEXCEPT
{
    Result result = GetServiceProxySystem().RequestChangeStateForceTimed(static_cast<int32_t>(targetState), timeout);
    return result;
}

// for test only.
nn::Result RequestChangeStateForceAsync(nn::npns::State targetState) NN_NOEXCEPT
{
    Result result = GetServiceProxySystem().RequestChangeStateForceAsync(static_cast<int32_t>(targetState));
    return result;
}

//----------------------------------------------------------------------------------

}}  // namespace nn::npns

