﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/nim/srv/nim_NetworkInstallUrl.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include "nim_Json.h"
#include "nim_SystemUpdateMetaUtil.h"

namespace nn { namespace nim { namespace srv {
    namespace {
        Result ParseSystemUpdateMetaForContentDeliveryResponse(ncm::ContentMetaKey* outValue, char* readBuffer) NN_NOEXCEPT
        {
            nne::rapidjson::Document document;
            NN_RESULT_THROW_UNLESS(!document.ParseInsitu(readBuffer).HasParseError(), ResultUnexpectedResponseSystemUpdateMetaForContentDeliveryParseError());

            auto titleIdObj = document.FindMember("contents_delivery_required_title_id");
            auto versionObj = document.FindMember("contents_delivery_required_title_version");
            NN_RESULT_THROW_UNLESS(titleIdObj != document.MemberEnd() && titleIdObj->value.IsString(), ResultUnexpectedResponseSystemUpdateMetaForContentDeliveryTitleIdNotFound());
            NN_RESULT_THROW_UNLESS(versionObj != document.MemberEnd() && versionObj->value.IsInt(), ResultUnexpectedResponseSystemUpdateMetaForContentDeliveryVersionNotFound());

            Bit64 contentMetaId = std::strtoull(titleIdObj->value.GetString(), nullptr, 16);
            uint32_t version = static_cast<uint32_t>(versionObj->value.GetInt());

            *outValue = ncm::ContentMetaKey::Make(contentMetaId, version, ncm::ContentMetaType::SystemUpdate);
            NN_RESULT_SUCCESS;
        }

        util::optional<ncm::ContentMetaKey> GetDebugSystemUpdateMetaForContentDelivery() NN_NOEXCEPT
        {
#ifdef NN_BUILD_CONFIG_OS_WIN
            return util::nullopt;
#else
            char debugIdString[64] = {};
            settings::fwdbg::GetSettingsItemValue(debugIdString, sizeof(debugIdString), "systemupdate", "debug_id_for_content_delivery");
            debugIdString[63] = '\0';

            auto debugId = std::strtoull(debugIdString, nullptr, 16);
            if (debugId == 0)
            {
                return util::nullopt;
            }

            uint32_t debugVersion;
            settings::fwdbg::GetSettingsItemValue(&debugVersion, sizeof(debugVersion), "systemupdate", "debug_version_for_content_delivery");

            return ncm::ContentMetaKey::Make(debugId, debugVersion, ncm::ContentMetaType::SystemUpdate);
#endif
        }
    }

    Result CheckSystemUpdateMetaForContentDelivery(HttpConnection* connection, Bit64 deviceId) NN_NOEXCEPT
    {
        Url url;
        MakeSystemUpdateMetaForContentDeliveryUrl(&url, deviceId);

        const char* header("Accept:application/json");

        char readBuffer[1024];
        size_t readSize = 0;
        NN_RESULT_DO(connection->Get(url,
            [&readSize, &readBuffer](const void* buffer, size_t size) -> Result
            {
                NN_RESULT_THROW_UNLESS(readSize + size < sizeof(readBuffer), ResultUnexpectedResponseSystemUpdateMetaForContentDeliveryTooLong());
                std::memcpy(&readBuffer[readSize], buffer, size);

                readSize += size;

                NN_RESULT_SUCCESS;
            },
            &header, 1));
        readBuffer[readSize] = '\0';
        NN_DETAIL_NIM_TRACE("[GetSystemUpdateMetaForContentDelivery] Response %s\n", readBuffer);

        ncm::ContentMetaKey latest;
        auto debugSystemUpdateMeta = GetDebugSystemUpdateMetaForContentDelivery();
        if (debugSystemUpdateMeta)
        {
            latest = *debugSystemUpdateMeta;
            NN_DETAIL_NIM_TRACE("[CheckSystemUpdateMetaForContentDelivery] Debug system update meta 0x%016llx version %u\n", latest.id, latest.version);
        }
        else
        {
            NN_RESULT_DO(ParseSystemUpdateMetaForContentDeliveryResponse(&latest, readBuffer));
        }

        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::BuiltInSystem));

        ncm::ContentMetaKey current;
        NN_RESULT_TRY(db.GetLatest(&current, latest.id))
            NN_RESULT_CATCH_CONVERT(ncm::ResultContentMetaNotFound, ResultSystemUpdateRequiredForContentDelivery())
        NN_RESULT_END_TRY
        NN_RESULT_THROW_UNLESS(current.type == ncm::ContentMetaType::SystemUpdate && current.version >= latest.version, ResultSystemUpdateRequiredForContentDelivery());

        NN_RESULT_SUCCESS;
    }
}}}
