﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "nim_ShopServiceAccessServerImpl.h"

namespace nn { namespace nim { namespace srv {

struct ShopServiceAccessServiceServer
{
    /**
     * @brief       各種共通定数。
     */
    struct Requirement
    {
        /**
         * @brief   トークン認証用リクエストヘッダプレフィクス最大長。( null 終端含まない )
         */
        static const int RequestHeaderPrefixLengthMaxForToken = 48;

        /**
         * @brief   エラーコード( エラーなし )
         */
        static const ::nn::err::ErrorCode ErrorCodeNone;
    };

    /**
     * @brief       サーバーエンドポイントベースURL情報構造体。
     */
    struct BaseUrl
    {
        const char* const pData;
        const Bit32 length;

        /**
         * @brief   コンストラクタ。
         */
        BaseUrl(const char* const pData_, const Bit32 length_) NN_NOEXCEPT : pData(pData_), length(length_) {}
    };

    /**
     * @brief   接続先サーバー固有実装基底
     */
    class Accessor
    {
    public:
        typedef ShopServiceAccess::RequestCountRestriction::Canceler    RestrictionCanceler;
        typedef TokenStore::DeviceAuth                                  TokenStoreDevice;
        typedef TokenStore::NsaId                                       TokenStoreNsa;
        typedef TokenStore::NaId                                        TokenStoreNa;
        typedef TokenStore::CancelerSet                                 CancelerSet;

        /**
         * @brief       ターゲットサーバーのベースURL文字列を取得します。
         *
         * @param[out]  pOutValue   取得先バッファ。充分な容量を確保してください。
         *
         * @return      ベースURL文字列の文字列長。( 終端 null を含みません。 )
         */
        const size_t QueryTargetBaseUrl(char* pOutValue) const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutValue);
            std::strcpy(pOutValue, m_BaseUrl.pData);
            return static_cast<size_t>(m_BaseUrl.length);
        }

        /**
         * @brief   ターゲットサーバーのベースURL文字列長を取得します。
         *
         * @return  ベースURL文字列の文字列長。(終端 null を含みません。)
         */
        NN_FORCEINLINE const size_t GetTargetBaseUrlLength() const NN_NOEXCEPT
        {
            return static_cast<size_t>(m_BaseUrl.length);
        }

        /**
         * @brief  リクエストヘッダ NA-ID-TOKEN 上限長。
         */
        static const size_t LengthMaxForNaIdToken = static_cast<size_t>(Requirement::RequestHeaderPrefixLengthMaxForToken + TokenStoreNa::LengthMax);

        /**
         * @brief  リクエストヘッダ NSA-ID-TOKEN 上限長。
         */
        static const size_t LengthMaxForNsaIdToken = static_cast<size_t>(Requirement::RequestHeaderPrefixLengthMaxForToken + TokenStoreNsa::LengthMax);

        /**
         * @brief  リクエストヘッダ デバイス認証トークン上限長。
         */
        static const size_t LengthMaxForDeviceToken = static_cast<size_t>(Requirement::RequestHeaderPrefixLengthMaxForToken + TokenStoreDevice::LengthMax);

        /**
         * @brief  リクエストヘッダ アプリケーション詐称検証用呼び出し元アプリケーションID 上限長。
         */
        static const size_t LengthMaxForApplicationIdVerify = static_cast<size_t>(Requirement::RequestHeaderPrefixLengthMaxForToken + ShopServiceAccess::Requirement::LengthMaxForStringApplicationId + 1);

        /**
         * @brief       リクエストヘッダ生成及び登録処理用コンテキストハンドル構造体。
         */
        struct RequestHeaderGenerationHandle
        {
            HttpConnection*                     pConnection;    //!< 通信要求コンテキストハンドル。
            HttpConnection::TransactionHandle*  pTransaction;   //!< 通信要求コンテキストハンドル。
            char*                               pTemporary;     //!< 生成用テンポラリ領域。
            size_t                              temporarySize;  //!< 生成用テンポラリ領域容量。( バイト単位 )
            ::nn::ncm::ApplicationId            applicationId;  //!< 呼び出し元アプリケーションID。
        };

        /**
         * @brief       リクエストヘッダ生成及び登録。
         *
         * @param[in/out]   pHandle         生成/登録処理用コンテキストハンドル。
         * @param[in]       uid             NSA-IDトークン取得対象ユーザアカウントID。
         * @param[out]      pCanceler       外部からキャンセルする際のハンドルを割り当てるキャンセラコンテナへの参照。
         *
         * @return      処理結果が返ります。
         * @retval      nn::ResultSuccess                   成功しました。
         * @retval      nn::nim::ResultBufferNotEnough      生成用テンポラリ領域が不足しています。
         *
         * @pre
         *              - pHandle != nullptr
         *              - pCanceler != nullptr
         *              - handle.pTemporary != nullptr
         *              - handle.pConnection != nullptr
         *              - handle.pTransaction != nullptr
         *              - handle.temporarySize >= LengthMaxForNaIdToken
         *              - handle.temporarySize >= LengthMaxForNsaIdToken
         *              - handle.temporarySize >= LengthMaxForDeviceToken
         */
        virtual Result RegisterRequestHeaders(RequestHeaderGenerationHandle* pHandle, const ::nn::account::Uid& uid, CancelerSet* pCanceler) const NN_NOEXCEPT = 0;

        /**
         * @brief       既存の Result がサーバーレスポンス中エラーコードを持つ場合にエラーコードを取得します。
         *
         * @param[out]  pOutErrorCode       サーバーエラーレスポンスから検出されたエラーコード値。
         * @param[in]   nowResult           現在発生している Result 値。
         * @param[in]   pReceivedResponse   サーバーレスポンスストリーム。
         * @param[in]   responseSize        サーバーレスポンスストリーム容量。
         *
         * @details     補正対象外の Result であった場合は何もしません。
         */
        virtual Result ResolveServerErrorResult(::nn::err::ErrorCode* pOutErrorCode, const Result& nowResult, const char* pReceivedResponse, size_t responseSize) const NN_NOEXCEPT = 0;

        /**
         * @brief       制限許可申請。
         *
         * @param[out]  pCanceler   ロック中キャンセルイベントコンテナ。@n
         *                          制限オーバーに伴ってストールする場合、ストール期間中のみキャンセル用のイベントを格納します。
         *
         * @return      許可申請結果を返します。
         * @retval      ResultSuccess                   許可申請が受理されました。
         * @retval      ResultShopServiceAccessCanceled 申請者によって申請がキャンセルされました。
         *
         * @details     制限オーバー時は時間経過まで呼び出し元スレッドをストールします。@n
         *              ストール中に他スレッドから呼び出した場合は、ストール解除まで待機させられます。@n
         *              キャンセルされた場合は時間制限中であってもメソッドから復帰しますが、制限対象処理を実施しない事を想定しています。
         *
         * @pre
         *              - pCanceler != nullptr
         */
        virtual Result AcquireRestrictPermission(RestrictionCanceler* pCanceler) NN_NOEXCEPT = 0;

        /**
         * @brief       サーバー情報取得制限対象ユーザかどうかの問い合わせ。
         *
         * @param[in]   uid     検証対象ローカルユーザID。
         *
         * @return      制限確認結果を返します。
         * @retval      ResultSuccess                                       制限対象のユーザではない、もしくはサーバー側の制限はありません。
         * @retval      nn::nim::ResultAgeRestriction                       年齢制限対象です。
         * @retval      nn::nim::ResultNetworkTimeUnavailable               ネットワーク時計が取れませんでした。
         * @retval      nn::account::ResultNetworkServiceAccountUnavailable 指定されたユーザアカウントのネットワークサービスアカウントが利用できません。
         */
        virtual Result QueryRestrictUserAvailability(const ::nn::account::Uid& uid) NN_NOEXCEPT;

    protected:
        /**
         * @brief       コンストラクタ
         */
        Accessor(const char* pBaseUrl, Bit32 length) NN_NOEXCEPT : m_BaseUrl(pBaseUrl, length) {}

        /**
         * @brief       NSA-ID-TOKEN リクエストヘッダ文字列生成。
         *
         * @param[in]   pTemporary      生成用テンポラリ領域。
         * @param[in]   temporarySize   生成用テンポラリ領域容量。( バイト単位 )
         * @param[in]   uid             トークン取得対象ユーザアカウントID。
         * @param[out]  pCanceler       外部からキャンセルする際のハンドルを割り当てるキャンセラコンテナへの参照。
         *
         * @return      処理結果が返ります。
         * @retval      nn::ResultSuccess                   成功しました。
         * @retval      nn::nim::ResultBufferNotEnough      生成用テンポラリ領域が不足しています。
         *
         * @pre
         *      - pCanceler != nullptr
         *      - pTemporary != nullptr
         *      - temporarySize >= LengthMaxForNsaIdToken
         */
        Result CreateHeaderForNsaIdToken(char* pTemporary, size_t temporarySize, const ::nn::account::Uid& uid, TokenStoreNsa::ICanceler* pCanceler) const NN_NOEXCEPT;


        /**
         * @brief       NA-ID-TOKEN リクエストヘッダ文字列生成。
         *
         * @param[in]   pTemporary      生成用テンポラリ領域。
         * @param[in]   temporarySize   生成用テンポラリ領域容量。( バイト単位 )
         * @param[in]   uid             トークン取得対象ユーザアカウントID。
         * @param[out]  pCanceler       外部からキャンセルする際のハンドルを割り当てるキャンセラコンテナへの参照。
         *
         * @return      処理結果が返ります。
         * @retval      nn::ResultSuccess                   成功しました。
         * @retval      nn::nim::ResultBufferNotEnough      生成用テンポラリ領域が不足しています。
         *
         * @pre
         *      - pCanceler != nullptr
         *      - pTemporary != nullptr
         *      - temporarySize >= LengthMaxForNaIdToken
         */
        Result CreateHeaderForNaIdToken(char* pTemporary, size_t temporarySize, const ::nn::account::Uid& uid, TokenStoreNa::ICanceler* pCanceler) const NN_NOEXCEPT;

        /**
         * @brief       デバイス認証トークンリクエストヘッダ文字列生成。
         *
         * @param[in]   pTemporary      生成用テンポラリ領域。
         * @param[in]   temporarySize   生成用テンポラリ領域容量。( バイト単位 )
         * @param[out]  pCanceler       外部からキャンセルする際のハンドルを割り当てるキャンセラコンテナへの参照。
         *
         * @return      処理結果が返ります。
         * @retval      nn::ResultSuccess                   成功しました。
         * @retval      nn::nim::ResultBufferNotEnough      生成用テンポラリ領域が不足しています。
         *
         * @pre
         *      - pCanceler != nullptr
         *      - pTemporary != nullptr
         *      - temporarySize >= LengthMaxForDeviceToken
         */
        Result CreateHeaderForDeviceAuthenticationToken(char* pTemporary, size_t temporarySize, TokenStoreDevice::ICanceler* pCanceler) const NN_NOEXCEPT;

        /**
         * @brief       アプリケーション詐称検証用呼び出し元アプリケーションIDリクエストヘッダ文字列生成。
         *
         * @param[in]   pTemporary      生成用テンポラリ領域。
         * @param[in]   temporarySize   生成用テンポラリ領域容量。( バイト単位 )
         * @param[in]   uid             検証対象アプリケーションID。
         *
         * @return      処理結果が返ります。
         * @retval      nn::ResultSuccess                   成功しました。
         * @retval      nn::nim::ResultBufferNotEnough      生成用テンポラリ領域が不足しています。
         *
         * @pre
         *      - pCanceler != nullptr
         *      - pTemporary != nullptr
         *      - temporarySize >= LengthMaxForApplicationIdVerify
         */
        Result CreateHeaderForApplicationIdVerification(char* pTemporary, size_t temporarySize, const ::nn::ncm::ApplicationId& applicationId) const NN_NOEXCEPT;

    private:
        BaseUrl m_BaseUrl;
    };

    /**
     * @brief   接続先サーバー( Sugar )固有実装
     */
    class AccessorForSugar : public Accessor
    {
    private:
        /**
         * @brief  サーバー用エンドポイントベースURL( サービスディスカバリ経由 )。
         */
        static const char ServiceDiscoveryBaseUrl[];

        /**
         * @brief  サーバー固有エラーコードモジュールID。
         */
        static const Bit32 ErrorCodeModuleId = 2309;

        /**
         * @brief   １分間の要求制限数。
         */
        static const int CountOfRestrictInOneMinutes = 10;

        /**
         * @brief   Sugar クライアントREST APIレスポンスJson文字列、想定構成定義。
         *
         * @detail  想定パス及び深さ。
         *          想定最長パス  : { "rights": [ { "consumption_request_id": "" } ] }
         *          パス            : 35 -> "$.rights[*].consumption_request_id"。
         *          Depth           :  5。
         * @note    http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=277358966
         */
        typedef ::nn::http::json::JsonPath<8, 64> JsonPathType;

    public:
        AccessorForSugar() NN_NOEXCEPT;

        virtual Result RegisterRequestHeaders(RequestHeaderGenerationHandle* pHandle, const ::nn::account::Uid& uid, CancelerSet* pCanceler) const NN_NOEXCEPT NN_OVERRIDE;

        virtual Result ResolveServerErrorResult(::nn::err::ErrorCode* pOutErrorCode, const Result& nowResult, const char* pReceivedResponse, size_t responseSize) const NN_NOEXCEPT NN_OVERRIDE;

        virtual Result AcquireRestrictPermission(RestrictionCanceler* pCanceler) NN_NOEXCEPT NN_OVERRIDE;

    private:
        ShopServiceAccess::RequestCountRestriction  m_Restriction;
    };

    /**
     * @brief   接続先サーバー( Civil )固有実装
     */
    class AccessorForCivil : public Accessor
    {
    private:
        /**
         * @brief  サーバー用エンドポイントベースURL( サービスディスカバリ経由 )。
         */
        static const char ServiceDiscoveryBaseUrl[];

        /**
         * @brief  サーバー固有エラーコードモジュールID。
         */
        static const Bit32 ErrorCodeModuleId = 2308;

        /**
         * @brief   １分間の要求制限数。
         */
        static const int CountOfRestrictInOneMinutes = 30;

        /**
         * @brief   Sugar クライアントREST APIレスポンスJson文字列、想定構成定義。
         *
         * @detail  想定パス及び深さ。
         *          想定最長パス  : { "rights": [ { "consumption_request_id": "" } ] }
         *          パス            : 35 -> "$.rights[*].consumption_request_id"。
         *          Depth           :  5。
         * @note    エラーレスポンス書式は Sugar と同じ。
         *          http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=277358966
         *          http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=288002651
         *          http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=290948912
         */
        typedef ::nn::http::json::JsonPath<8, 64> JsonPathType;

    public:
        AccessorForCivil() NN_NOEXCEPT;

        virtual Result RegisterRequestHeaders(RequestHeaderGenerationHandle* pHandle, const ::nn::account::Uid& uid, CancelerSet* pCanceler) const NN_NOEXCEPT NN_OVERRIDE;

        virtual Result ResolveServerErrorResult(::nn::err::ErrorCode* pOutErrorCode, const Result& nowResult, const char* pReceivedResponse, size_t responseSize) const NN_NOEXCEPT NN_OVERRIDE;

        virtual Result AcquireRestrictPermission(RestrictionCanceler* pCanceler) NN_NOEXCEPT NN_OVERRIDE;

        virtual Result QueryRestrictUserAvailability(const ::nn::account::Uid& uid) NN_NOEXCEPT NN_OVERRIDE;

    private:
        ShopServiceAccess::RequestCountRestriction  m_Restriction;
    };

    /**
     * @brief       接続先サーバー固有実装オブジェクトの取得。
     *
     * @param[in]   server      ターゲットサーバー指定。
     */
    static Accessor* GetPresetAccessor(const ShopServiceAccessTypes::Server server) NN_NOEXCEPT;
};

}}}
