﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "nim_ShopServiceAccessAsyncImpl.h"
#include "nim_RequestQueueScheduler.h"
#include <nn/util/util_BitArray.h>

namespace nn { namespace nim { namespace srv {

class ShopServiceAccessServerImpl;

/**
 * @brief   アクセッサ実装定義。
 */
class ShopServiceAccessorImpl : public ::nn::sf::ISharedObject
{
    NN_DISALLOW_COPY(ShopServiceAccessorImpl);
    NN_DISALLOW_MOVE(ShopServiceAccessorImpl);

public:
    /**
     * @brief   通信ハンドル実装型
     */
    typedef ShopServiceAccessAsyncImpl::ConnectionImpl ConnectionHandleImpl;

    /**
     * @brief   通信ハンドル型
     */
    typedef ConnectionHandleImpl* ConnectionHandle;

    /**
     * @brief   コンストラクタ
     */
    NN_IMPLICIT ShopServiceAccessorImpl(ShopServiceAccessServerImpl* pServer, const ::nn::nim::ShopServiceAccessTypes::Server& inServer) NN_NOEXCEPT;

    /**
     * @brief   デストラクタ
     */
    ~ShopServiceAccessorImpl() NN_NOEXCEPT;

    /**
     * @brief       IPC: 非同期要求ハンドルサービスオブジェクトの生成要求。
     */
    Result CreateAsyncInterface(::nn::sf::Out<::nn::sf::NativeHandle> outEventHandle, ::nn::sf::Out<::nn::sf::SharedPointer<::nn::nim::detail::IShopServiceAsync>> outAsync, const ::nn::nim::ShopServiceAccessTypes::FixedParams& inFixedParams) NN_NOEXCEPT;

    /**
     * @brief       通信実施要求。
     *
     * @param[in]   pRequest        要求対象非同期要求ハンドルインスタンスへのポインタ。
     */
    Result Request(ShopServiceAccessAsyncImpl* pRequest) NN_NOEXCEPT;

    /**
     * @brief       通信ハンドルの獲得。
     *
     * @param[out]  pOutConnection  通信ハンドル受領先メモリ領域のアドレス先頭へのポインタ。@n
     *                              獲得に成功したハンドルは通信処理実施可能である事が保証されます。
     *
     * @return      獲得結果が返ります。
     * @retval      nn::ResultSuccess                       正常に終了しました。
     * @retval      nn::nim::ResultCurl*                    Curlモジュールの初期設定シーケンスに失敗しました。
     * @retval      nn::nim::ResultCurlEasyInitFailure      通信用 Curlモジュール初期化に失敗しました。
     * @retval      nn::nim::ResultAllocationMemoryFailed   獲得可能な通信モジュールリソースが見つかりませんでした。
     */
    Result AcquireConnection(ConnectionHandle* pOutConnection) NN_NOEXCEPT;

    /**
     * @brief       通信ハンドルの返却。
     *
     * @param[in]   pConnection     使用済通信ハンドル。
     */
    void ReleaseConnection(ConnectionHandle pConnection) NN_NOEXCEPT;

    /**
     * @brief       アクセス要求本アプリケーションIDの取得。
     */
    ::nn::ncm::ApplicationId GetOwnerApplicationId() const NN_NOEXCEPT;

    /**
     * @brief       アクセス対象サーバー情報の取得。
     */
    NN_FORCEINLINE const ShopServiceAccessTypes::Server& GetTargetServer() const NN_NOEXCEPT
    {
        return m_TargetServer;
    }

private:
    /**
     * @brief   依存親型宣言。
     */
    typedef sf::SharedPointer<ShopServiceAccessServerImpl>  ParentInterface;

    bool FinalizeConnection() NN_NOEXCEPT;

    const ParentInterface               m_pServer;          //!< サーバーオーナー( 親寿命管理用共有ポインタ )
    ConnectionHandleImpl                m_Connection[ShopServiceAccessServer::Configuration::ParallelExecutionCount];
    mutable ::nn::os::SdkRecursiveMutex m_ConnectMapLock;
    ::nn::util::BitArray                m_ConnectMapArray;
    Bit64                               m_ConnectMapValue;
    ShopServiceAccessTypes::Server      m_TargetServer;
};

/**
 * @brief   アクセッササーバー用実装。
 */
class ShopServiceAccessServerImpl : public ::nn::sf::ISharedObject
{
    NN_DISALLOW_COPY(ShopServiceAccessServerImpl);
    NN_DISALLOW_MOVE(ShopServiceAccessServerImpl);

    /**
     * @brief   型エイリアス
     */
    typedef ShopServiceAccessServer::Configuration  Configuration;
    typedef ShopServiceAccess::TransferStorage      TransferStorage;

    /**
     * @brief   リクエストスケジューラ基底
     */
    typedef RequestQueueScheduler<ShopServiceAccessAsyncImpl, Configuration::EnsureSchedulerThreadStackSize, Configuration::RequestQueueCapacity> RequestSchedulerBase;

    /**
     * @brief   リクエストスケジューラ
     */
    class RequestScheduler : public RequestSchedulerBase
    {
        NN_DISALLOW_COPY(RequestScheduler);
        NN_DISALLOW_MOVE(RequestScheduler);

    public:
        typedef RequestSchedulerBase BaseType;

        static const int ThreadCount = Configuration::ParallelExecutionCount;
        static const size_t EnsureExpectStackSize = (Configuration::EnsureWorkerThreadStackSize * ThreadCount) + EnsureExpectSchedulerStackSize;

        explicit RequestScheduler(char* pStackTop) NN_NOEXCEPT
            : BaseType(ThreadCount, m_Handles, pStackTop, Configuration::EnsureWorkerThreadStackSize)
        {
        }

    protected:
        virtual void RunOnThread(ShopServiceAccessAsyncImpl* const pRequest) NN_NOEXCEPT NN_OVERRIDE;

        virtual void FinishOnThread(ShopServiceAccessAsyncImpl* const pRequest) NN_NOEXCEPT NN_OVERRIDE;

        virtual void OnCancelRequestCalled(ShopServiceAccessAsyncImpl* const pRequest) NN_NOEXCEPT NN_OVERRIDE;

        virtual void OnWorkerThreadDestroyed(int index) NN_NOEXCEPT NN_OVERRIDE;

        virtual void OnObserverFinished() NN_NOEXCEPT NN_OVERRIDE;

    private:
        BaseType::ThreadHandleImpl  m_Handles[ThreadCount];
    };

public:
    /**
     * @brief   リクエストスケジューラスタック容量要件
     */
    static const size_t EnsureExpectStackSize = RequestScheduler::EnsureExpectStackSize;

    /**
     * @brief   アクセス要求本アプリケーションIDの取得。
     */
    NN_FORCEINLINE::nn::ncm::ApplicationId GetOwnerApplicationId() const NN_NOEXCEPT
    {
        return m_OwnerApplicationId;
    }

    /**
     * @brief   コンストラクタ。
     */
    explicit ShopServiceAccessServerImpl(char* pStackTop) NN_NOEXCEPT;

    /**
     * @brief   デストラクタ(セッション切断時の呼び出し)。
     */
    ~ShopServiceAccessServerImpl() NN_NOEXCEPT;

    /**
     * @brief   サーバー起動。
     */
    Result Initialize(const ::nn::os::ProcessId& clientProcessId, const ::nn::sf::NativeHandle& inTransferHandle, uint64_t inTransferTotalSize) NN_NOEXCEPT;

    /**
     * @brief   非同期要求ハンドル準備。
     */
    Result Prepare(ShopServiceAccessAsyncImpl* pRequest, const ::nn::nim::ShopServiceAccessTypes::FixedParams& inFixedParams) NN_NOEXCEPT;

    /**
     * @brief   通信実施要求( キュー登録要求 )。
     */
    Result Request(ShopServiceAccessAsyncImpl* pRequest) NN_NOEXCEPT;

    /**
     * @brief   実行要求強制終了。
     */
    void Finish() NN_NOEXCEPT;

    /**
     * @brief   IPC: ShopServiceAccessorImpl サービスオブジェクト生成要求。
     */
    Result CreateAccessorInterface(::nn::sf::Out<::nn::sf::SharedPointer<::nn::nim::detail::IShopServiceAccessor>> outValue, const ::nn::nim::ShopServiceAccessTypes::Server& inServer) NN_NOEXCEPT;

    /**
     * @brief       デバイス認証トークン取得要求。
     *
     * @param[out]  pCanceler           外部からキャンセルする際のハンドルを割り当てるキャンセラコンテナへの参照。
     * @param[in]   clientId            デバイス認証トークン用クライアントID。
     * @param[out]  pOutToken           トークンをコピーする領域の先頭アドレスへのポインタ。@n
     *                                  最小容量要件として @ref DeviceAuthenticationTokenCapacity を満たしてください。
     * @param[in]   availableCapacity   @a pOutToken が示すメモリ領域の書き込み可能な容量。( バイトサイズ )@n
     *                                  トークンサイズ最大は @ref nn::nim::srv::TokenStore::DeviceAuth::LengthMax。
     * @param[out]  pOutTokenEnd        トークンをコピーした領域の終端文字を示す末尾アドレスを格納する領域の先頭アドレスへのポインタ。@n
     *                                  nullptr 指定の場合、出力されません。
     *
     *              - pCanceler != nullptr
     *              - pOutToken != nullptr
     *              - availableCapacity >= nn::nim::srv::TokenStore::DeviceAuth::LengthMax
     */
    static Result AcquireTokenForDeviceAuthentication(TokenStore::DeviceAuth::ICanceler* pCanceler, const uint64_t clientId, char* pOutToken, const size_t availableCapacity, char** pOutTokenEnd = nullptr) NN_NOEXCEPT;

    /**
     * @brief       Nintendo Account ID トークン取得要求。
     *
     * @param[out]  pOutToken           出力先メモリ領域先頭アドレスへのポインタ。
     * @param[in]   availableCapacity   @a pOutToken が示すメモリ領域の書き込み可能な容量。( バイトサイズ )@n
     *                                  トークンサイズ最大は @ref nn::nim::srv::TokenStore::NaId::LengthMax。
     * @param[in]   uid                 Accountライブラリから取得する NA-ID-TOKEN の対象ユーザアカウントID。
     * @param[out]  pCanceler           外部からキャンセルする際のハンドルを割り当てるキャンセラコンテナへの参照。
     *
     * @pre
     *              - pCanceler != nullptr
     *              - pOutToken != nullptr
     *              - availableCapacity >= nn::nim::srv::TokenStore::NaId::LengthMax
     *              - uid が示すユーザが存在する。
     */
    static Result AcquireTokenForNaId(char* pOutToken, const size_t availableCapacity, const ::nn::account::Uid& uid, TokenStore::NaId::ICanceler* pCanceler) NN_NOEXCEPT;

    /**
     * @brief       Network Service Account IDトークン取得要求。
     *
     * @param[out]  pOutToken           出力先メモリ領域先頭アドレスへのポインタ。
     * @param[in]   availableCapacity   @a pOutToken が示すメモリ領域の書き込み可能な容量。( バイトサイズ )@n
     *                                  トークンサイズ最大は @ref nn::nim::srv::TokenStore::NsaId::LengthMax。
     * @param[in]   uid                 Accountライブラリから取得する NSA-ID-TOKEN の対象ユーザアカウントID。
     * @param[in]   info                トークンIDに含めるシステムプログラムの識別情報。
     * @param[out]  pCanceler           外部からキャンセルする際のハンドルを割り当てるキャンセラコンテナへの参照。
     *
     * @pre
     *              - pCanceler != nullptr
     *              - pOutToken != nullptr
     *              - availableCapacity >= nn::nim::srv::TokenStore::NsaId::LengthMax
     *              - uid が示すユーザが存在する。
     */
    static Result AcquireTokenForNsaId(char* pOutToken, const size_t availableCapacity, const ::nn::account::Uid& uid, const ::nn::account::SystemProgramIdentification& info, TokenStore::NsaId::ICanceler* pCanceler) NN_NOEXCEPT;

private:
    RequestScheduler            m_Scheduler;
    TransferStorage             m_TransferStorage;
    ::nn::ncm::ApplicationId    m_OwnerApplicationId;
};

}}}
