﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <memory>
#include <nn/dauth/dauth_Api.h>
#include <nn/dauth/dauth_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nim/nim_Result.h>

#include "nim_Json.h"
#include "nim_EShopUtil.h"
#include "nim_StringUtil.h"

NN_STATIC_ASSERT(sizeof(nn::ec::system::DeviceAuthenticationToken) >= nn::dauth::RequiredBufferSizeForDeviceAuthenticationToken);

namespace nn { namespace nim { namespace srv {

    namespace
    {
        // Shop 用 ClientId
        const uint64_t ClientId = 0x93af0acb26258de9ull;
    }

    Result GetShopDeviceAuthenticationToken(ec::system::DeviceAuthenticationToken* outToken, util::Cancelable* pCancelable) NN_NOEXCEPT
    {
        TimeSpan expiration;
        NN_RESULT_DO(GetShopDeviceAuthenticationToken(&expiration, outToken, pCancelable));

        NN_RESULT_SUCCESS;
    }
    Result GetShopDeviceAuthenticationToken(TimeSpan* outExpiration, ec::system::DeviceAuthenticationToken* outToken, util::Cancelable* pCancelable) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        int acquiredTokenSize;
        NN_RESULT_TRY(dauth::AcquireDeviceAuthenticationToken(
            outExpiration,
            &acquiredTokenSize,
            outToken->data,
            sizeof(outToken->data),
            ClientId,
            true,                       // true を指定するとキャッシュを無視します
            pCancelable
        ))
            NN_RESULT_CATCH_CONVERT(dauth::ResultCancelled, nim::ResultHttpConnectionCanceled())
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;
#else
        NN_UNUSED(outExpiration);
        NN_UNUSED(outToken);
        NN_UNUSED(pCancelable);
        NN_RESULT_THROW(ResultNotSupported());
#endif
    }

    Result ParseEShopErrorResponseDestructively(int* outValue, char* response) NN_NOEXCEPT
    {
        nne::rapidjson::Document document;
        NN_RESULT_THROW_UNLESS(!document.ParseInsitu(response).HasParseError(), ResultInvalidJson());

        auto errorObj = document.FindMember("error");
        NN_RESULT_THROW_UNLESS(errorObj != document.MemberEnd() && errorObj->value.IsObject(), ResultUnexpectedEShopErrorResponse());

        auto codeObj = errorObj->value.FindMember("code");
        NN_RESULT_THROW_UNLESS(codeObj != errorObj->value.MemberEnd() && codeObj->value.IsString(), ResultUnexpectedEShopErrorResponse());
        NN_RESULT_THROW_UNLESS(codeObj->value.GetStringLength() == 4, ResultUnexpectedEShopErrorCodeFormat());

        char* endPtr;
        auto code = std::strtol(codeObj->value.GetString(), &endPtr, 10);
        NN_RESULT_THROW_UNLESS(*endPtr == '\0', ResultUnexpectedEShopErrorCodeFormat());

        *outValue = code;
        NN_RESULT_SUCCESS;
    }

    Result ToEShopResultResponseDestructively(char* response) NN_NOEXCEPT
    {
        int errorCode;
        NN_RESULT_DO(ParseEShopErrorResponseDestructively(&errorCode, response));

        switch (errorCode)
        {
        case    0: NN_RESULT_THROW(ResultShogunInvalidParameters());
        case    2: NN_RESULT_THROW(ResultShogunServiceMaintenance());
        case    3: NN_RESULT_THROW(ResultShogunNotAuthorized());
        case    9: NN_RESULT_THROW(ResultShogunUnexpectedError());
        case   20: NN_RESULT_THROW(ResultAccountAlreadyDeviceLinked());
        case   85: NN_RESULT_THROW(ResultShogunPointUnreleased());
        case   86: NN_RESULT_THROW(ResultShogunPointExpired());
        case   87: NN_RESULT_THROW(ResultShogunPointNotFound());
        case   88: NN_RESULT_THROW(ResultShogunPointInvalidCountry());

        case 1000: NN_RESULT_THROW(ResultGameCardRegistrationInvalidParameters());
        case 1002: NN_RESULT_THROW(ResultGameCardRegistrationServiceMaintenance());
        case 1003: NN_RESULT_THROW(ResultGameCardRegistrationNotAuthorized());
        case 1008: NN_RESULT_THROW(ResultGameCardRegistrationUnexpectedDatabaseError());
        case 1009: NN_RESULT_THROW(ResultGameCardRegistrationUnexpectedError());
        case 1010: NN_RESULT_THROW(ResultGameCardRegistrationAlreadyRegistered());
        case 1011: NN_RESULT_THROW(ResultGameCardRegistrationUnexpectedGoldPoint());
        case 1012: NN_RESULT_THROW(ResultGameCardRegistrationInvalidSignature());
        case 1020: NN_RESULT_THROW(ResultGameCardRegistrationDeviceIdNotMatch());

        case 9600: NN_RESULT_THROW(ResultNdasInvalidDeviceToken());
        case 9601: NN_RESULT_THROW(ResultNdasConnectionFailure());
        case 9602: NN_RESULT_THROW(ResultNdasBadResponse());
        case 9609: NN_RESULT_THROW(ResultNdasUnexpectedError());
        case 9610: NN_RESULT_THROW(ResultNdasInvalidDeviceToken());
        case 9611: NN_RESULT_THROW(ResultNdasDeviceTokenExpired());

        case 9700: NN_RESULT_THROW(ResultNasConnectionFailure());
        case 9701: NN_RESULT_THROW(ResultNasBadResponse());
        case 9702: NN_RESULT_THROW(ResultNasInvalidParameters());
        case 9709: NN_RESULT_THROW(ResultNasUnexpectedError());
        case 9710: NN_RESULT_THROW(ResultNasInvalidTokenFormat());
        case 9711: NN_RESULT_THROW(ResultNasJwtVerifyFailed());
        case 9712: NN_RESULT_THROW(ResultNasJwtHasExpired());
        case 9713: NN_RESULT_THROW(ResultNasPublicKeyEndpointMismatch());

        case 9900: NN_RESULT_THROW(ResultEcgsConnectionFailure());
        case 9901: NN_RESULT_THROW(ResultEcgsBadResponse());

        default: NN_RESULT_THROW(ResultUnexpectedEShopErrorCode());

        }
    }

    Result ParseEciErrorResponseDestructively(int* outValue, char* response) NN_NOEXCEPT
    {
        nne::rapidjson::Document document;
        NN_RESULT_THROW_UNLESS(!document.ParseInsitu(response).HasParseError(), ResultInvalidJson());

        auto errorObj = document.FindMember("error");
        NN_RESULT_THROW_UNLESS(errorObj != document.MemberEnd() && errorObj->value.IsObject(), ResultUnexpectedEShopErrorResponse());

        auto codeObj = errorObj->value.FindMember("code");
        NN_RESULT_THROW_UNLESS(codeObj != errorObj->value.MemberEnd() && codeObj->value.IsInt(), ResultUnexpectedEShopErrorResponse());

        *outValue = codeObj->value.GetInt();
        NN_RESULT_SUCCESS;
    }

}}}
