﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/fs/fs_PriorityPrivate.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/nim/srv/nim_BackgroundDownloadStressTask.h>
#include <nn/nim/nim_BackgroundDownloadStressTaskInfo.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/nim/nim_Result.h>
#include "nim_StringUtil.h"

namespace nn { namespace nim { namespace srv {

    namespace {
        const size_t StackSize = 16 * 1024;
        NN_OS_ALIGNAS_THREAD_STACK char g_BackgroundDownloadStressTaskTaskStack[StackSize];
        const int ThreadPriority = NN_SYSTEM_THREAD_PRIORITY(nim, BackgroundDownloadStressTask);
        const char* ThreadName = NN_SYSTEM_THREAD_NAME(nim, BackgroundDownloadStressTask);

        const int WaitTime = 60;
        const size_t BufferSize = 256 * 1024;

        void GetContentIdFromUrl(ncm::ContentId* pOutContentId, const char* url) NN_NOEXCEPT
        {
            char charBytes[3] = {};

            const char* pTmp = nullptr;
            const char* contentIdStr = url;
            while((pTmp = std::strchr(contentIdStr, '/')) != nullptr)
            {
                contentIdStr = pTmp + 1;
            }

            for (int i = 0; i < sizeof(pOutContentId->data); i++)
            {
                charBytes[0] = contentIdStr[2 * i];
                charBytes[1] = contentIdStr[2 * i + 1];

                pOutContentId->data[i] = static_cast<Bit8>(NN_NIM_STR_TO_ULL(charBytes, nullptr, 16));
            }
        }

        Result WaitInternetConnection()
        {
            nifm::NetworkConnection connection(os::EventClearMode_ManualClear);
            NN_RESULT_DO(nifm::SetRequestRequirementPreset(connection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessPersistent));
            connection.SubmitRequest();

            auto& connectionEvent = connection.GetSystemEvent();
            while (!connection.IsAvailable())
            {
                NN_DETAIL_NIM_TRACE("[BackgroundDownloadStressTask] Connection is not available. Wait for connection\n");

                connectionEvent.Wait();
                connectionEvent.Clear();
            }
            NN_DETAIL_NIM_TRACE("[BackgroundDownloadStressTask] Connection is available.\n");

            NN_RESULT_SUCCESS;
        }

        Result FindContentLength(util::optional<int64_t>* outValue, const char* buffer, size_t size) NN_NOEXCEPT
        {
            util::optional<HttpHeaderValue> value;
            NN_RESULT_DO(FindHttpHeader(&value, "Content-Length", buffer, size));
            if (!value)
            {
                *outValue = util::nullopt;
                NN_RESULT_SUCCESS;
            }

            *outValue =  NN_NIM_STR_TO_LL(value->string, nullptr, 10);
            NN_RESULT_SUCCESS;
        }

        Result GetContentLength(int64_t* outValue, const char* url, HttpConnection* pConnection)
        {
            util::optional<int64_t> contentLength;
            NN_RESULT_DO(pConnection->Head(url, [&contentLength](const char* buffer, size_t size) -> Result
                {
                    if (!contentLength)
                    {
                        NN_RESULT_DO(FindContentLength(&contentLength, static_cast<const char*>(buffer), size));
                    }
                    NN_RESULT_SUCCESS;
                }));

            *outValue = contentLength ? *contentLength : -1;
            NN_RESULT_SUCCESS;
        }
    }

    const char* BackgroundDownloadStressTask::DefaultDownloadUrl = "https://atumn-dummy.hac.dd1.d4c.nintendo.net/c/c/606672cd6ba0df07a5a5d3b57365f863";

    BackgroundDownloadStressTask::BackgroundDownloadStressTask() NN_NOEXCEPT
        : m_Url()
        , m_PlaceHolderId()
        , m_StorageId(ncm::StorageId::None)
        , m_Connection()
        , m_Thread()
        , m_Event(os::EventClearMode_AutoClear)
        , m_BeginMeasurement()
        , m_BufferFilledSize()
        , m_WriteOffset()
        , m_MeasurementSize()
        , m_LastResult()
        , m_IsInitialized()
        , m_State(BackgroundDownloadStressTaskState::NotRunning)
    {
    }

    BackgroundDownloadStressTask::~BackgroundDownloadStressTask() NN_NOEXCEPT
    {
        if (m_IsInitialized)
        {
            m_Event.Signal();
            os::WaitThread(&m_Thread);
            os::DestroyThread(&m_Thread);
        }
    }

    Result BackgroundDownloadStressTask::Initialize(const char* url, ncm::StorageId storageId, DeviceContext* pDeviceContext) NN_NOEXCEPT
    {
        m_Url = url;
        m_StorageId = storageId;
        m_BufferSize = BufferSize;

        NN_RESULT_DO(m_Connection.Initialize(pDeviceContext));

        NN_RESULT_DO(os::CreateThread(&m_Thread, [](void* arg)
            {
                BackgroundDownloadStressTask* pTask = reinterpret_cast<BackgroundDownloadStressTask*>(arg);
                pTask->Run();
            }, this, g_BackgroundDownloadStressTaskTaskStack, StackSize, ThreadPriority));
        os::SetThreadNamePointer(&m_Thread, ThreadName);
        os::StartThread(&m_Thread);

        m_IsInitialized = true;
        NN_RESULT_SUCCESS;
    }

    void BackgroundDownloadStressTask::Run() NN_NOEXCEPT
    {
        // ncm で発生する fs アクセスの優先度を設定する
        // 注意:    IPC 先で fs アクセスが発生するときに影響するので、
        //          他のプロセスへの IPC を行うときには注意
        fs::SetPriorityRawOnCurrentThread(fs::PriorityRaw_Background);
        while (!m_Event.TryWait())
        {
            m_LastResult = Execute();
            if (m_LastResult.IsFailure())
            {
                m_State = BackgroundDownloadStressTaskState::Error;
                m_Throughput = 0.0;
                os::SleepThread(TimeSpan::FromSeconds(WaitTime));
                continue;
            }
        }
    }

    Result BackgroundDownloadStressTask::Execute() NN_NOEXCEPT
    {
        m_LastResult = ResultSuccess();
        m_State = BackgroundDownloadStressTaskState::Waiting;

        m_Buffer = new Bit8[m_BufferSize];
        NN_RESULT_THROW_UNLESS(m_Buffer != nullptr, ResultDownloadStressTestOutOfMemory());
        NN_UTIL_SCOPE_EXIT { delete[] m_Buffer; };

        NN_RESULT_DO(WaitInternetConnection());

        NN_RESULT_DO(SetupContentStorage());
        NN_UTIL_SCOPE_EXIT
        {
            CleanupContentStorage();
        };

        m_State = BackgroundDownloadStressTaskState::Downloading;
        const char* rangeHeader;
        m_WriteOffset = 0;
        m_BufferFilledSize = 0;
        m_MeasurementSize = 0;
        m_BeginMeasurement = os::GetSystemTick();
        NN_RESULT_DO(m_Connection.Get(m_Url, [this](const void* buffer, size_t bufferSize) -> Result
            {

                Measure(bufferSize);
                if (bufferSize > m_BufferSize)
                {
                    NN_RESULT_DO(Flush());
                    NN_RESULT_DO(WriteToPlaceHolder(buffer, bufferSize));
                    NN_RESULT_SUCCESS;
                }

                auto filledSize = FillBuffer(buffer, bufferSize);
                if (filledSize < bufferSize)
                {
                    NN_RESULT_DO(Flush());
                    FillBuffer(reinterpret_cast<const char*>(buffer) - filledSize, bufferSize - filledSize);
                }

                NN_RESULT_SUCCESS;
            }, &rangeHeader, 0));
        Measure(0);
        NN_RESULT_DO(Flush());

        NN_RESULT_SUCCESS;
    }

    Result BackgroundDownloadStressTask::Flush() NN_NOEXCEPT
    {
        NN_RESULT_DO(WriteToPlaceHolder(m_Buffer, m_BufferFilledSize));
        m_BufferFilledSize = 0;
        NN_RESULT_SUCCESS;
    }

    Result BackgroundDownloadStressTask::WriteToPlaceHolder(const void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_ContentStorage.WritePlaceHolder(m_PlaceHolderId, m_WriteOffset, buffer, bufferSize));
        m_WriteOffset += bufferSize;
        NN_RESULT_SUCCESS;
    }

    Result BackgroundDownloadStressTask::SetupContentStorage() NN_NOEXCEPT
    {
        int64_t contentLength;
        NN_RESULT_DO(GetContentLength(&contentLength, m_Url, &m_Connection));
        NN_RESULT_THROW_UNLESS(contentLength > 0, nim::ResultUnexpectedDataReceived());

        ncm::ContentId contentId;
        GetContentIdFromUrl(&contentId, m_Url);

        NN_RESULT_DO(ncm::OpenContentStorage(&m_ContentStorage, m_StorageId));

        m_PlaceHolderId = m_ContentStorage.GeneratePlaceHolderId();
        NN_RESULT_DO(m_ContentStorage.CreatePlaceHolder(m_PlaceHolderId, contentId, contentLength));

        NN_RESULT_SUCCESS;
    }

    void BackgroundDownloadStressTask::CleanupContentStorage() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_ContentStorage.DeletePlaceHolder(m_PlaceHolderId));
    }

    void BackgroundDownloadStressTask::Measure(size_t downloadSize) NN_NOEXCEPT
    {
        auto endMeasurement = os::GetSystemTick();
        m_MeasurementSize += downloadSize;

        if (m_MeasurementSize > 0x100000 || m_MeasurementSize == 0)
        {
            auto time = (endMeasurement - m_BeginMeasurement).ToTimeSpan().GetSeconds();
            if (time > 0)
            {
                m_Throughput = (m_MeasurementSize / time) / 1000.0;
                m_MeasurementSize = 0;
                m_BeginMeasurement = os::GetSystemTick();
            }
        }
    }

    void BackgroundDownloadStressTask::GetInfo(BackgroundDownloadStressTaskInfo* outValue) const NN_NOEXCEPT
    {
        *outValue = { m_IsInitialized, m_State, m_StorageId, {}, m_LastResult.GetInnerValueForDebug(), m_Throughput };
    }

    size_t BackgroundDownloadStressTask::FillBuffer(const void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        auto fillSize = std::min(m_BufferSize - m_BufferFilledSize, bufferSize);

        std::memcpy(m_Buffer + m_BufferFilledSize, buffer, fillSize);
        m_BufferFilledSize += fillSize;

        return fillSize;
    }
}}}
