﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/es/es_Api.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include "nim_AsyncSyncTicketImpl.h"
#include "nim_AsyncTaskBufferAllocator.h"
#include "nim_EcSystemThreadAllocator.h"

namespace nn { namespace nim { namespace srv {

    namespace {
        const int AccountListETicketIdMinimumResponseSize = 2048;
    }

    AsyncSyncTicketImpl::AsyncSyncTicketImpl(DeviceAccountStore* store) NN_NOEXCEPT : AsyncResultImpl(this, &GetEcSystemThreadAllocator()), m_Store(store), m_Mutex(false) {}

    AsyncSyncTicketImpl::~AsyncSyncTicketImpl() NN_NOEXCEPT
    {
        AsyncResultImpl<AsyncSyncTicketImpl>::Join();
        if (m_WorkBuffer)
        {
            GetAsyncTaskBufferAllocator()->Free(m_WorkBuffer, SyncTicketTaskBufferFlag::Mask);
            m_WorkBuffer = nullptr;
        }
    }

    Result AsyncSyncTicketImpl::Initialize(DeviceContext* deviceContext) NN_NOEXCEPT
    {
        // プログレスを初期化
        SetProgress({ 0, 0 });

        m_DeviceContext = deviceContext;
        return m_Connection.Initialize(deviceContext);
    }

    Result AsyncSyncTicketImpl::Execute() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_WorkBuffer == nullptr);
        size_t size;
        m_WorkBuffer = reinterpret_cast<char*>(GetAsyncTaskBufferAllocator()->Allocate(&size, SyncTicketTaskBufferFlag::Mask));

        // アロケータの初期化
        mem::StandardAllocator allocator(m_WorkBuffer, size);

        EciAccessor accessor(m_DeviceContext, &m_Connection);
        DeviceAccountInfo deviceAccountInfo;
        NN_RESULT_DO(m_Store->Get(&deviceAccountInfo));

        // レスポンスを格納するバッファを確保(最小のレスポンスサイズ + tiv の最大文字数 * 最大チケット数)
        size_t responseBufferSize = static_cast<size_t>(AccountListETicketIdMinimumResponseSize + (sizeof("18446744073709551615") + 5) * MaxTicketCount);
        char* responseBuffer = new(allocator.Allocate(responseBufferSize)) char[responseBufferSize];

        // TicketId のリストを格納する配列を確保
        size_t ticketIdListSize = sizeof(es::TicketId) * MaxTicketCount;
        Bit64* ticketIdList = new(allocator.Allocate(ticketIdListSize)) Bit64[MaxTicketCount];

        // 権利を持っているチケット一覧を取得
        int ticketCount;
        NN_RESULT_TRY(accessor.AccountListETicketIds(responseBuffer, responseBufferSize, &ticketCount, ticketIdList, MaxTicketCount, deviceAccountInfo.id, deviceAccountInfo.token))
            NN_RESULT_CATCH(ResultPersonalizedTicketNotFound)
            {
                // パーソナライズドチケットと予約購入情報をチケット DB から全削除
                es::DeleteAllPersonalizedTicket();
                es::DeleteAllPrepurchaseRecord();

                // チケット同期フラグを落とす
                NN_RESULT_DO(accessor.CompleteETicketSync(deviceAccountInfo.token));

                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY

        // 権利を持っていないチケットをチケット DB から削除
        es::DeleteAllPersonalizedTicket(ticketIdList, ticketCount);

        size_t missingTicketIdListSize = sizeof(es::TicketId) * ticketCount;
        Bit64* missingTicketIdList = new(allocator.Allocate(missingTicketIdListSize)) Bit64[ticketCount];

        // 権利を持っているチケット一覧からチケット DB にないチケット一覧を取得
        int missingTicketCount = es::ListMissingPersonalizedTicket(missingTicketIdList, ticketCount, ticketIdList, ticketCount);

        // プログレスを更新
        SetProgress({ 0, missingTicketCount });

        // 一度にチケットを落とす数(とりあえず 32 枚に固定する、これを修正するときには AccountGetETickets の Post バッファがあふれないか確認が必要)
        const int OneRequestDownloadTicketCount = 32;

        for (int i = 0; i < missingTicketCount; i += OneRequestDownloadTicketCount)
        {
            int downloadTicketCount = i + OneRequestDownloadTicketCount <= missingTicketCount ? OneRequestDownloadTicketCount : missingTicketCount - i;

            // チケットをダウンロードしてインストールする
            NN_RESULT_DO(accessor.AccountGetETicketsForSyncTicket(responseBuffer, responseBufferSize, deviceAccountInfo.id, deviceAccountInfo.token, missingTicketIdList + i, downloadTicketCount));

            // プログレスを更新
            SetProgress({ i + downloadTicketCount, missingTicketCount });
        }

        // プログレスを更新
        SetProgress({ missingTicketCount, missingTicketCount });

        // チケット同期フラグを落とす
        NN_RESULT_DO(accessor.CompleteETicketSync(deviceAccountInfo.token));

        NN_RESULT_SUCCESS;
    }

    Result AsyncSyncTicketImpl::Cancel() NN_NOEXCEPT
    {
        m_Connection.Cancel();
        NN_RESULT_SUCCESS;
    }

    Result AsyncSyncTicketImpl::GetProgress(sf::Out<AsyncProgress> outValue) NN_NOEXCEPT
    {
        *outValue = GetProgress();
        NN_RESULT_SUCCESS;
    }

}}}
