﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/util/util_StringUtil.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentInfoUtil.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_ContentMetaUtil.h>
#include <nn/ncm/ncm_StorageUtil.h>

#include <nn/nim/srv/nim_ApplyDownloadedDeltaTask.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include "nim_ResultMappingUtil.h"

namespace nn { namespace nim { namespace srv {

namespace
{
Result CalculateActualOccupiedSize(int64_t* outSize, const ncm::ContentMetaKey& key, ncm::ContentMetaDatabase* db, ncm::ContentStorage* storage) NN_NOEXCEPT
{
    int64_t size = 0;

    bool has;
    NN_RESULT_DO(db->Has(&has, key));
    if (has)
    {
        const int InfoBufferSize = 16;
        ncm::ContentInfo infos[InfoBufferSize];
        int offset = 0;
        int outCount;
        for (;;)
        {
            NN_RESULT_DO(db->ListContentInfo(&outCount, infos, InfoBufferSize, key, offset));
            for (int i = 0; i < outCount; ++i)
            {
                bool hasContent;
                NN_RESULT_DO(storage->Has(&hasContent, infos[i].id));
                if (hasContent)
                {
                    size += infos[i].GetSize();
                }
            }
            if (outCount != InfoBufferSize)
            {
                break;
            }
            offset += outCount;
        }
    }
    *outSize = size;

    NN_RESULT_SUCCESS;
}
}

Result FileApplyDownloadedDeltaTaskMeta::Create(const char* path, ncm::ApplicationId appId, const ncm::ContentMetaKey& sourceKey, const ncm::ContentMetaKey* keyList, int count, ncm::StorageId storageId) NN_NOEXCEPT
{
    // バージョンが昇順であることの確認
    auto currentVersion = sourceKey.version;
    NN_RESULT_THROW_UNLESS(sourceKey.type == ncm::ContentMetaType::Patch, ResultUnexpectedContentMetaTypeExist());

    for (int i = 0; i < count; ++i)
    {
        NN_RESULT_THROW_UNLESS(keyList[i].type == ncm::ContentMetaType::Patch, ResultUnexpectedContentMetaTypeExist());
        NN_RESULT_THROW_UNLESS(currentVersion < keyList[i].version, ResultOutOfOrderPatchKeys());
        currentVersion = keyList[i].version;
    }

    auto header = MakeHeader(appId, sourceKey, count, storageId, MetaVersion::Version0);

    fs::FileHandle file;
    NN_RESULT_DO(fs::OpenFile(&file, path, fs::OpenMode_Write | fs::OpenMode_AllowAppend));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

    NN_RESULT_DO(fs::WriteFile(file, 0, &header, sizeof(header), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));

    NN_RESULT_DO(fs::WriteFile(file, sizeof(header), keyList, sizeof(ncm::ContentMetaKey) * count, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));

    NN_RESULT_SUCCESS;
}

Result FileApplyDownloadedDeltaTaskMeta::Create(const char* path, ncm::ApplicationId appId, const ncm::ContentMetaKey& sourceKey, ncm::StorageId storageId, const ncm::StorageContentMetaKey* keyList, int count) NN_NOEXCEPT
{
    // バージョンが昇順であることの確認
    auto currentVersion = sourceKey.version;
    NN_RESULT_THROW_UNLESS(sourceKey.type == ncm::ContentMetaType::Patch, ResultUnexpectedContentMetaTypeExist());

    for (int i = 0; i < count; ++i)
    {
        NN_RESULT_THROW_UNLESS(keyList[i].key.type == ncm::ContentMetaType::Patch, ResultUnexpectedContentMetaTypeExist());
        NN_RESULT_THROW_UNLESS(currentVersion < keyList[i].key.version, ResultOutOfOrderPatchKeys());
        currentVersion = keyList[i].key.version;
    }

    auto header = MakeHeader(appId, sourceKey, count, storageId, MetaVersion::Version1);

    fs::FileHandle file;
    NN_RESULT_DO(fs::OpenFile(&file, path, fs::OpenMode_Write | fs::OpenMode_AllowAppend));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

    NN_RESULT_DO(fs::WriteFile(file, 0, &header, sizeof(header), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));

    NN_RESULT_DO(fs::WriteFile(file, sizeof(header), keyList, sizeof(ncm::StorageContentMetaKey) * count, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));

    NN_RESULT_SUCCESS;
}

Result FileApplyDownloadedDeltaTaskMeta::Initialize(const char* path) NN_NOEXCEPT
{
    fs::FileHandle file;
    NN_RESULT_DO(fs::OpenFile(&file, path, fs::OpenMode_Read | fs::OpenMode_Write | fs::OpenMode_AllowAppend));
    m_File = file;
    NN_RESULT_DO(fs::ReadFile(*m_File, 0, &m_Header, sizeof(m_Header)));

    NN_RESULT_SUCCESS;
}
Result FileApplyDownloadedDeltaTaskMeta::GetContentMeta(ncm::StorageContentMetaKey* outValue, int index) NN_NOEXCEPT
{
    switch(m_Header.version)
    {
    case MetaVersion::Version0:
        {
            ncm::ContentMetaKey key;
            NN_RESULT_DO(fs::ReadFile(*m_File, sizeof(m_Header) + sizeof(ncm::ContentMetaKey) * index, &key, sizeof(key)));
            *outValue = {key, m_Header.storageId};
            NN_RESULT_SUCCESS;
        }
    case MetaVersion::Version1:
        {
            NN_RESULT_DO(fs::ReadFile(*m_File, sizeof(m_Header) + sizeof(ncm::StorageContentMetaKey) * index, outValue, sizeof(*outValue)));
            NN_RESULT_SUCCESS;
        }
    default:
        NN_RESULT_THROW(ResultUnknownFormatVersion());
    }
}

FileApplyDownloadedDeltaTaskMeta::~FileApplyDownloadedDeltaTaskMeta() NN_NOEXCEPT
{
    if (m_File)
    {
        fs::CloseFile(*m_File);
    }
}

Result FileApplyDownloadedDeltaTaskData::Create(const char* path, int totalContentMeta) NN_NOEXCEPT
{

    fs::FileHandle file;
    size_t size = sizeof(Data) + sizeof(ncm::ApplyDeltaTaskBase::TaskState);
    NN_RESULT_DO(fs::OpenFile(&file, path, fs::OpenMode_Write | fs::OpenMode_AllowAppend));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

    NN_RESULT_DO(fs::SetFileSize(file, size));

    Data data = {};
    data.applyDownloadedState = ApplyDownloadedDeltaProgressState::NotRunning;
    data.totalContentMeta = totalContentMeta;

    NN_RESULT_DO(fs::WriteFile(file, 0, &data, sizeof(data), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));

    NN_RESULT_SUCCESS;
}


Result FileApplyDownloadedDeltaTaskData::Initialize(const char* path, FileApplyDownloadedDeltaTaskMeta* meta) NN_NOEXCEPT
{
    m_Meta = meta;
    util::Strlcpy(m_Path, path, static_cast<int>(sizeof(m_Path)));

    {
        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, path, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };
        NN_RESULT_DO(fs::ReadFile(file, 0, &m_Data, sizeof(m_Data)));
        NN_RESULT_DO(fs::ReadFile(file, sizeof(m_Data), &m_ApplyTaskState, sizeof(m_ApplyTaskState)));
    }

    if (GetState() == ApplyDownloadedDeltaProgressState::NotRunning)
    {
        m_Data.sourceKey = m_Meta->GetSourceKey();
        ncm::StorageContentMetaKey key;
        NN_RESULT_DO(m_Meta->GetContentMeta(&key, 0));
        m_Data.destinationKey = key.key;
        m_Data.destinationStorageId = key.storageId;
        NN_RESULT_DO(Flush());
    }

    NN_RESULT_SUCCESS;
}

Result FileApplyDownloadedDeltaTaskData::SetState(ApplyDownloadedDeltaProgressState state) NN_NOEXCEPT
{
    m_Data.applyDownloadedState = state;
    NN_RESULT_DO(Flush());
    NN_RESULT_SUCCESS;
}
Result FileApplyDownloadedDeltaTaskData::SetLastResult(Result result) NN_NOEXCEPT
{
    auto mappedResult = MapTaskResult(result);
    std::memcpy(&(m_Data.lastResult), &mappedResult, sizeof(mappedResult));
    NN_RESULT_DO(Flush());
    NN_RESULT_SUCCESS;
}
ApplyDownloadedDeltaProgress FileApplyDownloadedDeltaTaskData::GetProgress(const util::optional<ncm::ApplyDeltaProgress>& ncmProgress) NN_NOEXCEPT
{
    const int64_t ProgressPerDelta = 10000; // 1 パッチ間差分あたりどのくらいプログレスを進めるか

    ApplyDownloadedDeltaProgress progress = {ProgressPerDelta * m_Data.currentIndex, ProgressPerDelta * m_Data.totalContentMeta};
    if (ncmProgress)
    {
        switch (ncmProgress->state)
        {
        case ncm::ApplyDeltaProgressState_NotPrepared:
            // nothing to do
            break;
        case ncm::ApplyDeltaProgressState_Prepared:
        case ncm::ApplyDeltaProgressState_Applying:
        case ncm::ApplyDeltaProgressState_Suspended:
            // 取得したプログレスだけ進める
            progress.applied += ncmProgress->appliedDeltaSize * ProgressPerDelta / ncmProgress->totalDeltaSize;
            break;
        case ncm::ApplyDeltaProgressState_DeltaApplied:
        case ncm::ApplyDeltaProgressState_Commited:
            // 全体分進める
            progress.applied += ProgressPerDelta;
            break;
        default:
            // とりあえずそのまま
            break;
        }
    }
    // ロックをしていない関係上、タイミングによっては巻き戻るのでその回避
    if (progress.applied < m_Data.applied)
    {
        progress.applied = m_Data.applied;
    }
    else
    {
        // 本当はここで flush したいが、電源断で progress が戻る可能性があるだけなので flush 無しとしておく
        m_Data.applied = progress.applied;
    }
    progress.state = GetState();
    if (progress.state == ApplyDownloadedDeltaProgressState::Fatal)
    {
        progress.lastResult = m_Data.lastResult;
    }
    return progress;
}
Result FileApplyDownloadedDeltaTaskData::SetRequiredSize(int64_t size) NN_NOEXCEPT
{
    m_Data.requiredSize = size;
    NN_RESULT_DO(Flush());

    NN_RESULT_SUCCESS;
}

Result FileApplyDownloadedDeltaTaskData::SetTaskStateAvailable(bool flag) NN_NOEXCEPT
{
    m_Data.taskStateAvailable = flag;
    NN_RESULT_DO(Flush());

    NN_RESULT_SUCCESS;
}
Result FileApplyDownloadedDeltaTaskData::MoveNext(bool* outComplete) NN_NOEXCEPT
{
    m_Data.sourceKey = GetDestinationKey().key;
    m_Data.sourceKey.installType = ncm::ContentInstallType::Full;
    m_Data.currentIndex++;
    m_Data.taskStateAvailable = false;

    if (m_Data.currentIndex == m_Data.totalContentMeta)
    {
        *outComplete = true;
    }
    else
    {
        *outComplete = false;
        ncm::StorageContentMetaKey key;
        NN_RESULT_DO(m_Meta->GetContentMeta(&key, m_Data.currentIndex));
        m_Data.destinationKey = key.key;
        m_Data.destinationStorageId = key.storageId;
    }
    NN_RESULT_DO(Flush());

    NN_RESULT_SUCCESS;
}
Result FileApplyDownloadedDeltaTaskData::Flush() NN_NOEXCEPT
{
    fs::FileHandle file;
    NN_RESULT_DO(fs::OpenFile(&file, m_Path, fs::OpenMode_Read | fs::OpenMode_Write | fs::OpenMode_AllowAppend));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

    NN_RESULT_DO(fs::WriteFile(file, 0, &m_Data, sizeof(m_Data), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
    NN_RESULT_DO(fs::WriteFile(file, sizeof(m_Data), &m_ApplyTaskState, sizeof(m_ApplyTaskState), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));

    NN_RESULT_SUCCESS;
}

void FileApplyDownloadedDeltaTaskData::ResetThroughput() NN_NOEXCEPT
{
    m_InitialProcessedSize = m_Data.applied;
    m_BeginMeasurementTick = os::Tick(0);
    m_TotalExecuteTime = 0;
}

void FileApplyDownloadedDeltaTaskData::EndMeasurement() NN_NOEXCEPT
{
    auto end = os::GetSystemTick();
    m_TotalExecuteTime += (end - m_BeginMeasurementTick).ToTimeSpan().GetMilliSeconds();
    m_BeginMeasurementTick = os::Tick(0);
}

ApplyDeltaThroughput FileApplyDownloadedDeltaTaskData::GetThroughput() NN_NOEXCEPT
{
    // 実行中の場合、現状の処理時間を加算する
    int64_t extraTime = (m_BeginMeasurementTick == os::Tick(0)) ? 0 : ((os::GetSystemTick() - m_BeginMeasurementTick).ToTimeSpan().GetMilliSeconds());
    int64_t executeTime = m_TotalExecuteTime + extraTime;
    ApplyDeltaThroughput throughput = {m_Data.applied - m_InitialProcessedSize, TimeSpan::FromMilliSeconds(executeTime)};
    return throughput;
}

Result ApplyDownloadedDeltaTask::Initialize(const char* metaFilePath, const char* dataFilePath) NN_NOEXCEPT
{
    NN_UNUSED(dataFilePath);

    NN_RESULT_DO(m_Meta.Initialize(metaFilePath));
    NN_RESULT_DO(m_Data.Initialize(dataFilePath, &m_Meta));

    m_IsTaskInitialized = false;
    m_CancelToken = false;
    m_ApplyTaskProgressAvailable = false;
    m_Data.ResetThroughput();

    NN_RESULT_SUCCESS;
}
Result ApplyDownloadedDeltaTask::InitialPrepareImpl() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Data.GetState() == ApplyDownloadedDeltaProgressState::NotRunning, ResultApplyDeltaInvalidState());
    // 更新対象のパッチが見つからない場合は、StorageId::None が設定されてくるとしている
    NN_RESULT_THROW_UNLESS(ncm::IsInstallableStorage(m_Data.GetSourceKey().storageId), ResultSourcePatchNotFound());

    ncm::StorageContentMetaKey source = m_Data.GetSourceKey();
    ncm::StorageContentMetaKey destination = m_Data.GetDestinationKey();

    NN_RESULT_DO(m_ApplyTask.Initialize(source, destination, m_Data.GetTaskState()));
    m_IsTaskInitialized = true;
    NN_RESULT_DO(m_Data.SetTaskStateAvailable(true));
    NN_RESULT_TRY(m_ApplyTask.Prepare())
        NN_RESULT_CATCH(ncm::ResultNotEnoughSpaceToApplyDelta)
        {
            int64_t sizeRequired;
            NN_RESULT_DO(m_ApplyTask.CalculateRequiredSizeMaxForPrepare(&sizeRequired));
            NN_RESULT_DO(m_Data.SetRequiredSize(sizeRequired));
            NN_RESULT_DO(m_Data.SetState(ApplyDownloadedDeltaProgressState::NotEnoughSpace));
            NN_RESULT_THROW(ResultApplyDeltaNotEnoughSpace());
        }
    NN_RESULT_END_TRY;

    NN_RESULT_DO(m_Data.SetState(ApplyDownloadedDeltaProgressState::InitialPrepared));

    NN_RESULT_SUCCESS;
}
Result ApplyDownloadedDeltaTask::InitialPrepare() NN_NOEXCEPT
{
    auto result = InitialPrepareImpl();
    if (result.IsFailure() && !ResultApplyDeltaNotEnoughSpace::Includes(result))
    {
        NN_RESULT_DO(m_Data.SetLastResult(result));
        NN_RESULT_DO(m_Data.SetState(ApplyDownloadedDeltaProgressState::Fatal));
        NN_RESULT_THROW(result);
    }
    else
    {
        NN_RESULT_DO(m_Data.Flush());
    }
    NN_RESULT_SUCCESS;
}
Result ApplyDownloadedDeltaTask::PrepareAndExecuteImpl() NN_NOEXCEPT
{
    // ApplyPatchDeltaTask を順次実行する
    // Commit を逐次して、ApplyDownloadedDeltaTask の Commit は何もしないとする
    NN_RESULT_THROW_UNLESS(
        m_Data.GetState() == ApplyDownloadedDeltaProgressState::InitialPrepared ||
        m_Data.GetState() == ApplyDownloadedDeltaProgressState::Suspended,
        ResultApplyDeltaInvalidState()
    );
    NN_RESULT_DO(ThrowIfCancelRequested());

    NN_RESULT_DO(m_Data.SetState(ApplyDownloadedDeltaProgressState::Applying));
    for(;;)
    {
        if (!m_IsTaskInitialized)
        {
            if (m_Data.IsTaskStateAvailable())
            {
                NN_RESULT_DO(m_ApplyTask.InitializeForResume(m_Data.GetTaskState()));
            }
            else
            {
                ncm::StorageContentMetaKey source = m_Data.GetSourceKey();
                ncm::StorageContentMetaKey destination = m_Data.GetDestinationKey();
                NN_RESULT_DO(m_ApplyTask.Initialize(source, destination, m_Data.GetTaskState()));
                NN_RESULT_DO(m_Data.SetTaskStateAvailable(true));
            }
            m_IsTaskInitialized = true;
        }
        {
            m_ApplyTaskProgressAvailable = true;
            NN_UTIL_SCOPE_EXIT
            {
                // 現時点の進捗を永続化しておくために、GetProgress を呼んでおく
                GetProgress();
                m_ApplyTaskProgressAvailable = false;
            };
            if (m_ApplyTask.GetProgress().state == ncm::ApplyDeltaProgressState_NotPrepared)
            {
                NN_RESULT_DO(m_ApplyTask.Prepare());
            }
            // execute 不可な状態でここに来ることは無い想定
            NN_SDK_ASSERT(m_ApplyTask.GetProgress().state == ncm::ApplyDeltaProgressState_Prepared
                          || m_ApplyTask.GetProgress().state == ncm::ApplyDeltaProgressState_Suspended);

            // タイミングの問題があるので、ここでもチェック
            NN_RESULT_DO(ThrowIfCancelRequested());

            NN_RESULT_TRY(m_ApplyTask.Execute())
                NN_RESULT_CATCH(ncm::ResultCancelled)
                {
                    NN_RESULT_DO(m_Data.SetState(ApplyDownloadedDeltaProgressState::Suspended));
                    NN_RESULT_THROW(ResultApplyDeltaCancelled());
                }
                NN_RESULT_CATCH(ncm::ResultNotEnoughSpaceToApplyDelta)
                {
                    int64_t sizeRequired;
                    NN_RESULT_DO(m_ApplyTask.CalculateRequiredSizeMaxForResume(&sizeRequired));
                    NN_RESULT_DO(m_Data.SetRequiredSize(sizeRequired));

                    NN_RESULT_DO(m_Data.SetState(ApplyDownloadedDeltaProgressState::NotEnoughSpace));
                    NN_RESULT_THROW(ResultApplyDeltaNotEnoughSpace());
                }
            NN_RESULT_END_TRY
            NN_RESULT_DO(m_ApplyTask.Commit());
        }

        bool isComplete;
        NN_RESULT_DO(MoveNext(&isComplete));
        if (isComplete)
        {
            break;
        }
        NN_RESULT_DO(ThrowIfCancelRequested());
    }
    NN_RESULT_DO(m_Data.SetState(ApplyDownloadedDeltaProgressState::AllApplied));

    NN_RESULT_SUCCESS;
}
Result ApplyDownloadedDeltaTask::PrepareAndExecute() NN_NOEXCEPT
{
    m_Data.BeginMeasurement();
    NN_UTIL_SCOPE_EXIT { m_Data.EndMeasurement(); };

    auto result = PrepareAndExecuteImpl();
    if (result.IsFailure() && !ResultApplyDeltaCancelled::Includes(result) && !ResultApplyDeltaNotEnoughSpace::Includes(result))
    {
        NN_RESULT_DO(m_Data.SetLastResult(result));
        NN_RESULT_DO(m_Data.SetState(ApplyDownloadedDeltaProgressState::Fatal));
        NN_RESULT_THROW(result);
    }
    else
    {
        NN_RESULT_DO(m_Data.Flush());
    }
    NN_RESULT_SUCCESS;
}
ApplyDownloadedDeltaProgress ApplyDownloadedDeltaTask::GetProgress() NN_NOEXCEPT
{
    util::optional<ncm::ApplyDeltaProgress> progress = util::nullopt;
    if (m_ApplyTaskProgressAvailable)
    {
        progress = m_ApplyTask.GetProgress();
    }
    return m_Data.GetProgress(progress);
}
Result ApplyDownloadedDeltaTask::CommitImpl(bool doCleanup) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Data.GetState() == ApplyDownloadedDeltaProgressState::AllApplied, ResultApplyDeltaInvalidState());
    NN_RESULT_DO(m_Data.SetState(ApplyDownloadedDeltaProgressState::Commited));

    if (doCleanup)
    {
        NN_RESULT_DO(Cleanup());
    }

    NN_RESULT_SUCCESS;
}

Result ApplyDownloadedDeltaTask::Commit(bool doCleanup) NN_NOEXCEPT
{
    auto result = CommitImpl(doCleanup);
    if (result.IsFailure())
    {
        NN_RESULT_DO(m_Data.SetLastResult(result));
        NN_RESULT_DO(m_Data.SetState(ApplyDownloadedDeltaProgressState::Fatal));
        NN_RESULT_THROW(result);
    }

    NN_RESULT_SUCCESS;
}
Result ApplyDownloadedDeltaTask::CalculateRequiredSize(int64_t* outValue) NN_NOEXCEPT
{
    *outValue = m_Data.GetRequiredSize();

    NN_RESULT_SUCCESS;
}
Result ApplyDownloadedDeltaTask::CalculateOccupiedSize(int64_t* outValue, ncm::StorageId storageId) NN_NOEXCEPT
{
    // 更新途中のパッチは、このタスクが計上すべきなので計上する
    // TODO: 更新対象のパッチが記録よりも古く、かつ差分更新途中で記録と同じバージョンのパッチになると
    //       二重計上されてしまうことになることをどう考えるか。
    //       レアなので考えないとするか、更新途中で完全なパッチになった状態は計上しないとするか、、くらい。
    //       とりあえずここでは前者とする。
    int64_t size = 0;

    {
        ncm::ContentMetaDatabase db;
        ncm::ContentStorage storage;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, storageId));
        NN_RESULT_DO(ncm::OpenContentStorage(&storage, storageId));

        for (int i = 0; i < m_Meta.CountContentMeta(); ++i)
        {
            ncm::StorageContentMetaKey key;
            NN_RESULT_DO(m_Meta.GetContentMeta(&key, i));
            if (key.storageId == storageId)
            {
                // パッチ間差分の計上
                int64_t occupied;
                NN_RESULT_DO(CalculateActualOccupiedSize(&occupied, key.key, &db, &storage));
                size += occupied;

                // 更新途中のパッチの計上
                // TODO で触れていた話で安全側に倒すなら、ここをスキップする
                key.key.installType = ncm::ContentInstallType::Full;
                NN_RESULT_DO(CalculateActualOccupiedSize(&occupied, key.key, &db, &storage));
                size += occupied;
            }
        }
    }
    // 更新途中の placeholder の計上
    if (m_IsTaskInitialized)
    {
        int64_t occupied;
        NN_RESULT_DO(m_ApplyTask.CalculateOccupiedPlaceholderSize(&occupied, storageId));
        size += occupied;
    }
    *outValue = size;

    NN_RESULT_SUCCESS;
}
Result ApplyDownloadedDeltaTask::Cleanup() NN_NOEXCEPT
{
    if (m_IsTaskInitialized)
    {
        NN_RESULT_DO(m_ApplyTask.Cleanup());
    }
    ncm::ContentMetaDatabase sourceDb;
    ncm::ContentStorage sourceStorage;
    ncm::StorageList commitList;

    bool isSourceAvailable = ncm::IsInstallableStorage(m_Meta.GetStorageId());
    if (isSourceAvailable)
    {
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&sourceDb, m_Meta.GetStorageId()));
        NN_RESULT_DO(ncm::OpenContentStorage(&sourceStorage, m_Meta.GetStorageId()));
    }

    ncm::ContentMetaKey latestKey;
    ncm::ContentMetaKey* latestKeyPointer = isSourceAvailable ? &latestKey : nullptr;
    if (isSourceAvailable)
    {
        NN_RESULT_TRY(sourceDb.GetLatest(&latestKey, m_Meta.GetSourceKey().id))
            NN_RESULT_CATCH(ncm::ResultContentMetaNotFound) {latestKeyPointer = nullptr;}
        NN_RESULT_END_TRY
    }
    for (int i = 0; i < m_Meta.CountContentMeta(); ++i)
    {
        ncm::StorageContentMetaKey key;
        NN_RESULT_DO(m_Meta.GetContentMeta(&key, i));

        ncm::ContentMetaDatabase deltaDb;
        ncm::ContentStorage deltaStorage;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&deltaDb, key.storageId));
        NN_RESULT_DO(ncm::OpenContentStorage(&deltaStorage, key.storageId));

        bool has;
        NN_RESULT_DO(deltaDb.Has(&has, key.key));
        if (has)
        {
            // 登録されているパッチを削除
            // ただし削除するのは fragment-only のもの
            ncm::ContentManagerAccessor accessor(&deltaDb, &deltaStorage);
            auto result = accessor.DeleteRedundant(key.key, (m_Meta.GetStorageId() == key.storageId) ? latestKeyPointer : nullptr);
            if (result.IsFailure())
            {
                NN_DETAIL_NIM_TRACE("[ApplyDownloadedDeltaTask] Failed to delete as 0x%08x\n", result.GetInnerValueForDebug());
            }
            commitList.Push(key.storageId);
        }
        if (isSourceAvailable && m_Data.GetState() != ApplyDownloadedDeltaProgressState::Commited)
        {
            // commit 以外の状態で cleanup された場合は更新対象だったパッチ以外は削除する
            // 仮に正しく更新された状態だったとしても、ns の ApplicationRecord に記録されないのでどうせ利用ができない
            key.key.installType = ncm::ContentInstallType::Full;
            NN_RESULT_DO(sourceDb.Has(&has, key.key));
            if (has)
            {
                ncm::ContentManagerAccessor accessor(&sourceDb, &sourceStorage);
                auto result = accessor.DeleteRedundant(key.key, latestKeyPointer);
                if (result.IsFailure())
                {
                    NN_DETAIL_NIM_TRACE("[ApplyDownloadedDeltaTask] Failed to delete as 0x%08x\n", result.GetInnerValueForDebug());
                }
            }
            commitList.Push(m_Meta.GetStorageId());
        }
    }
    for (int i = 0; i < commitList.Count(); ++i)
    {
        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, commitList[i]));
        NN_RESULT_DO(db.Commit());
    }

    NN_RESULT_SUCCESS;
}
Result ApplyDownloadedDeltaTask::Cancel() NN_NOEXCEPT
{
    m_CancelToken = true;
    if (m_IsTaskInitialized)
    {
        NN_RESULT_DO(m_ApplyTask.Cancel());
    }
    NN_RESULT_SUCCESS;
}
Result ApplyDownloadedDeltaTask::ResetCancel() NN_NOEXCEPT
{
    m_CancelToken = false;
    NN_RESULT_SUCCESS;
}
Result ApplyDownloadedDeltaTask::ClearNotEnoughSpaceState() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Data.GetState() == ApplyDownloadedDeltaProgressState::NotEnoughSpace, ResultApplyDeltaInvalidState());
    if (m_Data.IsFirstApply() && !m_IsTaskInitialized && !m_Data.IsTaskStateAvailable())
    {
        // InitialPrepare で失敗した場合なので、再度 InitialPrepare を行う
        NN_RESULT_DO(m_Data.SetState(ApplyDownloadedDeltaProgressState::NotRunning));
        NN_RESULT_DO(InitialPrepare());
    }
    else
    {
        NN_RESULT_DO(m_Data.SetState(ApplyDownloadedDeltaProgressState::Suspended));
    }

    NN_RESULT_SUCCESS;
}
Result ApplyDownloadedDeltaTask::SetBuffer(void* buffer, size_t size) NN_NOEXCEPT
{
    NN_RESULT_DO(m_ApplyTask.SetBuffer(buffer, size));

    NN_RESULT_SUCCESS;
}
ncm::ApplicationId ApplyDownloadedDeltaTask::GetApplicationId() NN_NOEXCEPT
{
    return m_Meta.GetApplicationId();
}
Result ApplyDownloadedDeltaTask::ListContentMetaKey(int* outCount, ncm::StorageContentMetaKey* outList, int count, int offset) NN_NOEXCEPT
{
    int c = 0;
    for (int i = 0; i < count && offset + i < m_Meta.CountContentMeta(); ++i)
    {
        NN_RESULT_DO(m_Meta.GetContentMeta(&(outList[i]), offset + i));
        c++;
    }
    *outCount = c;

    NN_RESULT_SUCCESS;
}

Result ApplyDownloadedDeltaTask::MoveNext(bool* outComplete) NN_NOEXCEPT
{
    NN_RESULT_DO(m_Data.MoveNext(outComplete));
    m_IsTaskInitialized = false;
    NN_RESULT_SUCCESS;
}

Result ApplyDownloadedDeltaTask::ThrowIfCancelRequested() NN_NOEXCEPT
{
    if (m_CancelToken)
    {
        NN_RESULT_DO(m_Data.SetState(ApplyDownloadedDeltaProgressState::Suspended));
        NN_RESULT_THROW(ResultApplyDeltaCancelled());
    }
    NN_RESULT_SUCCESS;
}

}}}

