﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/nifm_ApiNetworkProfile.h>

#include <nn/nifm/detail/nifm_TemporaryNetworkProfileClient.h>

#include <nn/nifm/detail/nifm_CommonDetail.h>
#include <nn/nifm/detail/util/nifm_SfUtility.h>


namespace nn
{
namespace nifm
{

namespace detail
{
    nn::Result GetGeneralServicePointer(nn::sf::SharedPointer<detail::IGeneralService>* ppGeneralService) NN_NOEXCEPT;
}

nn::Result GetCurrentNetworkProfile( NetworkProfileData *pOutNetworkProfileData ) NN_NOEXCEPT
{
    nn::sf::SharedPointer<detail::IGeneralService> pGeneralService;

    NN_RESULT_DO(detail::GetGeneralServicePointer(&pGeneralService));

    detail::sf::NetworkProfileData profile;
    NN_RESULT_DO(pGeneralService->GetCurrentNetworkProfile( &profile ));

    detail::ConvertNetworkProfileDataToNifmFromSf( pOutNetworkProfileData, profile );

    NN_RESULT_SUCCESS;
}

nn::Result GetCurrentAccessPoint( AccessPointData* pOutAccessPointData ) NN_NOEXCEPT
{
    nn::sf::SharedPointer<detail::IGeneralService> pGeneralService;

    NN_RESULT_DO(detail::GetGeneralServicePointer(&pGeneralService));

    detail::sf::AccessPointData accessPoint;
    NN_RESULT_DO(pGeneralService->GetCurrentAccessPoint( &accessPoint ));

    detail::ConvertAccessPointDataToNifmFromSf( pOutAccessPointData, accessPoint );

    NN_RESULT_SUCCESS;
}

nn::Result EnumerateNetworkInterfaces( NetworkInterfaceInfo* pOutNetworkInterfaceInfo, int* pOutCount, int size, nn::Bit32 filterFlags ) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(NetworkInterfaceInfo) >= sizeof(detail::sf::NetworkInterfaceInfo));
    NN_SDK_REQUIRES_NOT_NULL(pOutNetworkInterfaceInfo);

    nn::sf::SharedPointer<detail::IGeneralService> pGeneralService;

    NN_RESULT_DO(detail::GetGeneralServicePointer(&pGeneralService));

    int interfaceCount = 0;
    auto pOutMemory = reinterpret_cast<detail::sf::NetworkInterfaceInfo*>(pOutNetworkInterfaceInfo);

    NN_RESULT_DO(pGeneralService->EnumerateNetworkInterfaces(
        nn::sf::OutArray<nn::nifm::detail::sf::NetworkInterfaceInfo>(pOutMemory, size),
        &interfaceCount,
        filterFlags
    ));

    *pOutCount = interfaceCount;
    int copyCount = std::min<int>( size, interfaceCount );

    for (int i = copyCount - 1; i >= 0; --i)
    {
        detail::sf::NetworkInterfaceInfo networkInterfaceInfoTemp = pOutMemory[i];
        detail::ConvertNetworkInterfaceInfoToNifmFromSf(&pOutNetworkInterfaceInfo[i], networkInterfaceInfoTemp);
    }

    NN_RESULT_SUCCESS;
}

nn::Result EnumerateNetworkProfiles(NetworkProfileBasicInfo* pOutNetworkProfileBasicInfo, int* pOutCount, int size, nn::Bit8 typeFlags) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(NetworkProfileBasicInfo) >= sizeof(detail::sf::NetworkProfileBasicInfo));
    NN_SDK_REQUIRES_NOT_NULL(pOutNetworkProfileBasicInfo);

    nn::sf::SharedPointer<detail::IGeneralService> pGeneralService;

    NN_RESULT_DO(detail::GetGeneralServicePointer(&pGeneralService));

    int basicInfoCount = 0;
    auto pOutMemory = reinterpret_cast<detail::sf::NetworkProfileBasicInfo*>(pOutNetworkProfileBasicInfo);

    NN_RESULT_DO(pGeneralService->EnumerateNetworkProfiles(
        nn::sf::OutArray<nn::nifm::detail::sf::NetworkProfileBasicInfo>(pOutMemory, size),
        &basicInfoCount,
        typeFlags));

    *pOutCount = basicInfoCount;
    int copyCount = std::min<int>(size, basicInfoCount);

    for (int i = copyCount - 1; i >= 0; --i)
    {
        detail::sf::NetworkProfileBasicInfo networkProfileBasicInfoTemp = pOutMemory[i];
        detail::ConvertNetworkProfileBasicInfoToNifmFromSf(&pOutNetworkProfileBasicInfo[i], networkProfileBasicInfoTemp);
    }

    NN_RESULT_SUCCESS;
}

nn::Result GetNetworkProfile( NetworkProfileData* pOutNetworkProfileData, const nn::util::Uuid& id ) NN_NOEXCEPT
{
    nn::sf::SharedPointer<detail::IGeneralService> pGeneralService;

    NN_RESULT_DO(detail::GetGeneralServicePointer(&pGeneralService));

    detail::sf::NetworkProfileData profile;
    NN_RESULT_DO(pGeneralService->GetNetworkProfile( &profile, id ));

    detail::ConvertNetworkProfileDataToNifmFromSf( pOutNetworkProfileData, profile );

    NN_RESULT_SUCCESS;
}

nn::Result SetNetworkProfile( nn::util::Uuid* pOutId, const NetworkProfileData& networkProfileData ) NN_NOEXCEPT
{
    nn::sf::SharedPointer<detail::IGeneralService> pGeneralService;

    NN_RESULT_DO(detail::GetGeneralServicePointer(&pGeneralService));

    detail::sf::NetworkProfileData profile;
    detail::ConvertNetworkProfileDataToSfFromNifm( &profile, networkProfileData );

    NN_RESULT_THROW(pGeneralService->SetNetworkProfile( pOutId, profile ));
}

nn::Result RemoveNetworkProfile( const nn::util::Uuid& id ) NN_NOEXCEPT
{
    nn::sf::SharedPointer<detail::IGeneralService> pGeneralService;

    NN_RESULT_DO(detail::GetGeneralServicePointer(&pGeneralService));

    NN_RESULT_THROW(pGeneralService->RemoveNetworkProfile( id ));
}

nn::Result GetSsidListVersion(SsidListVersion* pOutSsidListVersion) NN_NOEXCEPT
{
    nn::sf::SharedPointer<detail::IGeneralService> pGeneralService;

    NN_RESULT_DO(detail::GetGeneralServicePointer(&pGeneralService));

    NN_RESULT_THROW(pGeneralService->GetSsidListVersion(pOutSsidListVersion));
}

nn::Result PersistTemporaryNetworkProfile(NetworkProfileHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle._value != nullptr, "handle is invalid.\n");

    NN_RESULT_THROW(reinterpret_cast<detail::TemporaryNetworkProfileClient*>(handle._value)->Persist());
}

nn::Result GetCurrentAccessPointChannel(uint16_t* pOutChannel) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutChannel);

    AccessPointData accessPointData;

    NN_RESULT_DO(GetCurrentAccessPoint(&accessPointData));

    *pOutChannel = accessPointData.channel;

    NN_RESULT_SUCCESS;
}

}
}

