﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/nifm_ApiHandleNetworkRequestResult.h>

#include <nn/nifm/detail/nifm_RequestClient.h>

#include <nn/la/la_AppletToNifmArgumentsReader.h>

#include <nn/util/util_ScopeExit.h>

namespace nn
{
namespace nifm
{
    nn::Result PrepareHandlingNetworkRequestResult(RequestHandle handle, nn::applet::LibraryAppletHandle* pOutLibraryAppletHandle) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(handle._value != nullptr, "handle is invalid.\n");

        auto requestResult = reinterpret_cast<detail::RequestClient*>(handle._value)->GetResult();

        NN_UNUSED(requestResult);
        NN_SDK_REQUIRES(requestResult.IsFailure());

        auto appletResult = reinterpret_cast<detail::RequestClient*>(handle._value)->PrepareLibraryApplet(pOutLibraryAppletHandle);

        if (appletResult.IsFailure())
        {
            // Windows 版など、アプレットが起動できない場合にはログにエラーコードを出力
            NN_DETAIL_NIFM_TRACE_V3("[nifm] Network request is rejected: 2%03u-%04u\n", requestResult.GetModule(), requestResult.GetDescription());
        }

        NN_RESULT_THROW(appletResult);
    }

    nn::Result StartHandlingNetworkRequestResult(const nn::applet::LibraryAppletHandle& libraryAppletHandle, void* pUserArgument) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)

        NN_RESULT_THROW_UNLESS(libraryAppletHandle != nn::applet::InvalidLibraryAppletHandle, ResultLibraryAppletInvalidHandle());

        nn::Result result = nn::applet::StartLibraryApplet(libraryAppletHandle, pUserArgument);

        if (result.IsFailure())
        {
            nn::applet::CloseLibraryApplet(libraryAppletHandle);
        }

        NN_RESULT_THROW(result);

#else
        NN_UNUSED(libraryAppletHandle);
        NN_UNUSED(pUserArgument);
        NN_RESULT_THROW(ResultNotImplemented());
#endif
    }

    nn::Result FinishHandlingNetworkRequestResult(const nn::applet::LibraryAppletHandle& libraryAppletHandle) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)

        NN_RESULT_THROW_UNLESS(libraryAppletHandle != nn::applet::InvalidLibraryAppletHandle, ResultLibraryAppletInvalidHandle());

        NN_UTIL_SCOPE_EXIT
        {
            nn::applet::CloseLibraryApplet(libraryAppletHandle);
        };

        nn::applet::JoinLibraryApplet(libraryAppletHandle);
        nn::applet::LibraryAppletExitReason exitReason = nn::applet::GetLibraryAppletExitReason(libraryAppletHandle);

        NN_ABORT_UNLESS_NOT_EQUAL(nn::applet::LibraryAppletExitReason_Abnormal, exitReason);

        switch (exitReason)
        {
        case nn::applet::LibraryAppletExitReason_Normal:
            break;
        case nn::applet::LibraryAppletExitReason_Canceled:
            break;
        case nn::applet::LibraryAppletExitReason_Unexpected:
            NN_FALL_THROUGH;
        default:
            NN_RESULT_THROW(ResultLibraryAppletExitUnexpectedly());
        }

        // アプレットが正常終了した場合の応答を取得
        nn::la::AppletToNifmArgumentsReader appletToNifmArgumentsReader;
        NN_RESULT_THROW_UNLESS(appletToNifmArgumentsReader.TryPopFromOutChannel(libraryAppletHandle), ResultErrorHandlingFailure());
        NN_RESULT_THROW(appletToNifmArgumentsReader.GetResult());
#else
        NN_UNUSED(libraryAppletHandle);
        NN_RESULT_THROW(ResultNotImplemented());
#endif
    }

    nn::Result GetResultHandlingNetworkRequestResult(const nn::applet::StorageHandle& storageHandle) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        // アプレットが正常終了した場合の応答を取得
        nn::la::AppletToNifmArgumentsReader appletToNifmArgumentsReader;
        NN_RESULT_THROW_UNLESS(appletToNifmArgumentsReader.ReadFromStorage(storageHandle), ResultErrorHandlingFailure());
        NN_RESULT_THROW(appletToNifmArgumentsReader.GetResult());
#else
        NN_UNUSED(storageHandle);
        NN_RESULT_THROW(ResultNotImplemented());
#endif
    }

}
}
