﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/util/util_StringUtil.h>
#include <nn/nifm/detail/util/nifm_WlanUtility.h>

namespace nn
{
namespace nifm
{
namespace detail
{
nn::Result ConvertConnectionStatusToResult(const nn::wlan::ConnectionStatus& connectionStatus) NN_NOEXCEPT
{
    switch( connectionStatus.state )
    {
    case nn::wlan::ConnectionState_Connected:
        NN_RESULT_SUCCESS;

    case nn::wlan::ConnectionState_Disconnected:
    case nn::wlan::ConnectionState_Idle:
        NN_DETAIL_NIFM_INFO("GetConnectionStatus.cause = %d\n", connectionStatus.cause);
        switch( connectionStatus.cause )
        {
        // Association 失敗
        case nn::wlan::CauseOfInfo_NoBss:             //!< NO_NETWORK_AVAIL  所望の BSS が見つからない。
            return nn::nifm::ResultBssNotFound();
        case nn::wlan::CauseOfInfo_ConnectFailure:    //!< AUTH_FAILED/ASSOC_FAILED/NO_RESOURCES_AVAIL  Connect 試行の失敗。 StatusCodeが入る。
            switch (connectionStatus.statusReasonCode)
            {
            case nn::wlan::Dot11StatusCode_Success: // 0
                NN_RESULT_THROW(ResultConnectionFailedWithStatusCodeUnexpected());
            case nn::wlan::Dot11StatusCode_UnspecifiedFailure:  // 1
                NN_RESULT_THROW(ResultConnectionFailedWithStatusCodeUnspecifiedFailure());
            case nn::wlan::Dot11StatusCode_UnsupportedCapabilities: // 10
                NN_RESULT_THROW(ResultConnectionFailedWithStatusCodeUnsupportedCapabilities());
            case nn::wlan::Dot11StatusCode_AssocDeniedUnspec:   // 12
                NN_RESULT_THROW(ResultConnectionFailedWithStatusCodeAssocDeniedUnspec());
            case nn::wlan::Dot11StatusCode_UnsupportedAuthAlgorithm:    // 13
                NN_RESULT_THROW(ResultConnectionFailedWithStatusCodeUnsupportedAuthAlgorithm());
            case nn::wlan::Dot11StatusCode_AuthSequenceError:   // 14
                NN_RESULT_THROW(ResultConnectionFailedWithStatusCodeAuthSequenceError());
            case nn::wlan::Dot11StatusCode_AuthChallengeFailed: // 15
                NN_RESULT_THROW(ResultConnectionFailedWithStatusCodeAuthChallengeFailed());
            case nn::wlan::Dot11StatusCode_AuthFrameTimeout:    // 16
                NN_RESULT_THROW(ResultConnectionFailedWithStatusCodeAuthFrameTimeout());
            case nn::wlan::Dot11StatusCode_AssocDeniedNoResource:   // 17
                NN_RESULT_THROW(ResultConnectionFailedWithStatusCodeAssocDeniedNoResource());
            case nn::wlan::Dot11StatusCode_AssocDeniedUnsupportedRate:  // 18
                NN_RESULT_THROW(ResultConnectionFailedWithStatusCodeAssocDeniedUnsupportedRate());
            default:
                NN_DETAIL_NIFM_WARN_V3("connectionStatus::(cause,statusReasonCode)=%d,%d\n", connectionStatus.cause, connectionStatus.statusReasonCode);
                NN_RESULT_THROW(ResultConnectionFailedWithStatusCode());
            }

        // Association 後の 4-way Handshake などで失敗
        case nn::wlan::CauseOfInfo_ConnectFailureAfterAssoc:    //!< Infra専用。WPAでのAssociation後の認証フェーズでの失敗。 ReasonCodeが入る。
            switch (connectionStatus.statusReasonCode)
            {
            // case nn::wlan::Dot11ReasonCode_Reserved:
            case nn::wlan::Dot11ReasonCode_UnspecifiedFailure:  // 1
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCodeUnspecifiedFailure());
            case nn::wlan::Dot11ReasonCode_AuthInvalid: // 2
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCodeAuthInvalid());
            case nn::wlan::Dot11ReasonCode_DeauthStaLeave:  // 3
                // WPA サプリカントでタイムアウトが発生して自発的に切断をおこなう場合に
                // サプリカントが入れることがある
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCodeDeauthStaLeave());
            case nn::wlan::Dot11ReasonCode_DisassocInactivity:  // 4
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCodeDisassocInactivity());
            case nn::wlan::Dot11ReasonCode_DisassocUnableToHandle:  // 5
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCodeDisassocUnableToHandle());
            case nn::wlan::Dot11ReasonCode_NonauthClass2Frame:  // 6
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCodeNonauthClass2Frame());
            case nn::wlan::Dot11ReasonCode_NonAssocClass3Frame: // 7
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCodeNonAssocClass3Frame());
            case nn::wlan::Dot11ReasonCode_DisassocStaLeaveLEV: // 8
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCodeDisassocStaLeaveLEV());
            case nn::wlan::Dot11ReasonCode_AssocReqWithoutAuth: // 9
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCodeAssocReqWithoutAuth());
            case nn::wlan::Dot11ReasonCode_DisassocPowerCapability: // 10
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCodeDisassocPowerCapability());
            case nn::wlan::Dot11ReasonCode_DisassocUnacceptableSupportedCh: // 11
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCodeDisassocUnacceptableSupportedCh());
            case nn::wlan::Dot11ReasonCode_InvalidIe:   // 13
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCodeInvalidIe());
            case nn::wlan::Dot11ReasonCode_MicError:    // 14
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCodeMicError());
            case nn::wlan::Dot11ReasonCode_4WayHandshakeTimeout:  // 15
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCode4WayHandshakeTimeout());
            case nn::wlan::Dot11ReasonCode_GroupKeyHandshakeTimeout:    // 16
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCodeGroupKeyHandshakeTimeout());
            case nn::wlan::Dot11ReasonCode_4WayIeMismatch:  // 17
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCode4WayIeMismatch());
            case nn::wlan::Dot11ReasonCode_InvalidGroupCipher:  // 18
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCodeInvalidGroupCipher());
            case nn::wlan::Dot11ReasonCode_InvalidPairwiseCipher:   // 19
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCodeInvalidPairwiseCipher());
            case nn::wlan::Dot11ReasonCode_InvalidAkmp: // 20
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCodeInvalidAkmp());
            case nn::wlan::Dot11ReasonCode_UnsupportedRsnIeVersion: // 21
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCodeUnsupportedRsnIeVersion());
            case nn::wlan::Dot11ReasonCode_InvalidRsnIeCapability:  // 22
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCodeInvalidRsnIeCapability());
            case nn::wlan::Dot11ReasonCode_8021xAuthFailed: // 23
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCode8021xAuthFailed());
            case nn::wlan::Dot11ReasonCode_CipherSuiteRejected: // 24
                NN_RESULT_THROW(ResultConnectionFailedWithReasonCodeCipherSuiteRejected());
            default:
                NN_DETAIL_NIFM_WARN_V3("connectionStatus::(cause,statusReasonCode)=%d,%d\n", connectionStatus.cause, connectionStatus.statusReasonCode);
                NN_RESULT_THROW(nn::nifm::ResultConnectionFailedWithReasonCode());
            }

        // 能動的切断
        case nn::wlan::CauseOfInfo_DisconnectReq:     //!< DISCONNECT_CMD    コマンド要求による切断。 Status(0:成功)  MasterではReasonCodeが入る。
            NN_RESULT_THROW(nn::nifm::ResultDisconnected());

        // 受動的切断
        case nn::wlan::CauseOfInfo_BeaconLost:        //!< LOST_LINK         Beacon Lost。
            NN_RESULT_THROW(nn::nifm::ResultBeaconLost());
        case nn::wlan::CauseOfInfo_RecieveDisconnect: //!< BSS_DISCONNECTED  Deauth による切断。  Reasonが入る。
            switch (connectionStatus.statusReasonCode)
            {
            // case nn::wlan::Dot11ReasonCode_Reserved:
            case nn::wlan::Dot11ReasonCode_UnspecifiedFailure:
                NN_RESULT_THROW(ResultDisconnectedWithReasonCodeUnspecifiedFailure());
            case nn::wlan::Dot11ReasonCode_AuthInvalid:
                NN_RESULT_THROW(ResultDisconnectedWithReasonCodeAuthInvalid());
            case nn::wlan::Dot11ReasonCode_DeauthStaLeave:
                NN_RESULT_THROW(ResultDisconnectedWithReasonCodeDeauthStaLeave());
            case nn::wlan::Dot11ReasonCode_DisassocInactivity:
                NN_RESULT_THROW(ResultDisconnectedWithReasonCodeDisassocInactivity());
            case nn::wlan::Dot11ReasonCode_DisassocUnableToHandle:
                NN_RESULT_THROW(ResultDisconnectedWithReasonCodeDisassocUnableToHandle());
            case nn::wlan::Dot11ReasonCode_NonauthClass2Frame:
                NN_RESULT_THROW(ResultDisconnectedWithReasonCodeNonauthClass2Frame());
            case nn::wlan::Dot11ReasonCode_NonAssocClass3Frame:
                NN_RESULT_THROW(ResultDisconnectedWithReasonCodeNonAssocClass3Frame());
            case nn::wlan::Dot11ReasonCode_DisassocStaLeaveLEV:
                NN_RESULT_THROW(ResultDisconnectedWithReasonCodeDisassocStaLeaveLEV());
            case nn::wlan::Dot11ReasonCode_AssocReqWithoutAuth:
                NN_RESULT_THROW(ResultDisconnectedWithReasonCodeAssocReqWithoutAuth());
            case nn::wlan::Dot11ReasonCode_DisassocPowerCapability:
                NN_RESULT_THROW(ResultDisconnectedWithReasonCodeDisassocPowerCapability());
            case nn::wlan::Dot11ReasonCode_DisassocUnacceptableSupportedCh:
                NN_RESULT_THROW(ResultDisconnectedWithReasonCodeDisassocUnacceptableSupportedCh());
            case nn::wlan::Dot11ReasonCode_InvalidIe:
                NN_RESULT_THROW(ResultDisconnectedWithReasonCodeInvalidIe());
            case nn::wlan::Dot11ReasonCode_MicError:
                NN_RESULT_THROW(ResultDisconnectedWithReasonCodeMicError());
            case nn::wlan::Dot11ReasonCode_4WayHandshakeTimeout:  // 15
                NN_RESULT_THROW(ResultDisconnectedWithReasonCode4WayHandshakeTimeout());
            case nn::wlan::Dot11ReasonCode_GroupKeyHandshakeTimeout:    // 16
                NN_RESULT_THROW(ResultDisconnectedWithReasonCodeGroupKeyHandshakeTimeout());
            case nn::wlan::Dot11ReasonCode_4WayIeMismatch:  // 17
                NN_RESULT_THROW(ResultDisconnectedWithReasonCode4WayIeMismatch());
            case nn::wlan::Dot11ReasonCode_InvalidGroupCipher:  // 18
                NN_RESULT_THROW(ResultDisconnectedWithReasonCodeInvalidGroupCipher());
            case nn::wlan::Dot11ReasonCode_InvalidPairwiseCipher:   // 19
                NN_RESULT_THROW(ResultDisconnectedWithReasonCodeInvalidPairwiseCipher());
            case nn::wlan::Dot11ReasonCode_InvalidAkmp: // 20
                NN_RESULT_THROW(ResultDisconnectedWithReasonCodeInvalidAkmp());
            case nn::wlan::Dot11ReasonCode_UnsupportedRsnIeVersion: // 21
                NN_RESULT_THROW(ResultDisconnectedWithReasonCodeUnsupportedRsnIeVersion());
            case nn::wlan::Dot11ReasonCode_InvalidRsnIeCapability:  // 22
                NN_RESULT_THROW(ResultDisconnectedWithReasonCodeInvalidRsnIeCapability());
            case nn::wlan::Dot11ReasonCode_8021xAuthFailed: // 23
                NN_RESULT_THROW(ResultDisconnectedWithReasonCode8021xAuthFailed());
            case nn::wlan::Dot11ReasonCode_CipherSuiteRejected: // 24
                NN_RESULT_THROW(ResultDisconnectedWithReasonCodeCipherSuiteRejected());
            default:
                NN_DETAIL_NIFM_WARN_V3("connectionStatus::(cause,statusReasonCode)=%d,%d\n", connectionStatus.cause, connectionStatus.statusReasonCode);
                NN_RESULT_THROW(nn::nifm::ResultDisconnectedWithReasonCode());
            }

        // 以下、インフラモードでは返らない想定の値
        case nn::wlan::CauseOfInfo_NoInfo:            //!< 原因なし。
        case nn::wlan::CauseOfInfo_ConnectRequest:    //!< Client時：Connect コマンド要求により試行。Master時：Clientからの接続試行。結果、接続完了。
        case nn::wlan::CauseOfInfo_JoinRequest:       //!< Join コマンド要求により試行。結果、Join 完了。
        case nn::wlan::CauseOfInfo_JoinFailure:       //!< Join 試行の失敗。
        case nn::wlan::CauseOfInfo_ReceiveBcDeauth:   //!< Spectator 時に Broadcast Deauthentication フレームを受信。
        case nn::wlan::CauseOfInfo_DecryptionError:   //!< 接続は維持されているが、暗号通信に失敗。
        case nn::wlan::CauseOfInfo_Unknown:           //!< CSERV_DISCONNECT/INVALID_PROFILE/DOT11H_CHANNEL_SWITCH/PROFILE_MISMATCH/ADHOC_MERGE
        default:
            NN_DETAIL_NIFM_ERROR_V3("connectionStatus::(cause,statusReasonCode)=%d,%d\n", connectionStatus.cause, connectionStatus.statusReasonCode);
            NN_RESULT_THROW(nn::nifm::ResultLinkLayerNotAvailable());    // TODO
        }
        break;

    case nn::wlan::ConnectionState_Joined:
    default:
        NN_RESULT_THROW(nn::nifm::ResultInternalError());   // TODO
    }
} // NOLINT(impl/function_size)

nn::wlan::SecurityMode ConvertNifmAuthEncryptionToWpaCipherSuiteType(nn::nifm::Authentication auth, nn::nifm::Encryption enc, int wepKeyLength) NN_NOEXCEPT
{
    if( auth == Authentication_Open )
    {
        if( enc == Encryption_None )
        {
            return nn::wlan::SecurityMode_Open;
        }
        else if( enc == Encryption_Wep && (wepKeyLength == Wep40KeyLength || wepKeyLength == Wep40KeyLengthAsHex) )
        {
            return nn::wlan::SecurityMode_Wep64Open;
        }
        else if( enc == Encryption_Wep && (wepKeyLength == Wep104KeyLength || wepKeyLength == Wep104KeyLengthAsHex) )
        {
            return nn::wlan::SecurityMode_Wep128Open;
        }
    }
    else if( auth == Authentication_Shared )
    {
        if( enc == Encryption_Wep && (wepKeyLength == Wep40KeyLength || wepKeyLength == Wep40KeyLengthAsHex) )
        {
            return nn::wlan::SecurityMode_Wep64Shared;
        }
        else if( enc == Encryption_Wep && (wepKeyLength == Wep104KeyLength || wepKeyLength == Wep104KeyLengthAsHex) )
        {
            return nn::wlan::SecurityMode_Wep128Shared;
        }
    }
    else if( auth == Authentication_WpaPsk )
    {
        if( enc == Encryption_Tkip )
        {
            return nn::wlan::SecurityMode_WpaTkip;
        }
        else if( enc == Encryption_Aes )
        {
            return nn::wlan::SecurityMode_WpaAes;
        }
    }
    else if( auth == Authentication_Wpa2Psk )
    {
        if( enc == Encryption_Tkip )
        {
            return nn::wlan::SecurityMode_Wpa2Tkip;
        }
        else if( enc == Encryption_Aes )
        {
            return nn::wlan::SecurityMode_Wpa2Aes;
        }
    }

    NN_DETAIL_NIFM_INFO("Unsupported authentication-encryption pair\n");
    return nn::wlan::SecurityMode_Open;
}

nn::nifm::Encryption ConvertWlanCipherSuiteTypeToEncryption(nn::wlan::WpaCipherSuiteType cipher) NN_NOEXCEPT
{
    switch( cipher )
    {
    case nn::wlan::WPA_CIPHER_WEP_40BIT:
    case nn::wlan::WPA_CIPHER_WEP_104BIT:
        return nn::nifm::Encryption_Wep;
    case nn::wlan::WPA_CIPHER_TKIP:
        return nn::nifm::Encryption_Tkip;
    case nn::wlan::WPA_CIPHER_AES_CCMP:
        return nn::nifm::Encryption_Aes;
    case nn::wlan::WPA_CIPHER_NONE:
        return nn::nifm::Encryption_None;
    case nn::wlan::WPA_CIPHER_AES_OCB:
    case nn::wlan::WPA_CIPHER_USE_GROUPKEY:
    default:
        return nn::nifm::Encryption_Invalid;
    }
}

int GetWpaCipherSuiteTypePriority( nn::wlan::WpaCipherSuiteType keyType ) NN_NOEXCEPT
{
    nn::wlan::WpaCipherSuiteType wpaCipherSuiteTypePriorityArray[] =
    {
        nn::wlan::WPA_CIPHER_AES_CCMP,
        nn::wlan::WPA_CIPHER_TKIP,
        nn::wlan::WPA_CIPHER_WEP_104BIT,
        nn::wlan::WPA_CIPHER_WEP_40BIT,
        nn::wlan::WPA_CIPHER_NONE,
    };

    int count = static_cast<int>(NN_ARRAY_SIZE( wpaCipherSuiteTypePriorityArray ));

    for( int i = 0; i < count; ++i )
    {
        if( keyType == wpaCipherSuiteTypePriorityArray[i] )
        {
            return i;
        }
    }

    return count;
}

void ConvertToAsciiCodeString( char* outBuffer, int outBufferSize, const char* inBuffer, int inBufferSize ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( outBufferSize >= (2 * inBufferSize - 1) );

    int count = std::min((outBufferSize - 1) / 2, inBufferSize);

    for( int i = 0; i < count; ++i )
    {
        nn::util::SNPrintf( &outBuffer[i * 2], 3, "%02x", inBuffer[i] );
    }
    outBuffer[count * 2] = '\0';
}

void ReadAccessPointFromBeaconDescriptionReader(WirelessAccessPoint* pWirelessAccessPoint, const nn::wlan::BeaconDescriptionReader& descriptionReader) NN_NOEXCEPT
{
    pWirelessAccessPoint->SetSsid(descriptionReader.GetSsid().GetSsidData(), descriptionReader.GetSsid().GetLength());
    pWirelessAccessPoint->SetBssid(*reinterpret_cast<nn::nifm::MacAddress*>(descriptionReader.GetBssid().GetMacAddressData()));
    pWirelessAccessPoint->SetRssi(descriptionReader.GetRssi());
    pWirelessAccessPoint->SetLinkLevel(static_cast<nn::nifm::LinkLevel>(descriptionReader.GetLinkLevel()));
    pWirelessAccessPoint->SetChannel(descriptionReader.GetChannel());

    if (descriptionReader.IsWpa2Supported() || descriptionReader.IsWpaSupported())
    {
        if (descriptionReader.IsWpa2Supported())
        {
            auto ie = descriptionReader.GetWpa2Ie();
            auto groupCipher = Encryption_None;
            switch (ie->GetGroupCipherSuite())
            {
            case nn::wlan::WPA_CIPHER_TKIP:
                groupCipher = Encryption_Tkip;
                break;
            case nn::wlan::WPA_CIPHER_AES_CCMP:
                groupCipher = Encryption_Aes;
                break;
            default:
                NN_DETAIL_NIFM_WARN("Invalid Group Cipher(%d) / WPA2-PSK\n", ie->GetGroupCipherSuite());
                break;
            }

            auto akmCount = ie->GetAuthKeyManagementSuiteCount();
            auto pcsCount = ie->GetPairwiseCipherSuiteCount();

            for (int i = 0; i < akmCount; ++i)
            {
                auto authKeyManagement = Authentication_Invalid;
                switch (ie->GetAuthKeyManagementSuite(i))
                {
                case nn::wlan::AKM_WPA_PSK:
                    authKeyManagement = Authentication_Wpa2Psk;
                    break;
                case nn::wlan::AKM_WPA_EAP:
                    authKeyManagement = Authentication_Wpa2;
                    break;
                default:
                    continue;
                }

                for (int j = 0; j < pcsCount; ++j)
                {
                    auto pairwiseCipher = Encryption_None;
                    switch (ie->GetPairwiseCipherSuite(j))
                    {
                    case nn::wlan::WPA_CIPHER_TKIP:
                        pairwiseCipher = Encryption_Tkip;
                        break;
                    case nn::wlan::WPA_CIPHER_AES_CCMP:
                        pairwiseCipher = Encryption_Aes;
                        break;
                    case nn::wlan::WPA_CIPHER_USE_GROUPKEY:
                        pairwiseCipher = groupCipher;
                        break;
                    default:
                        continue;
                    }
                    pWirelessAccessPoint->SetWpaAuthenticationAndEncryption(authKeyManagement, pairwiseCipher, groupCipher);
                }
            }
        }

        if (descriptionReader.IsWpaSupported())
        {
            auto ie = descriptionReader.GetWpaIe();
            auto groupCipher = Encryption_None;
            switch (ie->GetGroupCipherSuite())
            {
            case nn::wlan::WPA_CIPHER_TKIP:
                groupCipher = Encryption_Tkip;
                break;
            case nn::wlan::WPA_CIPHER_AES_CCMP:
                groupCipher = Encryption_Aes;
                break;
            default:
                NN_DETAIL_NIFM_WARN("Invalid Group Cipher(%d) / WPA-PSK\n", ie->GetGroupCipherSuite());
                break;
            }

            auto akmCount = ie->GetAuthKeyManagementSuiteCount();
            auto pcsCount = ie->GetPairwiseCipherSuiteCount();

            for (int i = 0; i < akmCount; ++i)
            {
                auto authKeyManagement = Authentication_Invalid;
                switch (ie->GetAuthKeyManagementSuite(i))
                {
                case nn::wlan::AKM_WPA_PSK:
                    authKeyManagement = Authentication_WpaPsk;
                    break;
                case nn::wlan::AKM_WPA_EAP:
                    authKeyManagement = Authentication_Wpa;
                    break;
                default:
                    continue;
                }

                for (int j = 0; j < pcsCount; ++j)
                {
                    auto pairwiseCipher = Encryption_None;
                    switch (ie->GetPairwiseCipherSuite(j))
                    {
                    case nn::wlan::WPA_CIPHER_TKIP:
                        pairwiseCipher = Encryption_Tkip;
                        break;
                    case nn::wlan::WPA_CIPHER_AES_CCMP:
                        pairwiseCipher = Encryption_Aes;
                        break;
                    case nn::wlan::WPA_CIPHER_USE_GROUPKEY:
                        pairwiseCipher = groupCipher;
                        break;
                    default:
                        continue;
                    }
                    pWirelessAccessPoint->SetWpaAuthenticationAndEncryption(authKeyManagement, pairwiseCipher, groupCipher);
                }
            }
        }
    }
    else if ((descriptionReader.GetCapabilityInformation() & 0x0010) == 0x0010) // Privacy bit
    {
        // ビーコンからは Open/Shared は判別できない
        pWirelessAccessPoint->SetWepAuthenticationAndEncryption(Authentication_Unknown, Encryption_Wep);
    }
    else
    {
        pWirelessAccessPoint->SetWepAuthenticationAndEncryption(Authentication_Open, Encryption_None);
    }

}  // NOLINT(impl/function_size)

void ConvertToWlanSsidFromNifm(nn::wlan::Ssid* pOutSsid, const Ssid& ssid) NN_NOEXCEPT
{
    bool ret = pOutSsid->Set(reinterpret_cast<const Bit8*>(ssid.hex), ssid.length);

    NN_UNUSED(ret);
    NN_SDK_ASSERT(ret);
}

void ConvertToWlanBssidFromNifm(nn::wlan::MacAddress* pOutBssid, const MacAddress& bssid) NN_NOEXCEPT
{
    pOutBssid->Set(bssid.data);
}

void ConvertToWlanSecurityFromNifm(nn::wlan::Security* pOutSecurity, const NetworkProfileData& networkProfile, const AccessPointData& accessPointData) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(NN_ARRAY_SIZE(pOutSecurity->key) >= SharedKey::KeyMaterialSize);

    nn::nifm::Authentication authentication = networkProfile.wirelessSetting.security.authEncryption.authentication;
    nn::nifm::Encryption encryption = networkProfile.wirelessSetting.security.authEncryption.encryption;
    nn::nifm::Encryption groupEncryption = accessPointData.groupEncryption;

    NN_DETAIL_NIFM_INFO("ConnectImpl: auth=%d, enc=%d, grp=%d\n", authentication, encryption, groupEncryption);

    int keyMaterialCount = networkProfile.wirelessSetting.security.sharedKey.length;

    pOutSecurity->privacyMode = ConvertNifmAuthEncryptionToWpaCipherSuiteType(authentication, encryption, keyMaterialCount);
    pOutSecurity->groupPrivacyMode = ConvertNifmAuthEncryptionToWpaCipherSuiteType(authentication, groupEncryption, keyMaterialCount);
    pOutSecurity->keyIdx = 0;

    const char* pKeyMaterial = networkProfile.wirelessSetting.security.sharedKey.keyMaterial;
    pOutSecurity->key[0] = '\0';

    if (encryption == Encryption_Wep)
    {
        if (keyMaterialCount == Wep40KeyLengthAsHex || keyMaterialCount == Wep104KeyLengthAsHex)
        {
            std::memcpy(pOutSecurity->key, pKeyMaterial, keyMaterialCount);
            pOutSecurity->key[keyMaterialCount] = '\0';
        }
        else if (keyMaterialCount == Wep40KeyLength || keyMaterialCount == Wep104KeyLength)
        {
            ConvertToAsciiCodeString(reinterpret_cast<char*>(pOutSecurity->key), NN_ARRAY_SIZE(pOutSecurity->key), pKeyMaterial, keyMaterialCount);
        }
    }
    else if (authentication == Authentication_WpaPsk || authentication == Authentication_Wpa2Psk)
    {
        if (NN_ARRAY_SIZE(pOutSecurity->key) > keyMaterialCount)
        {
            std::memcpy(pOutSecurity->key, pKeyMaterial, keyMaterialCount);
            pOutSecurity->key[keyMaterialCount] = '\0';
        }
    }
}

}
}
}
