﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/nifm_ScanRequestClient.h>


namespace nn
{
namespace nifm
{
namespace detail
{

nn::Result GetGeneralServicePointer(nn::sf::SharedPointer<detail::IGeneralService>* ppGeneralService) NN_NOEXCEPT;

ScanRequestClient::ScanRequestClient(nn::os::EventClearMode eventClearMode) NN_NOEXCEPT
    : m_pIScanRequest(nullptr)
{
    nn::Result result;

    nn::sf::SharedPointer<detail::IGeneralService> pGeneralService;
    result = GetGeneralServicePointer(&pGeneralService);

    NN_SDK_ASSERT(result.IsSuccess());
    if (result.IsFailure())
    {
        return;
    }

    nn::sf::SharedPointer<detail::IScanRequest> pIScanRequest;
    result = pGeneralService->CreateScanRequest(&pIScanRequest);

    NN_SDK_ASSERT(result.IsSuccess());
    if (result.IsFailure())
    {
        return;
    }

    nn::sf::NativeHandle systemEventReadableHandle;
    result = pIScanRequest->GetSystemEventReadableHandle(&systemEventReadableHandle);

    NN_SDK_ASSERT(result.IsSuccess());
    if (result.IsFailure())
    {
        return;
    }

    m_SystemEvent.AttachReadableHandle(systemEventReadableHandle.GetOsHandle(), systemEventReadableHandle.IsManaged(), eventClearMode);
    systemEventReadableHandle.Detach();

    m_pIScanRequest = pIScanRequest;
}

ScanRequestClient::~ScanRequestClient() NN_NOEXCEPT
{
}

void ScanRequestClient::Submit() NN_NOEXCEPT
{
    if (m_pIScanRequest != nullptr)
    {
        m_SystemEvent.Clear();
        m_pIScanRequest->Submit();
    }
}

bool ScanRequestClient::IsProcessing() NN_NOEXCEPT
{
    if (m_pIScanRequest == nullptr)
    {
        return false;
    }

    bool isProcessing;
    auto result = m_pIScanRequest->IsProcessing(&isProcessing);

    return result.IsSuccess() ? isProcessing : false;
}

nn::Result ScanRequestClient::GetResult() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_pIScanRequest != nullptr, ResultNotInitialized());

    return m_pIScanRequest->GetResult();
}

nn::os::SystemEvent& ScanRequestClient::GetSystemEvent() NN_NOEXCEPT
{
    return m_SystemEvent;
}

nn::Result ScanRequestClient::SetChannels(const int16_t scanChennels[], int count) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_pIScanRequest != nullptr, ResultNotInitialized());
    NN_RESULT_THROW_UNLESS(count >= 0, ResultInvalidArgument());

    NN_RESULT_THROW(m_pIScanRequest->SetChannels(nn::sf::InArray<int16_t>(scanChennels, count)));
}

}
}
}
