﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/core/nifm_ConnectionSelector.h>

#include <nn/nifm/detail/core/accessPoint/nifm_AccessPointBase.h>
#include <nn/nifm/detail/core/accessPoint/nifm_AccessPointList.h>
#include <nn/nifm/detail/core/networkInterface/nifm_NetworkInterfaceBase.h>
#include <nn/nifm/detail/core/networkInterface/nifm_WirelessInterface.h>

#include <nn/nifm/detail/util/nifm_CancelFlagHolder.h>
#include <nn/nifm/detail/util/nifm_DebugUtility.h>

#include <nn/nifm/nifm_TypesScan.h>

#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_LockGuard.h>
#include <nn/settings/system/settings_Network.h>


// TODO: [TORIAEZU]
// Result のモジュール番号取得のためのワークアラウンド
// Result のハンドリングが整備されたら外す
namespace nn { namespace eth {
#include <nn/eth/eth_Result.public.h>
}}
#include <nn/wlan/wlan_Result.h>
#include <nn/tsc/tsc_Result.h>


namespace nn
{
namespace nifm
{
namespace detail
{

namespace
{

// 統合された接続要求に応じて ConnectionSettings の試行優先度を判断するクラス
class IsPrior
{
private:
    static const NetworkInterfaceType s_InterfacePriorityOrdrer[2];
    static const NetworkProfileBase::ProfileType s_ProfileTypePriorityOrder[3];

public:

    static int GetPriority(NetworkProfileBase::ProfileType profileType) NN_NOEXCEPT
    {
        const int count = NN_ARRAY_SIZE(s_ProfileTypePriorityOrder);
        for( int i = 0; i < count; ++i )
        {
            if( s_ProfileTypePriorityOrder[i] == profileType )
            {
                return i;
            }
        }

        return count;
    }

    static int GetPriority(NetworkInterfaceType interfaceType) NN_NOEXCEPT
    {
        const int count = NN_ARRAY_SIZE(s_InterfacePriorityOrdrer);
        for( int i = 0; i < count; ++i )
        {
            if( s_InterfacePriorityOrdrer[i] == interfaceType )
            {
                return i;
            }
        }

        return count;
    }

    bool operator()(const ConnectionSettings* pLeft, const ConnectionSettings* pRight) NN_NOEXCEPT
    {
        // std::tie() < std::tie()
        auto lProfileType = GetPriority(pLeft->pNetworkProfile->GetType());
        auto lInterfaceType = GetPriority(pLeft->pNetworkProfile->GetNetworkInterfaceType());
        auto lIndex = pLeft->profileIndex;
        auto lAccessPointPriority = pLeft->pAccessPoint->GetPriority();
        auto rProfileType = GetPriority(pRight->pNetworkProfile->GetType());
        auto rInterfaceType = GetPriority(pRight->pNetworkProfile->GetNetworkInterfaceType());
        auto rIndex = pRight->profileIndex;
        auto rAccessPointPriority = pRight->pAccessPoint->GetPriority();

        if( lProfileType != rProfileType )
        {
            return lProfileType < rProfileType;
        }
        if( lInterfaceType != rInterfaceType )
        {
            return lInterfaceType < rInterfaceType;
        }
        if( lIndex != rIndex )
        {
            return lIndex > rIndex;
        }
        return lAccessPointPriority < rAccessPointPriority;
    }
};

const NetworkProfileBase::ProfileType IsPrior::s_ProfileTypePriorityOrder[] = { NetworkProfileBase::ProfileType::TemporaryProfile, NetworkProfileBase::ProfileType::UserProfile, NetworkProfileBase::ProfileType::SsidListProfile };
const NetworkInterfaceType IsPrior::s_InterfacePriorityOrdrer[] = { NetworkInterfaceType_Ethernet, NetworkInterfaceType_Ieee80211 };


// TODO: [TORIAEZU]
// Result のハンドリングが整備されたら外す
nn::Result WrapUpResultInResultRequestRejected(nn::Result result)
{
    int module = result.GetModule();

    if (module == nn::eth::ResultResetTimeout().GetModule())
    {
        NN_DETAIL_NIFM_WARN("An eth result is passed through: %08x\n", result.GetInnerValueForDebug());
        NN_RESULT_THROW(nn::nifm::ResultLinkLayerNotAvailable());
    }

    if (module == nn::wlan::ResultInvalidState().GetModule())
    {
        NN_DETAIL_NIFM_WARN("A wlan result is passed through: %08x\n", result.GetInnerValueForDebug());
        NN_RESULT_THROW(nn::nifm::ResultLinkLayerNotAvailable());
    }

    if( result <= nn::nifm::ResultRequestRejected() || result.IsSuccess())
    {
        NN_RESULT_THROW(result);
    }
    else
    {
        NN_DETAIL_NIFM_WARN("An unexpected nifm result is passed through: %08x\n", result.GetInnerValueForDebug());
        NN_RESULT_THROW(nn::nifm::ResultRequestRejected());
    }
}

bool UpdateResultIfPrior(nn::Result* pInOutResult, nn::Result result) NN_NOEXCEPT
{
    if (pInOutResult->IsSuccess())
    {
        return false;
    }

    if (result.IsSuccess())
    {
        *pInOutResult = result;
        return true;
    }

    result = WrapUpResultInResultRequestRejected(result);

    NN_SDK_ASSERT(*pInOutResult <= ResultRequestRejected());
    NN_SDK_ASSERT(result <= ResultRequestRejected());

    if (pInOutResult->GetDescription() < result.GetDescription())
    {
        *pInOutResult = result;
        return true;
    }

    return false;
}

} // namespace


ConnectionSelector::ConnectionSelector(NetworkProfileManager *pNetworkProfileManager) NN_NOEXCEPT
    : m_Mutex(),
      m_ScanCounter(InvalidScanCounter + 1),
      // m_ScanResult(),
      m_ScanInterval(nn::TimeSpan::FromSeconds(ScanIntervalInSeconds)),
      m_ScanTimerEvent(nn::os::EventClearMode_AutoClear),
      m_ScanTimerEventHandler(m_ScanTimerEvent),
      m_ScanTimerEventCallback(*this),
      m_pNetworkProfileManager(pNetworkProfileManager),
      m_IsAwake(true)
{
    m_ScanTimerEventHandler.Register(&m_ScanTimerEventCallback);

#if NN_DETAIL_NIFM_CONFIG_ENABLE_AUTO_SCAN
    m_ScanTimerEvent.Signal();
#endif
}

ConnectionSelector::~ConnectionSelector() NN_NOEXCEPT
{
}

nn::Result ConnectionSelector::ConfirmScanExecutable() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_IsAwake, ResultSuspended());

    TotalNetworkResourceInfo totalNetworkResourceInfo;
    GetTotalNetworkResourceInfo(&totalNetworkResourceInfo);

    switch (totalNetworkResourceInfo.networkType)
    {
    case NetworkType_None:
        NN_FALL_THROUGH;
    case NetworkType_Internet:
        break;

    case NetworkType_Local:
        NN_FALL_THROUGH;
    case NetworkType_NeighborDetection:
        NN_FALL_THROUGH;
    default:
        NN_RESULT_THROW(ResultSuspended());     // TODO
    }

    NN_RESULT_SUCCESS;
}

nn::Result ConnectionSelector::ScanCore() NN_NOEXCEPT
{
#if NN_DETAIL_NIFM_CONFIG_USE_WLAN_LIBRARY
    for (auto& networkInterface : m_NetworkInterfaceList)
    {
        if (networkInterface.GetNetworkInterfaceType() == NetworkInterfaceType_Ieee80211)
        {
            WirelessInterface& wirelessInterface = *static_cast<WirelessInterface*>(&networkInterface);
            NN_RESULT_DO(wirelessInterface.Scan());
        }
    }
#endif
    NN_RESULT_SUCCESS;
}

nn::Result ConnectionSelector::Scan() NN_NOEXCEPT
{
    auto result = ConfirmScanExecutable();

    NN_UTIL_SCOPE_EXIT
    {
        NN_UTIL_LOCK_GUARD(m_Mutex);

        do
        {
            ++m_ScanCounter;
        } while (m_ScanCounter == InvalidScanCounter);

        m_ScanResult = result;

        m_ScanSignalList.Signal();

        if (m_ScanResult.IsSuccess())
        {
            m_ScanTimerEvent.Clear();   // TriggerScan() によって即時シグナルされた場合、スキャン中にタイマーでシグナルしたかもしれないのをクリアする
            m_ScanTimerEvent.StartOneShot(m_ScanInterval);
        }

        for (auto&& networkInterface : m_NetworkInterfaceList)
        {
            networkInterface.ClearScanChannels();
        }
    };

    NN_RESULT_DO(result);

    result = ScanCore();    // 直前に ConfirmScanExecutable() で確認しても、無線オフなどが割り込んでここで失敗する可能性はある

    NN_RESULT_THROW(result);
}

void ConnectionSelector::SetScanInterval(const nn::TimeSpan& scanInterval) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_ScanInterval = scanInterval;
    m_ScanTimerEvent.StartOneShot(scanInterval);
}

void ConnectionSelector::TriggerScan(uint32_t* pOutScanCounter, SignalListItem& signalListItem) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutScanCounter);

    {
        NN_UTIL_LOCK_GUARD(m_Mutex);

        *pOutScanCounter = m_ScanCounter;
        m_ScanSignalList.Add(signalListItem);
    }

    m_ScanTimerEvent.Signal();
}

uint32_t ConnectionSelector::GetScanCounter() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    return m_ScanCounter;
}

nn::Result ConnectionSelector::GetScanResult() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    return m_ScanResult;
}

void ConnectionSelector::RemoveScanCompleteListener(SignalListItem& signalListItem) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_ScanSignalList.Remove(signalListItem);
}

nn::Result ConnectionSelector::Register(NetworkInterfaceBase* pNetworkInterface) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_SDK_ASSERT_NOT_NULL(pNetworkInterface);

    for( const auto& networkInterface : m_NetworkInterfaceList )
    {
        if( &networkInterface == pNetworkInterface )
        {
            NN_DETAIL_NIFM_TRACE("NIC already registered.\n");
            NN_RESULT_SUCCESS;
        }
    }

    m_ConnectionEventHandler.Add(pNetworkInterface->GetConnectionEventHandler());

    if (pNetworkInterface->GetNetworkInterfaceType() == NetworkInterfaceType_Ethernet)
    {
        m_NetworkInterfaceList.push_front(*pNetworkInterface);
    }
    else
    {
        m_NetworkInterfaceList.push_back(*pNetworkInterface);
    }

    NN_DETAIL_NIFM_INFO("NIC registered: \n");
    Dump(*pNetworkInterface);

    NN_SDK_ASSERT( m_NetworkInterfaceList.size() <= NetworkInterfaceCountMax );

    NN_RESULT_SUCCESS;
}

nn::Result ConnectionSelector::Unregister(NetworkInterfaceBase* pNetworkInterface) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_SDK_ASSERT_NOT_NULL(pNetworkInterface);

    for( auto itr = m_NetworkInterfaceList.cbegin(); itr != m_NetworkInterfaceList.cend(); ++itr )
    {
        if( &(*itr) == pNetworkInterface )
        {
            NN_DETAIL_NIFM_INFO("NIC unregistered: \n");
            Dump(*pNetworkInterface);
            m_ConnectionEventHandler.Remove(pNetworkInterface->GetConnectionEventHandler());
            m_NetworkInterfaceList.erase(itr);

            NN_RESULT_SUCCESS;
        }
    }

    NN_DETAIL_NIFM_TRACE("NIC not registered.\n");

    NN_RESULT_THROW(nn::nifm::ResultNetworkInterfaceNotFound());
}

nn::Result ConnectionSelector::Clear() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    // TODO?: 複数の無線インターフェースごとに管理

    for( const auto& networkInterface : m_NetworkInterfaceList )
    {
        // TODO:
        NN_UNUSED( networkInterface );
    }

    NN_RESULT_SUCCESS;
}

nn::Result ConnectionSelector::ConfirmKeepingCurrentConnection(const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT
{
    // TODO: [TORIAEZU] AggregatedRequest のネットワーク種別が1種類の前提

    nn::Result result = ResultRequestRejected();
    bool isAnyAcceptable = false;
    bool isAllAcceptable = true;

    for (auto&& networkInterface : m_NetworkInterfaceList)
    {
        nn::Result updateResult = networkInterface.ConfirmKeepingCurrentConnection(aggregatedRequest);

        if (updateResult.IsSuccess())
        {
            isAnyAcceptable = true;
        }
        else
        {
            UpdateResultIfPrior(&result, updateResult);
            isAllAcceptable = false;
        }
    }

    // NIC のどれかひとつに希望する接続があればそれで相乗り可能だが、
    // 接続なしの場合はすべての NIC で接続がない必要がある
    // 相乗り不可の場合は、最も先に進んだ失敗 Result を返す
    if (aggregatedRequest.networkType == NetworkType_None ? isAllAcceptable : isAnyAcceptable)
    {
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW(result);
}

nn::Result ConnectionSelector::EnsureConnectionFree() NN_NOEXCEPT
{
    for( auto&& networkInterface : m_NetworkInterfaceList )
    {
        NetworkResourceInfo networkResourceInfo;
        networkInterface.GetNetworkResourceInfo(&networkResourceInfo);

        switch (networkResourceInfo.networkResourceState)
        {
        case NetworkResourceState::Free:
            continue;

        case NetworkResourceState::Available:
            NN_RESULT_DO(networkInterface.Disconnect(ResultLowPriority()));    // TODO
            NN_FALL_THROUGH;

        case NetworkResourceState::Lost:
            NN_RESULT_THROW(ResultDisconnected());

        default:
            NN_RESULT_THROW(ResultInternalError());
        }
    }

    NN_RESULT_SUCCESS;
}

void ConnectionSelector::HandleNetworkProfileModification() NN_NOEXCEPT
{
    for (auto&& networkInterface : m_NetworkInterfaceList)
    {
        NetworkResourceInfo networkResourceInfo;
        networkInterface.GetNetworkResourceInfo(&networkResourceInfo);

        const auto& networkProfileId = networkResourceInfo.networkProfileId;

        if (networkProfileId != nn::util::InvalidUuid)
        {
            NetworkProfileData networkProfileData;
            if(m_pNetworkProfileManager->GetNetworkProfileData(&networkProfileData, networkProfileId).IsFailure())
            {
                // 使用中の接続設定が削除済みの場合は切断
                NN_DETAIL_NIFM_INFO("Disconnected due to using an obsolete profile.\n");
                networkInterface.Disconnect(ResultNetworkProfileNoLongerAvailable());
            }
            else
            {
                // 使用中の接続設定が存在する場合は更新
                networkInterface.UpdateCurrentNetworkProfile(networkProfileData);
            }
        }
    }
}

void ConnectionSelector::GetLatestAccessPointList(nn::TimeSpan timeSpan) NN_NOEXCEPT
{
    m_AccessPointList.Clear();

    for (auto&& networkInterface : m_NetworkInterfaceList)
    {
        networkInterface.GetLatestAccessPointList(&m_AccessPointList, timeSpan);
    }
}

bool ConnectionSelector::UpdateAccessPointList() NN_NOEXCEPT
{
    bool isUpdated = false;

    m_AccessPointList.Clear();

    for (auto&& networkInterface : m_NetworkInterfaceList)
    {
        if (networkInterface.UpdateAccessPointList(&m_AccessPointList))
        {
            isUpdated = true;
        }
    }

    return isUpdated;
}

void ConnectionSelector::EnumerateConnectionSettingsList(
    ConnectionSettingList* pOutConnectionSettingList,
    const AggregatedRequestType& aggregatedRequestType) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    pOutConnectionSettingList->count = 0;

    // ConnectionSettings の構築（ユーザー設定）
    int profListCount = m_pNetworkProfileManager->GetCopiedNetworkProfileCount( NetworkProfileBase::ProfileType::AnyIncludingDummy); // TODO

    NN_DETAIL_NIFM_INFO("apListCount: %d\n", m_AccessPointList.size());
    Dump(m_AccessPointList);
    NN_DETAIL_NIFM_INFO("profileListCount: %d\n", profListCount);
    Dump(*m_pNetworkProfileManager);

    for( auto&& accessPoint : m_AccessPointList )
    {
        for( int j = 0; j < profListCount; ++j )
        {
            const NetworkProfileBase& networkProfile = *m_pNetworkProfileManager->GetCopiedNetworkProfile( NetworkProfileBase::ProfileType::AnyIncludingDummy, j );

            if( accessPoint.IsPossiblyAvailable(networkProfile, aggregatedRequestType) )
            {
                // TODO: 打ち切り
                if( pOutConnectionSettingList->count < ConnectionSettingList::CountMax )
                {
                    ConnectionSettings* pConnectionSettings = &pOutConnectionSettingList->buffer[pOutConnectionSettingList->count];

                    pConnectionSettings->profileIndex = j;
                    pConnectionSettings->pAccessPoint = &accessPoint;
                    pConnectionSettings->pNetworkProfile = &networkProfile;

                    pOutConnectionSettingList->ptrList[pOutConnectionSettingList->count] = pConnectionSettings;

                    ++pOutConnectionSettingList->count;
                }
            }
        }
    }

    IsPrior isPrior;
    std::sort(&pOutConnectionSettingList->ptrList[0], &pOutConnectionSettingList->ptrList[pOutConnectionSettingList->count], isPrior);
/*
    for( int i = 0; i < pOutConnectionSettingList->count; ++i )
    {
        const NetworkProfileBase* p = pOutConnectionSettingList->ptrList[i]->pNetworkProfile;
        //const AccessPointBase* a = pOutConnectionSettingList->ptrList[i]->pAccessPoint;
        char idString[nn::util::Uuid::StringSize];
        p->GetId().ToString( idString, nn::util::Uuid::StringSize );
        //Ssid ssid;
        //a->GetSsid(&ssid);
        NN_DETAIL_NIFM_TRACE("  id: %s\n", idString );
        //NN_DETAIL_NIFM_TRACE("  ssid: %s\n", ssid.hex );
        NN_DETAIL_NIFM_TRACE("  profile type: %d\n", p->GetType());
        NN_DETAIL_NIFM_TRACE("  IF-type: %d\n", p->GetNetworkInterfaceType());
        NN_DETAIL_NIFM_TRACE("  index: %d\n", pOutConnectionSettingList->ptrList[i]->profileIndex);
        NN_DETAIL_NIFM_TRACE("-----------------------\n");
    }
*/
}

nn::Result ConnectionSelector::ConnectAny(
    nn::util::optional<AdditionalInfo>* pOutAdditionalInfo,
    ConnectionSettings* pOutConnectionSettings,
    nn::util::optional<IpAddressSetting>* pOutIpAddressSetting,
    nn::util::optional<DnsSetting>* pOutDnsSetting,
    const ConnectionSettingList& connectionSettingList,
    const AggregatedRequestType& aggregatedRequest) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutConnectionSettings);

    nn::Result result = ResultNetworkNotFound();   // いちばん優先順位の低い Result

    for( int i = 0; i < connectionSettingList.count; ++i )
    {
        const ConnectionSettings& connectionSettings = *connectionSettingList.ptrList[i];
        const NetworkProfileBase& networkProfile = *connectionSettings.pNetworkProfile;
        const AccessPointBase& accessPoint = *connectionSettings.pAccessPoint;

        if (aggregatedRequest.networkType == NetworkType_Internet)
        {
            // TODO: 無線OFF時にも，次ループでの有線接続試行は継続してよい
            NN_RESULT_DO(CancelFlagHolder::GetSingleton().ConfirmInternetAdmitted(accessPoint.GetNetworkInterface().GetNetworkInterfaceType()));
        }

#if 0
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
        {
            Ssid profileSsid;
            Ssid apSsid;

            networkProfile.GetWirelessSetting().GetSsid(&profileSsid);
            accessPoint.GetSsid(&apSsid);

            const int ssidStrSize = Ssid::HexSize * 2 + 1;
            char profileSsidStr[ssidStrSize];
            char apSsidStr[ssidStrSize];;
            WirelessSettingBase::ConvertSsidHexToString( profileSsidStr, ssidStrSize, profileSsid );
            WirelessSettingBase::ConvertSsidHexToString( apSsidStr, ssidStrSize, apSsid );

            NN_DETAIL_NIFM_TRACE("{%.32s, %.32s}\n", profileSsidStr, apSsidStr);
        }
#endif
#endif

        nn::Result connectResult = accessPoint.Connect(networkProfile, aggregatedRequest);

        NN_DETAIL_NIFM_TRACE("networkInterface.Connect: %08x\n", connectResult.GetInnerValueForDebug());

        AdditionalInfo additionalInfo = {};

        if (connectResult.IsSuccess())
        {
            additionalInfo.profileId = connectionSettings.pNetworkProfile->GetId();
            *pOutAdditionalInfo = additionalInfo;
            *pOutConnectionSettings = connectionSettings;
            NN_RESULT_SUCCESS;
        }
        if (connectResult <= ResultHotspotAuthenticationViaWebAuthAppletNeeded())
        {
            // 疎通確認中にリダイレクトされた場合、その情報を取得

            accessPoint.GetNetworkInterface().GetRedirectInfoIfAny(&additionalInfo);
            *pOutAdditionalInfo = additionalInfo;
        }

        if (UpdateResultIfPrior(&result, connectResult))
        {
            additionalInfo.profileId = connectionSettings.pNetworkProfile->GetId();
            *pOutAdditionalInfo = additionalInfo;
            // 失敗が更新された際の接続設定を保存
            *pOutConnectionSettings = connectionSettings;

            IpAddressSetting ipAddressSetting;
            DnsSetting dnsSetting;

            if (GetCurrentIpConfigInfo(&ipAddressSetting, &dnsSetting).IsSuccess())
            {
                *pOutIpAddressSetting = ipAddressSetting;
                *pOutDnsSetting = dnsSetting;
            }
            else
            {
                *pOutIpAddressSetting = nn::util::nullopt;
                *pOutDnsSetting = nn::util::nullopt;
            }
        }

        // 接続試行途中の失敗・切断は外に伝える必要はない
        accessPoint.Release();
    }

    NN_RESULT_THROW(result);
}

nn::Result ConnectionSelector::GetRedirectInfo(AdditionalInfo* pOutAdditionalInfo) const NN_NOEXCEPT
{
    nn::Result result = ResultInternalError(); // TODO

    for (const auto& networkInterface : m_NetworkInterfaceList)
    {
        result = networkInterface.GetRedirectInfoIfAny(pOutAdditionalInfo);
        if (result.IsSuccess())
        {
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(result);
}

nn::Result ConnectionSelector::Renew(AdditionalInfo* pOutAdditionalInfo, const AggregatedRequestType& aggregatedRequestType) NN_NOEXCEPT
{
    // totalNetworkResourceInfo.networkResourceState が ToBePrepared のうちに
    // この関数を抜けてはいけません。必ず Available か Lost になるまでブロックします。

    // この関数内で Free → ToBePrepared → Available → Lost と一気に遷移してはいけません。
    // Free から Available に変化したり Available から Lost に変化したら、
    // 必ず一度関数を抜けなければなりません。

    // Lost から Free へ遷移してもいけません。
    // Lost から Free への遷移は ReleaseNetworkResourceLost でおこないます。

    // この関数内で発生する Available から Lost への遷移はすべて、
    // 能動的な Disconnect によるものでなければなりません。外部要因の切断による遷移は
    // GetConnectionEventHandler().ExecuteCallbackIfSignaled() でおこないます。

    NN_DETAIL_NIFM_INFO("ConnectionSelector::Renew()\n");
    m_pNetworkProfileManager->CopyAllNetworkProfiles();

    TotalNetworkResourceInfo totalNetworkResourceInfo;
    GetTotalNetworkResourceInfo(&totalNetworkResourceInfo);

    // すでに切断状態であれば、先に Request の状態更新が必要
    NN_RESULT_THROW_UNLESS(
        totalNetworkResourceInfo.networkResourceState != NetworkResourceState::Lost,
        ResultDisconnected()
    );

    // BePrepared の状態でここに来てはいけない（Connect 呼び出し中のみの状態）
    // その他はそもそも列挙体の値域外
    NN_SDK_ASSERT(
        totalNetworkResourceInfo.networkResourceState == NetworkResourceState::Free ||
        totalNetworkResourceInfo.networkResourceState == NetworkResourceState::Available
    );
    NN_RESULT_THROW_UNLESS(
        totalNetworkResourceInfo.networkResourceState == NetworkResourceState::Free ||
        totalNetworkResourceInfo.networkResourceState == NetworkResourceState::Available,
        ResultInternalError()
    );

    NN_SDK_ASSERT(
        totalNetworkResourceInfo.networkResourceState != NetworkResourceState::Free ||
        totalNetworkResourceInfo.networkType == NetworkType_None
    );

    {
        nn::Result result = ConfirmKeepingCurrentConnection(aggregatedRequestType);
        if (result.IsSuccess() || result <= ResultHotspotAuthenticationViaWebAuthAppletNeeded())
        {
            // 相乗り可能ならテレメトリ情報を更新して終わり
            if (IsTelemetryUpdateNeeded(aggregatedRequestType))
            {
                nn::util::optional<AccessPointData> accessPoint;
                nn::util::optional<NetworkProfileData> networkProfile;
                nn::util::optional<IpAddressSetting> ipAddressSetting;
                nn::util::optional<DnsSetting> dnsSetting;

                AccessPointData accessPointData;
                NetworkProfileData networkProfileData;
                IpAddressSetting ipAddressSettingData;
                DnsSetting dnsSettingData;

                if (GetCurrentAccessPoint(&accessPointData).IsSuccess())
                {
                    accessPoint = accessPointData;
                }

                if (GetCurrentNetworkProfile(&networkProfileData).IsSuccess())
                {
                    networkProfile = networkProfileData;
                }

                if (GetCurrentIpConfigInfo(&ipAddressSettingData, &dnsSettingData).IsSuccess())
                {
                    ipAddressSetting = ipAddressSettingData;
                    dnsSetting = dnsSettingData;
                }

                nn::util::optional<AdditionalInfo> additionalInfo;
                if (result <= ResultHotspotAuthenticationViaWebAuthAppletNeeded())
                {
                    // 疎通確認中にリダイレクトされた場合、その情報を取得
                    GetRedirectInfo(pOutAdditionalInfo);
                    additionalInfo = *pOutAdditionalInfo;
                }

                UpdateTelemetryContext(result, networkProfile, accessPoint, ipAddressSetting, dnsSetting, additionalInfo);
            }

            NN_RESULT_THROW(result);
        }
    }

    // 相乗り不可で、異なるタイプの接続をしていたら切断
    // ローカル通信 ⇔ インフラ通信はモード遷移後の周囲の状況（APがあるか、子機がいるか）が不明で、
    // 事前に要求成就の可能性を検証できないので切ってしまう
    if (totalNetworkResourceInfo.networkType != aggregatedRequestType.networkType)
    {
        NN_RESULT_DO(EnsureConnectionFree());
    }

    // 以下、接続試行

    NN_SDK_ASSERT(
        totalNetworkResourceInfo.networkType == NetworkType_None ||
        (
            totalNetworkResourceInfo.networkType == aggregatedRequestType.networkType &&
            totalNetworkResourceInfo.networkResourceState == NetworkResourceState::Available
        )
    );

    NN_SDK_ASSERT_NOT_EQUAL(aggregatedRequestType.networkType, NetworkType_None);
    NN_RESULT_THROW_UNLESS(aggregatedRequestType.networkType != NetworkType_None, ResultInternalError());     // TODO

    NN_SDK_ASSERT(aggregatedRequestType.networkType == NetworkType_Internet ||
        aggregatedRequestType.networkType == NetworkType_Local ||
        aggregatedRequestType.networkType == NetworkType_NeighborDetection);

    // ローカル・すれちがい通信中はアクセスポイントをスキャンできないのでまず切断
    if ((totalNetworkResourceInfo.networkType == NetworkType_Local ||
        totalNetworkResourceInfo.networkType == NetworkType_NeighborDetection)
        && aggregatedRequestType.networkType == NetworkType_Internet)
    {
        NN_RESULT_DO(EnsureConnectionFree());
    }

    GetLatestAccessPointList(nn::TimeSpan::FromSeconds(ScanDataLifeTimeInSeconds));

    ConnectionSettings connectionSettings = {};

    {
        nn::Result result = ResultNetworkNotFound();  // いちばん優先順位の低い Result

        // テレメトリ用情報
        nn::util::optional<AccessPointData> accessPoint;
        nn::util::optional<NetworkProfileData> networkProfile;
        nn::util::optional<IpAddressSetting> ipAddressSetting;
        nn::util::optional<DnsSetting> dnsSetting;
        nn::util::optional<AdditionalInfo> additionalInfo;

        int loop = 0;
        NN_UNUSED(loop);
        do
        {
            ConnectionSettingList connectionSettingList;
            EnumerateConnectionSettingsList(&connectionSettingList, aggregatedRequestType);

            NN_DETAIL_NIFM_INFO("Number of candidates: %d (loop:%d)\n", connectionSettingList.count, ++loop);

            if (connectionSettingList.count == 0)
            {
                continue;
            }

            // 接続を試すべき（AP,Profile）の組が存在する場合、現在の接続をすべて切断する
            // TODO: 複数の NIC を同時利用できるようになったら要変更
            if (totalNetworkResourceInfo.networkResourceState == NetworkResourceState::Available)
            {
                NN_RESULT_DO(EnsureConnectionFree());
            }

            additionalInfo = nn::util::nullopt;
            if (UpdateResultIfPrior(&result, ConnectAny(&additionalInfo, &connectionSettings, &ipAddressSetting, &dnsSetting, connectionSettingList, aggregatedRequestType)))
            {
                if (additionalInfo)
                {
                    *pOutAdditionalInfo = *additionalInfo;
                }

                AccessPointData accessPointData;
                NetworkProfileData networkProfileData;

                if (connectionSettings.pAccessPoint != nullptr &&
                    connectionSettings.pAccessPoint->TryExport(&accessPointData).IsSuccess())
                {
                    accessPoint = accessPointData;
                }
                else
                {
                    accessPoint = nn::util::nullopt;
                }

                if (connectionSettings.pNetworkProfile != nullptr)
                {
                    connectionSettings.pNetworkProfile->Export(&networkProfileData);
                    networkProfile = networkProfileData;
                }
                else
                {
                    networkProfile = nn::util::nullopt;
                }
            }
        } while (result.IsFailure() && UpdateAccessPointList());

        // テレメトリ情報を更新
        if (IsTelemetryUpdateNeeded(aggregatedRequestType))
        {
            UpdateTelemetryContext(result, networkProfile, accessPoint, ipAddressSetting, dnsSetting, additionalInfo);
        }

        NN_RESULT_DO(result);
    }

    NN_UTIL_LOCK_GUARD(m_Mutex);

    // 接続成功したら履歴を更新
    if( connectionSettings.pNetworkProfile->GetType() == NetworkProfileBase::ProfileType::UserProfile )  // TODO: 真面目に判定する
    {
        m_pNetworkProfileManager->UpdateUserNetworkProfileOrder( connectionSettings.pNetworkProfile->GetId() );
        // TODO: 履歴更新に失敗した場合の処理?
    }

    NN_RESULT_SUCCESS;
} // NOLINT(readability/fn_size)

nn::Result ConnectionSelector::GetCurrentNetworkProfile( NetworkProfileData *pOutNetworkProfileData ) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    nn::Result result = ResultNotConnected();

    for (auto&& networkInterface : m_NetworkInterfaceList)
    {
        auto result2 = networkInterface.GetCurrentNetworkProfile(pOutNetworkProfileData);

        if (result2.IsSuccess())
        {
            NN_RESULT_SUCCESS;
        }
        else if (!(result2 <= ResultNotConnected()))
        {
            result = result2;
        }
    }

    NN_RESULT_THROW(result);
}

nn::Result ConnectionSelector::GetTotalNetworkResourceInfo(TotalNetworkResourceInfo* pTotalNetworkResourceInfo) const NN_NOEXCEPT
{
    for( const auto& networkInterface : m_NetworkInterfaceList )
    {
        NetworkResourceInfo networkResourceInfo;
        networkInterface.GetNetworkResourceInfo(&networkResourceInfo);

        if( networkResourceInfo.networkResourceState != NetworkResourceState::Free )
        {
            *pTotalNetworkResourceInfo = networkResourceInfo;

            NN_RESULT_SUCCESS;
        }
    }

    pTotalNetworkResourceInfo->Clear();

    NN_RESULT_SUCCESS;
}

nn::Result ConnectionSelector::ReleaseNetworkResourceLost(const TotalNetworkResourceInfo& totalNetworkResourceInfo) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    // totalNetworkResourceInfo で検知していない Lost は Release してはいけない

    if (totalNetworkResourceInfo.networkResourceState != NetworkResourceState::Lost)
    {
        NN_RESULT_SUCCESS;
    }

    for (auto&& networkInterface : m_NetworkInterfaceList)
    {
        NetworkResourceInfo networkResourceInfo;
        networkInterface.GetNetworkResourceInfo(&networkResourceInfo);

        if( networkResourceInfo.networkType == totalNetworkResourceInfo.networkType &&
            networkResourceInfo.networkResourceState == totalNetworkResourceInfo.networkResourceState &&
            networkResourceInfo.networkInterfaceType == totalNetworkResourceInfo.networkInterfaceType )
        {
            networkInterface.Release();

            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result ConnectionSelector::EnumerateNetworkInterfaces( NetworkInterfaceInfo *pNetworkInterfaceInfo, int* pOutCount, int inCount ) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    *pOutCount = 0;

    for( const auto& networkInterface : m_NetworkInterfaceList )
    {
        if( *pOutCount < inCount )
        {
            auto pInfo = &pNetworkInterfaceInfo[(*pOutCount)];
            pInfo->type = networkInterface.GetNetworkInterfaceType();
            networkInterface.GetMacAddress( &pInfo->macAddress );
            pInfo->isAvailable = true;
        }
        ++(*pOutCount);
    }

    NN_RESULT_SUCCESS;
}

nn::Result ConnectionSelector::GetScanData(AccessPointListBase* pOutAccessPointList) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    for (const auto& networkInterface : m_NetworkInterfaceList)
    {
        // TODO: 複数の NIC からスキャン結果を受けるようになったときの NIC をまたいだ全体的なソート
        NN_RESULT_DO(networkInterface.GetScanData(pOutAccessPointList));
    }

    NN_RESULT_SUCCESS;
}

nn::Result ConnectionSelector::GetCurrentIpAddress( IpV4Address* pOutIpAddress ) const NN_NOEXCEPT
{
    nn::Result result = ResultInternalError();

    // TODO: IP アドレスのリストを返す？
    for( const auto& networkInterface : m_NetworkInterfaceList )
    {
        result = networkInterface.GetCurrentIpAddress( pOutIpAddress );
        if( result.IsSuccess() )
        {
            break;
        }
    }

    NN_RESULT_THROW(result);
}

nn::Result ConnectionSelector::GetCurrentIpConfigInfo( IpAddressSetting* pOutIpAddressSetting, DnsSetting* pOutDnsSetting ) const NN_NOEXCEPT
{
    nn::Result result = ResultResourceNotFound();

    // TODO: IP アドレスのリストを返す？
    for( const auto& networkInterface : m_NetworkInterfaceList )
    {
        result = networkInterface.GetCurrentIpConfigInfo( pOutIpAddressSetting, pOutDnsSetting );
        if( result.IsSuccess() )
        {
            break;
        }
    }

    NN_RESULT_THROW(result);
}

nn::Result ConnectionSelector::GetCurrentAccessPoint(AccessPointData* pOutAccessPointData) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    nn::Result result = ResultNotConnected();

    for (auto&& networkInterface : m_NetworkInterfaceList)
    {
        auto result2 = networkInterface.GetCurrentAccessPoint(pOutAccessPointData);

        if (result2.IsSuccess())
        {
            NN_RESULT_SUCCESS;
        }
        else if (!(result2 <= ResultNotConnected()))
        {
            result = result2;
        }
    }

    NN_RESULT_THROW(result);
}

bool ConnectionSelector::IsNicSwitchingRequired() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    bool isWirelessNetworkAvailable = false;
    bool isAnyEthernetInterfaceLinkUp = false;

    for (auto&& networkInterface : m_NetworkInterfaceList)
    {
        switch (networkInterface.GetNetworkInterfaceType())
        {
        case NetworkInterfaceType_Ieee80211:
            {
                NetworkResourceInfo networkResourceInfo;
                networkInterface.GetNetworkResourceInfo(&networkResourceInfo);

                if (networkResourceInfo.networkResourceState == NetworkResourceState::Available)
                {
                    isWirelessNetworkAvailable = true;
                }
            }
            break;

        case NetworkInterfaceType_Ethernet:
            if (networkInterface.IsLinkUp())
            {
                isAnyEthernetInterfaceLinkUp = true;
            }
            break;

        default:
            // do nothing
            break;
        }
    }

    return isWirelessNetworkAvailable && isAnyEthernetInterfaceLinkUp;
}

bool ConnectionSelector::IsTelemetryUpdateNeeded(const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(uint32_t, telemetryLastRevision, =0);

    NN_DETAIL_NIFM_INFO("IsTelemetryUpdateNeeded, isInternalContextUpdatable=%s, lastRevision=%d, telemetryLastRevision=%d\n",
        aggregatedRequest.isInternalContextUpdatable ? "true" : "false", aggregatedRequest.lastRevision, telemetryLastRevision);

    // テレメトリコンテキストの更新が必要な統合利用要求 かつ 新たな要求の提出があった場合のみ更新
    if (aggregatedRequest.isInternalContextUpdatable &&
        aggregatedRequest.lastRevision != telemetryLastRevision)
    {
        telemetryLastRevision = aggregatedRequest.lastRevision;

        return true;
    }

    return false;
}

void ConnectionSelector::UpdateTelemetryContext(Result result,
    const nn::util::optional<NetworkProfileData>& networkProfileData,
    const nn::util::optional<AccessPointData>& accessPointData,
    const nn::util::optional<IpAddressSetting>& ipAddressSetting,
    const nn::util::optional<DnsSetting>& dnsSetting,
    const nn::util::optional<AdditionalInfo>& additionalInfo) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_TelemetryContext.SetResult(result);
    m_TelemetryContext.SetNetworkProfile(networkProfileData);
    m_TelemetryContext.SetAccessPoint(accessPointData);
    m_TelemetryContext.SetIpConfig(ipAddressSetting, dnsSetting);
    m_TelemetryContext.SetAdditionalInfo(additionalInfo);

    if(additionalInfo)
    {
        NN_SDK_LOG("UpdateTelemetryContext::AdditionalInfo %s\n", additionalInfo->redirectUrl);
    }

    m_TelemetryContext.SetEthernetMacAddress(nn::util::nullopt);
    m_TelemetryContext.SetWirelessMacAddress(nn::util::nullopt);

    for (const auto& networkInterface : m_NetworkInterfaceList)
    {
        MacAddress macAddress;
        networkInterface.GetMacAddress(&macAddress);
        if (networkInterface.GetNetworkInterfaceType() == NetworkInterfaceType_Ethernet)
        {
            m_TelemetryContext.SetEthernetMacAddress(macAddress);
        }
        else if (networkInterface.GetNetworkInterfaceType() == NetworkInterfaceType_Ieee80211)
        {
            m_TelemetryContext.SetWirelessMacAddress(macAddress);
        }
    }

    SsidListVersion ssidListVersion;
    if (m_pNetworkProfileManager->GetSsidListVersion(&ssidListVersion).IsSuccess())
    {
        m_TelemetryContext.SetSsidListVersion(ssidListVersion);
    }
    else
    {
        m_TelemetryContext.SetSsidListVersion(nn::util::nullopt);
    }

    m_TelemetryContext.Signal();
}

nn::os::NativeHandle ConnectionSelector::GetTelemetrySystemEventReadableHandle() NN_NOEXCEPT
{
    return m_TelemetryContext.GetSystemEventReadableHandle();
}

void ConnectionSelector::GetTelemetryInfo(TelemetryInfo* pOutTelemetryInfo) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_TelemetryContext.GetTelemetryInfo(pOutTelemetryInfo);
}

nn::Result ConnectionSelector::ConfirmInterfaceHealth() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    for (auto&& networkInterface : m_NetworkInterfaceList)
    {
        NN_RESULT_DO(networkInterface.ConfirmHealth());
    }

    NN_RESULT_SUCCESS;
}

nn::Result ConnectionSelector::SetTcpSessionInformation(const SocketInfo socketInfoArray[], int count) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_IsAwake, ResultInternalError());

    for (auto&& networkInterface : m_NetworkInterfaceList)
    {
        for (int i = 0; i < count; ++i)
        {
            networkInterface.SetTcpSessionInformation(socketInfoArray[i]);  // 非対応の NIC に設定した場合の失敗は無視
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result ConnectionSelector::SetScanChannels(const int16_t scanChannelArray[], int count) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    for (auto&& networkInterface : m_NetworkInterfaceList)
    {
        networkInterface.SetScanChannels(scanChannelArray, count);
    }

    NN_RESULT_SUCCESS;
}

bool ConnectionSelector::IsScanAllowed(const int16_t scanChannelArray[], int count) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    int16_t scanChannels[WirelessChannelsCountMax];
    int scanChannelCount = 0;
    bool ret = true;

    for (auto&& networkInterface : m_NetworkInterfaceList)
    {
        if (networkInterface.GetScanChannels(scanChannels, &scanChannelCount).IsSuccess())
        {
            ret = false; // 無線 NIC が存在すれば、それの結果が優先される（有線のみならばスキャン許可されるが実際には、何も処理しない）

            if (scanChannelCount == NetworkInterfaceBase::InvalidScanChannelCount)
            {
                SetScanChannels(scanChannelArray, count);
                return true; // スキャン中でなければ許可
            }
            else if(scanChannelCount != count)
            {
                continue;
            }

            bool allChannelsAllowed = true;
            for (int i = 0; i < scanChannelCount; ++i)
            {
                if (scanChannels[i] != scanChannelArray[i])
                {
                    allChannelsAllowed = false;
                    break;
                }
            }

            if (allChannelsAllowed)
            {
                return true; // 同一パラメータでのスキャン中なら許可
            }
        }
    }

    return ret;
}

nn::Result ConnectionSelector::PutToSleep() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_IsAwake = false;

    for (auto&& networkInterface : m_NetworkInterfaceList)
    {
        NN_RESULT_DO(networkInterface.PutToSleep());
    }

    NN_RESULT_SUCCESS;
}

nn::Result ConnectionSelector::WakeUp() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    for (auto&& networkInterface : m_NetworkInterfaceList)
    {
        NN_RESULT_TRY(networkInterface.WakeUp())
            NN_RESULT_CATCH(nn::wlan::ResultAlreadyAwake)
            {
                NN_DETAIL_NIFM_WARN("RequestWakeUp return AlreadyAwake\n");
                continue;
            }
        NN_RESULT_END_TRY;
    }

    m_IsAwake = true;

    NN_RESULT_SUCCESS;
}

}
}
}
