﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/core/connectionConfirmation/nifm_HttpClientBase.horizon.h>

#include <nn/nifm/detail/util/nifm_CancelFlagHolder.h>
#include <nn/nifm/detail/util/nifm_FwdbgSettings.h>

#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/nn_Version.h>

#include <nn/socket.h>


namespace nn
{
namespace nifm
{
namespace detail
{

namespace
{
}

HttpClientBase::HttpHeaderReaderBase::HttpHeaderReaderBase() NN_NOEXCEPT
{
}

HttpClientBase::HttpHeaderReaderBase::~HttpHeaderReaderBase() NN_NOEXCEPT
{
}

bool HttpClientBase::HttpHeaderReaderBase::ParseResponseHeaderLine(Field* pOutHttpHeaderField, const char* pInLine, int count) NN_NOEXCEPT
{
    pOutHttpHeaderField->name[0] = '\0';
    pOutHttpHeaderField->value[0] = '\0';

    int i = 0;
    int j = 0;

    while (pInLine[i] != ':')
    {
        if (i == count ||
            j == static_cast<int>(NN_ARRAY_SIZE(pOutHttpHeaderField->name)) - 1)
        {
            return false;
        }

        pOutHttpHeaderField->name[j] = pInLine[i];

        ++i;
        ++j;
    }

    pOutHttpHeaderField->name[j] = '\0';

    ++i;    // ':' をスキップ

    while (pInLine[i] == ' ' || pInLine[i] == '\t')
    {
        if (i == count)
        {
            return false;
        }

        ++i;
    }

    j = 0;

    while (pInLine[i] != '\r' && pInLine[i] != '\n')
    {
        if (i == count ||
            j == static_cast<int>(NN_ARRAY_SIZE(pOutHttpHeaderField->value)) - 1)
        {
            return false;
        }

        pOutHttpHeaderField->value[j] = pInLine[i];

        ++i;
        ++j;
    }

    pOutHttpHeaderField->value[j] = '\0';

    return true;
}

HttpClientBase::HttpClientBase() NN_NOEXCEPT
    : m_pProxySetting(nullptr),
      m_StatusCode(0),
      m_pHttpPostData(nullptr),
      m_pHttpResponse(nullptr)
{
    Initialize();
}

HttpClientBase::~HttpClientBase() NN_NOEXCEPT
{
    Finalize();
}

void HttpClientBase::Initialize() NN_NOEXCEPT
{
    m_pCurl = curl_easy_init();
    NN_ABORT_UNLESS_NOT_NULL(m_pCurl);
}

void HttpClientBase::Finalize() NN_NOEXCEPT
{
    m_pProxySetting = nullptr;
    //m_Url[0] = '\0';
    m_StatusCode = 0;

    if (m_pCurl)
    {
        curl_easy_cleanup(m_pCurl);
    }
}

void HttpClientBase::Reset() NN_NOEXCEPT
{
    if (m_pCurl)
    {
        Finalize();
        Initialize();
    }
}

void HttpClientBase::SetSequencePeriod(SequencePeriod sequencePeriod) NN_NOEXCEPT
{
    m_SequencePeriod = sequencePeriod;
}


Result HttpClientBase::CurlCodeToNnResult(CURLcode curlCode) const NN_NOEXCEPT
{
    if (m_SequencePeriod == SequencePeriod::ConnectionTest)
    {
        switch (curlCode)
        {
        case CURLE_OK:
            NN_RESULT_SUCCESS;

        case CURLE_COULDNT_RESOLVE_PROXY:
            NN_RESULT_THROW(ResultConnectionTestHttpCouldntResolveProxy());

        case CURLE_COULDNT_RESOLVE_HOST:
            NN_RESULT_THROW(ResultConnectionTestHttpCouldntResolveHost());

        case CURLE_COULDNT_CONNECT:
            NN_RESULT_THROW(ResultConnectionTestHttpCouldntConnect());

        case CURLE_OPERATION_TIMEDOUT:
            NN_RESULT_THROW(ResultConnectionTestHttpOperationTimedout());

        case CURLE_SEND_ERROR:
            NN_RESULT_THROW(ResultConnectionTestHttpSendError());

        case CURLE_RECV_ERROR:
            NN_RESULT_THROW(ResultConnectionTestHttpRecvError());

        default:
            NN_DETAIL_NIFM_WARN_V3("CurlCode=%d(%d)\n", curlCode, m_SequencePeriod);
            NN_RESULT_THROW(ResultConnectionTestCommunicationFailure());
        }
    }
    else if (m_SequencePeriod == SequencePeriod::AuthenticationNas)
    {
        switch (curlCode)
        {
        case CURLE_OK:
            NN_RESULT_SUCCESS;

        case CURLE_COULDNT_RESOLVE_PROXY:
            NN_RESULT_THROW(ResultNintendoHotspotAuthenticationHttpCouldntResolveProxy());

        case CURLE_COULDNT_RESOLVE_HOST:
            NN_RESULT_THROW(ResultNintendoHotspotAuthenticationHttpCouldntResolveHost());

        case CURLE_COULDNT_CONNECT:
            NN_RESULT_THROW(ResultNintendoHotspotAuthenticationHttpCouldntConnect());

        case CURLE_OPERATION_TIMEDOUT:
            NN_RESULT_THROW(ResultNintendoHotspotAuthenticationHttpOperationTimedout());

        case CURLE_SEND_ERROR:
            NN_RESULT_THROW(ResultNintendoHotspotAuthenticationHttpSendError());

        case CURLE_RECV_ERROR:
            NN_RESULT_THROW(ResultNintendoHotspotAuthenticationHttpRecvError());

        default:
            NN_DETAIL_NIFM_WARN_V3("CurlCode=%d(%d)\n", curlCode, m_SequencePeriod);
            NN_RESULT_THROW(ResultNintendoHotspotAuthenticationCommunicationFailure());
        }
    }
    else if (m_SequencePeriod == SequencePeriod::AuthenticationHotspot)
    {
        switch (curlCode)
        {
        case CURLE_OK:
            NN_RESULT_SUCCESS;

        case CURLE_COULDNT_RESOLVE_PROXY:
            NN_RESULT_THROW(ResultProviderHotspotAuthenticationHttpCouldntResolveProxy());

        case CURLE_COULDNT_RESOLVE_HOST:
            NN_RESULT_THROW(ResultProviderHotspotAuthenticationHttpCouldntResolveHost());

        case CURLE_COULDNT_CONNECT:
            NN_RESULT_THROW(ResultProviderHotspotAuthenticationHttpCouldntConnect());

        case CURLE_OPERATION_TIMEDOUT:
            NN_RESULT_THROW(ResultProviderHotspotAuthenticationHttpOperationTimedout());

        case CURLE_SEND_ERROR:
            NN_RESULT_THROW(ResultProviderHotspotAuthenticationHttpSendError());

        case CURLE_RECV_ERROR:
            NN_RESULT_THROW(ResultProviderHotspotAuthenticationHttpRecvError());

        default:
            NN_DETAIL_NIFM_WARN_V3("CurlCode=%d(%d)\n", curlCode, m_SequencePeriod);
            NN_RESULT_THROW(ResultConnectionTestCommunicationFailure());
        }
    }

    NN_RESULT_THROW(ResultInternetConnectionNotAvailable());
}

Result HttpClientBase::CurlMCodeToNnResult(CURLMcode curlMCode) const NN_NOEXCEPT
{
    if (curlMCode == CURLM_OK)
    {
        NN_RESULT_SUCCESS;
    }
    else
    {
        if (m_SequencePeriod == SequencePeriod::ConnectionTest)
        {
            NN_DETAIL_NIFM_INFO_V3("CurlCode=%d(%d)\n", curlMCode, m_SequencePeriod);
            NN_RESULT_THROW(ResultConnectionTestCommunicationFailure());
        }
        else if (m_SequencePeriod == SequencePeriod::AuthenticationNas)
        {
            NN_DETAIL_NIFM_INFO_V3("CurlCode=%d(%d)\n", curlMCode, m_SequencePeriod);
            NN_RESULT_THROW(ResultNintendoHotspotAuthenticationCommunicationFailure());
        }
        else if (m_SequencePeriod == SequencePeriod::AuthenticationHotspot)
        {
            NN_DETAIL_NIFM_INFO_V3("CurlCode=%d(%d)\n", curlMCode, m_SequencePeriod);
            NN_RESULT_THROW(ResultConnectionTestCommunicationFailure());
        }

        NN_RESULT_THROW(ResultInternetConnectionNotAvailable());
    }
}

void HttpClientBase::SetNetworkInterfaceType(nn::nifm::NetworkInterfaceType networkInterfaceType) NN_NOEXCEPT
{
    m_NetworkInterfaceType = networkInterfaceType;
}

void HttpClientBase::SetProxySetting(const ProxySetting& proxySetting) NN_NOEXCEPT
{
    m_pProxySetting = &proxySetting;
}

int HttpClientBase::CurlSetSockOptsCallout(void *pData, curl_socket_t curlfd, curlsocktype purpose)
{
    NN_UNUSED(pData);
    NN_UNUSED(purpose);

    // Setting linger option so that sockets do not consume memory after being closed
    linger soLinger = {true,  1};
    nn::socket::SetSockOpt(curlfd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Nn_Linger, &soLinger, sizeof(soLinger));

    return CURL_SOCKOPT_OK;
}

void HttpClientBase::SetCurlCommonOpt() NN_NOEXCEPT
{
    curl_easy_setopt(m_pCurl, CURLOPT_TIMEOUT_MS, FwdbgSettings::GetSingleton().GetConnectionTestTimeout().GetMilliSeconds());
    curl_easy_setopt(m_pCurl, CURLOPT_PROXYAUTOCONFIG, false);
    curl_easy_setopt(m_pCurl, CURLOPT_SOCKOPTFUNCTION, CurlSetSockOptsCallout);
    curl_easy_setopt(m_pCurl, CURLOPT_USERAGENT, "NX NIFM/00");

    if (m_pProxySetting != nullptr && m_pProxySetting->isEnabled)
    {
        curl_easy_setopt(m_pCurl, CURLOPT_PROXY, m_pProxySetting->proxy);
        curl_easy_setopt(m_pCurl, CURLOPT_PROXYPORT, m_pProxySetting->port);

        if (m_pProxySetting->authentication.isEnabled)
        {
            // Basic authentication
            curl_easy_setopt(m_pCurl, CURLOPT_PROXYUSERNAME, m_pProxySetting->authentication.username);
            curl_easy_setopt(m_pCurl, CURLOPT_PROXYPASSWORD, m_pProxySetting->authentication.password);
        }
    }
}

void HttpClientBase::SetRequestUrl(const char* pUrl) NN_NOEXCEPT
{
    //auto l = nn::util::Strlcpy(m_Url, pUrl, sizeof(m_Url));
    //NN_ABORT_UNLESS(l < sizeof(m_Url), "Insufficient buffer for user agent: required %d bytes\n", l + 1);

    curl_easy_setopt(m_pCurl, CURLOPT_URL, pUrl);
}

void HttpClientBase::SetRequesetMethod(HttpRequestMethod method) NN_NOEXCEPT
{
    switch (method)
    {
    case HttpRequestMethod::Get:
        // set nothing
        break;
    case HttpRequestMethod::Post:
        curl_easy_setopt(m_pCurl, CURLOPT_POST, 1);
        break;
    case HttpRequestMethod::Put:
        curl_easy_setopt(m_pCurl, CURLOPT_CUSTOMREQUEST, "PUT");
        break;
    case HttpRequestMethod::Delete:
        curl_easy_setopt(m_pCurl, CURLOPT_CUSTOMREQUEST, "DELETE");
        break;
    default:
        NN_DETAIL_NIFM_WARN("Invalid http request method\n");
        break;
    }
}

void HttpClientBase::SetPostDataBuffer(HttpPostData* pPostData) NN_NOEXCEPT
{
    m_pHttpPostData = pPostData;
}

void HttpClientBase::SetResponseBuffer(HttpResponse* pResponse) NN_NOEXCEPT
{
    m_pHttpResponse = pResponse;
}

nn::Result HttpClientBase::Connect() NN_NOEXCEPT
{
    {
        auto cancelResult = CancelFlagHolder::GetSingleton().ConfirmConnectionConfirmationAdmitted(m_NetworkInterfaceType);
        if (cancelResult.IsFailure())
        {
            NN_DETAIL_NIFM_INFO("HTTP communication is cancelled.\n");
            NN_RESULT_THROW(cancelResult);
        }
    }

    {
        // 以下のサンプルコードを参考
        // https://curl.haxx.se/libcurl/c/curl_multi_wait.html
        // https://curl.haxx.se/libcurl/c/multi-app.html

        CURLM* pCurlMulti = curl_multi_init();
        curl_multi_add_handle(pCurlMulti, m_pCurl);

        NN_UTIL_SCOPE_EXIT
        {
            curl_multi_remove_handle(pCurlMulti, m_pCurl);
            curl_multi_cleanup(pCurlMulti);
        };

        {
            int runningCount = 0;

            CURLMcode multiCode = curl_multi_perform(pCurlMulti, &runningCount);
            NN_RESULT_THROW_UNLESS(multiCode == CURLM_OK, CurlMCodeToNnResult(multiCode));

            bool isSleepRequired = false;

            while (runningCount > 0)
            {
                auto cancelResult = CancelFlagHolder::GetSingleton().ConfirmConnectionConfirmationAdmitted(m_NetworkInterfaceType);
                if (cancelResult.IsFailure())
                {
                    NN_DETAIL_NIFM_INFO("HTTP communication is cancelled.\n");
                    NN_RESULT_THROW(cancelResult);
                }

                int fdCount = 0;

                multiCode = curl_multi_wait(pCurlMulti, NULL, 0, 100, &fdCount);
                NN_RESULT_THROW_UNLESS(multiCode == CURLM_OK, CurlMCodeToNnResult(multiCode));

                // fdCount が 0 の場合は、タイムアウトしたか、待つべき fd が存在しないか
                if (fdCount == 0)
                {
                    // curl_multi_wait() が2回続けて fdCount に 0 を返したらウェイトする
                    if (isSleepRequired)
                    {
                        // 待つべき fd がないときは次の curl_multi_perform() の返す
                        // runningCount が 0 になると思われるので、
                        // ここを通ることはないと思うがサンプルに従ってウェイトを入れておく

                        NN_DETAIL_NIFM_TRACE("curl_multi_wait returned 0 as fdCount more than once.\n");
                        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                    }
                    isSleepRequired = true;
                }
                else
                {
                    isSleepRequired = false;
                }

                multiCode = curl_multi_perform(pCurlMulti, &runningCount);
                NN_RESULT_THROW_UNLESS(multiCode == CURLM_OK, CurlMCodeToNnResult(multiCode));
            }
        }

        {
            CURLcode easyCode = CURLE_OK;

            for (;;)
            {
                int messageCount;
                CURLMsg* pMessage = curl_multi_info_read(pCurlMulti, &messageCount);

                if (pMessage == nullptr)
                {
                    break;
                }

                if (pMessage->msg == CURLMSG_DONE)
                {
                    if (pMessage->easy_handle == m_pCurl)
                    {
                        easyCode = pMessage->data.result;
                    }
                }
            }

            NN_RESULT_DO(CurlCodeToNnResult(easyCode));
        }
    }

    long statusCode = 0;
    NN_RESULT_DO(CurlCodeToNnResult(curl_easy_getinfo(m_pCurl, CURLINFO_RESPONSE_CODE, &statusCode)));

    m_StatusCode = static_cast<uint16_t>(statusCode);

    NN_RESULT_SUCCESS;
}

void HttpClientBase::Clear() NN_NOEXCEPT
{
    m_StatusCode = 0;

    if (m_pHttpPostData != nullptr)
    {
        m_pHttpPostData->Clear();
    }

    if( m_pHttpResponse != nullptr)
    {
        m_pHttpResponse->Clear();
    }
}

}
}
}
