﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/core/connectionConfirmation/nifm_AuthenticationClient.horizon.h>

#include <nn/nifm/detail/util/nifm_Base64.h>

#include <nn/os/os_SdkMutex.h>

#include <nn/ssl/ssl_Context.h>

#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/settings/system/settings_Region.h>
#include <nn/settings/system/settings_SerialNumber.h>

#include <nn/nn_Version.h>
#include <nn/nn_Macro.h>

#include <algorithm>

namespace nn
{
namespace nifm
{
namespace detail
{

const char AuthenticationClient::AuthenticationUrl[] = "https://nasc.nintendowifi.net/ac";

namespace
{
#define NN_NIFM_SDK_VERSION_STRING(major, minor, micro, relstep) \
    # major "." # minor "." # micro "." # relstep

    const char* GetUserAgent() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(os::SdkMutexType, g_Mutex, = NN_OS_SDK_MUTEX_INITIALIZER());
        g_Mutex.Lock();
        NN_UTIL_SCOPE_EXIT
        {
            g_Mutex.Unlock();
        };

#define NN_NIFM_UA_AGENT     "libcurl"
#define NN_NIFM_UA_MODULE    "nnNifm"

#if defined(NN_BUILD_CONFIG_SPEC_GENERIC)
#define NN_NIFM_UA_PLATFORM  "GenericWindows" // Generic Windows（未使用）
#elif defined(NN_BUILD_CONFIG_SPEC_NX)
#if defined(NN_BUILD_CONFIG_OS_WIN)
#define NN_NIFM_UA_PLATFORM  "NXonWindows" // NX Windows（未使用）
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
#define NN_NIFM_UA_PLATFORM  "NX" // NX Horizon
#else
#   error "Unsupported OS"
#endif // OS
#else
#   error "Unsupported Spec"
#endif // Spec

        using UserAgentStringType = char[256];
        NN_FUNCTION_LOCAL_STATIC(UserAgentStringType, g_UserAgent, ={});

        NN_FUNCTION_LOCAL_STATIC(bool, g_Initialized, = false);
        if (!g_Initialized)
        {

#if defined(NN_BUILD_CONFIG_SPEC_NX)
            auto l = util::SNPrintf(
                g_UserAgent, sizeof(g_UserAgent),
                "%s (%s; %s; SDK %d.%d.%d.%d; Add-on %d.%d.%d.%d)",
                NN_NIFM_UA_AGENT, NN_NIFM_UA_MODULE, NN_NIFM_UA_PLATFORM,
                NN_SDK_VERSION_MAJOR, NN_SDK_VERSION_MINOR, NN_SDK_VERSION_MICRO, NN_SDK_VERSION_RELSTEP,
                NN_NX_ADDON_VERSION_MAJOR, NN_NX_ADDON_VERSION_MINOR, NN_NX_ADDON_VERSION_MICRO, NN_NX_ADDON_VERSION_RELSTEP);
#else
            auto l = util::SNPrintf(
                g_UserAgent, sizeof(g_UserAgent),
                "%s (%s; %s; SDK %d.%d.%d.%d)",
                NN_NIFM_UA_AGENT, NN_NIFM_UA_MODULE, NN_NIFM_UA_PLATFORM,
                NN_SDK_VERSION_MAJOR, NN_SDK_VERSION_MINOR, NN_SDK_VERSION_MICRO, NN_SDK_VERSION_RELSTEP);
#endif
            NN_ABORT_UNLESS(l < sizeof(g_UserAgent), "Error: Insufficient buffer for user agent: required %d bytes\n", l + 1);

            NN_DETAIL_NIFM_INFO("User-agent is initialized as \"%s\"\n", g_UserAgent);
            g_Initialized = true;
        }
        return g_UserAgent;
    }

    void GetSdkVersionString(char *pOutString, size_t size)
    {
#if defined(NN_BUILD_CONFIG_SPEC_NX)
        nn::util::SNPrintf(pOutString, size, "%02d%02d%02d", NN_NX_ADDON_VERSION_MAJOR, NN_NX_ADDON_VERSION_MINOR, NN_NX_ADDON_VERSION_MICRO);
#else
        nn::util::SNPrintf(pOutString, size, "%02d%02d%02d", NN_SDK_VERSION_MAJOR, NN_SDK_VERSION_MINOR, NN_SDK_VERSION_MICRO);
#endif
    }

    void ConvertToStringFromMacAddress(char* pOutString, size_t size, const MacAddress& macAddress)
    {
        NN_SDK_ASSERT(size >= sizeof("001122334455"));

        nn::util::SNPrintf(pOutString, size, "%02x%02x%02x%02x%02x%02x",
            macAddress.data[0], macAddress.data[1], macAddress.data[2],
            macAddress.data[3], macAddress.data[4], macAddress.data[5]);
    }

    nn::Result ConvertToResultFromReturnCode(char* pReturnCodeStr)
    {
        NN_SDK_ASSERT_NOT_NULL(pReturnCodeStr);
        NN_RESULT_THROW_UNLESS(pReturnCodeStr[0] != '\0', ResultNintendoHotspotAuthenticationResponseError());

        uint32_t returnCode = strtoul(pReturnCodeStr, nullptr, 10);

        NN_DETAIL_NIFM_INFO("returnCode=%d\n", returnCode);

        switch (returnCode)
        {
        case 4:
            NN_RESULT_THROW(ResultNintendoHotspotAuthenticationResponseParseSuccess());
        case 8:
            NN_RESULT_THROW(ResultNintendoHotspotAuthenticationResponseMessageSuccess());
        case 34:
            NN_RESULT_THROW(ResultNintendoHotspotAuthenticationResponseAlreadyAuthenticated());
        case 102:
            NN_RESULT_THROW(ResultNintendoHotspotAuthenticationResponseDeviceBanned());
        case 109:
            NN_RESULT_THROW(ResultNintendoHotspotAuthenticationResponseInvalidRequestParameter());
        case 501:
            NN_RESULT_THROW(ResultNintendoHotspotAuthenticationResponseUnknownError());
        case 503:
            NN_RESULT_THROW(ResultNintendoHotspotAuthenticationResponseParseError());
        default:
            NN_DETAIL_NIFM_WARN_V3("Unxepected return code : %d\n", returnCode);
            break;
        }

        NN_RESULT_THROW(ResultNintendoHotspotAuthenticationResponseError());
    }
}


AuthenticationClient::HttpHeaderReaderForAuth::HttpHeaderReaderForAuth() NN_NOEXCEPT
{
}

AuthenticationClient::HttpHeaderReaderForAuth::~HttpHeaderReaderForAuth() NN_NOEXCEPT
{
}

void AuthenticationClient::HttpHeaderReaderForAuth::Parse(const char* pInLine, size_t size) NN_NOEXCEPT
{
}

void AuthenticationClient::HttpHeaderReaderForAuth::Clear() NN_NOEXCEPT
{
}


size_t AuthenticationClient::ResponseBodyWriteCallbackStatic(void *pData, size_t size, size_t nmemb, void *pContext) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pContext);
    return reinterpret_cast<AuthenticationClient*>(pContext)->ResponseBodyWriteCallback(pData, size, nmemb);
}

size_t AuthenticationClient::ResponseBodyWriteCallback(void *pData, size_t size, size_t nmemb) NN_NOEXCEPT
{
    size_t count = size * nmemb;

    if (m_pHttpResponse == nullptr)
    {
        return 0;
    }

    if (count > 0)
    {
        auto writeSize = m_pHttpResponse->Add(reinterpret_cast<char*>(pData), count);

        NN_UNUSED(writeSize);
        NN_DETAIL_NIFM_INFO_V2("[%d/%d] %s", writeSize, m_pHttpResponse->GetLength(), &m_pHttpResponse->GetPointer()[m_pHttpResponse->GetLength() - writeSize]);

        // バッファに書き込めない応答が返った場合，RetrieveDataFromNasResponse か Hotspot 認証サーバの認証で失敗する
    }

    return count;
}

CURLcode AuthenticationClient::CurlSslContextCallbackStatic(CURL* curl, void* sslContext, void* pContext) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pContext);
    auto client = reinterpret_cast<AuthenticationClient*>(pContext);

    return client->CurlSslContextCallback(curl, sslContext, client->IsClientCertificateUsed());
}

CURLcode AuthenticationClient::CurlSslContextCallback(CURL* pCurl, void* sslContext, bool isClientCertificateUsed) NN_NOEXCEPT
{
    NN_UNUSED(pCurl);

    nn::ssl::Context* context = reinterpret_cast<nn::ssl::Context*>(sslContext);

    if (context->Create(nn::ssl::Context::SslVersion_Auto).IsFailure())
    {
        NN_DETAIL_NIFM_ERROR("ssl::Context::Create failed");
        return CURLE_ABORTED_BY_CALLBACK;
    }

    if(isClientCertificateUsed)
    {
        ssl::CertStoreId deviceCertId;
        nn::Result result = context->RegisterInternalPki(&deviceCertId, ssl::Context::InternalPki_DeviceClientCertDefault);
        if (result.IsFailure())
        {
            NN_DETAIL_NIFM_ERROR("Failed to register the device unique client certificate.\n");
        }
    }

    return CURLE_OK;
}

void AuthenticationClient::SetClientCertificateUsed(bool isClientCertificateUsed) NN_NOEXCEPT
{
    m_ClientCertificateUsed = isClientCertificateUsed;
}

bool AuthenticationClient::IsClientCertificateUsed() NN_NOEXCEPT
{
    return m_ClientCertificateUsed;
}

AuthenticationClient::AuthenticationClient(const HttpResponse* pResponse) NN_NOEXCEPT
    : m_pConnTestResponse(pResponse)
{
    SetSequencePeriod(SequencePeriod::AuthenticationNas);
}

AuthenticationClient::~AuthenticationClient() NN_NOEXCEPT
{
}

nn::Result AuthenticationClient::AuthenticateNas(const char* pUrl) NN_NOEXCEPT
{
    m_pHttpResponse->Clear();

    // 共通の設定
    SetCurlCommonOpt();

    // 独自の設定
    SetClientCertificateUsed(true);
    SetRequestUrl(pUrl);

    SetRequesetMethod(HttpRequestMethod::Post);

    char sdkVersionString[7];
    GetSdkVersionString(sdkVersionString, sizeof(sdkVersionString));

    char macAddressString[MacAddress::Size * 2 + 1];
    ConvertToStringFromMacAddress(macAddressString, sizeof(macAddressString), m_MacAddress);

    char bssidString[MacAddress::Size * 2 + 1];
    ConvertToStringFromMacAddress(bssidString, sizeof(bssidString), m_Bssid);

    nn::settings::system::RegionCode regionCode;
    nn::settings::system::GetRegionCode(&regionCode);

    char regionCodeString[3];
    nn::util::SNPrintf(regionCodeString, sizeof(regionCodeString), "%02d", regionCode);

    nn::settings::system::SerialNumber serialNumber;
    nn::settings::system::GetSerialNumber(&serialNumber);
    int serialNumberLen = nn::util::Strnlen(serialNumber.string, sizeof(serialNumber.string));
    const int serialNumberLenMax = 15; // サーバー仕様
    NN_SDK_ASSERT_LESS_EQUAL(serialNumberLen, serialNumberLenMax);
    serialNumberLen = std::min(serialNumberLen, serialNumberLenMax);

    NN_RESULT_DO(AddPostData("action", "parse", sizeof("parse") - 1));
    NN_RESULT_DO(AddPostData("HTML", m_pConnTestResponse->GetPointer(), m_pConnTestResponse->GetLength()));
    NN_RESULT_DO(AddPostData("sdkver", sdkVersionString, sizeof(sdkVersionString) - 1));
    NN_RESULT_DO(AddPostData("unitcd", "3", 1));
    NN_RESULT_DO(AddPostData("macadr", macAddressString, sizeof(macAddressString) - 1));
    NN_RESULT_DO(AddPostData("bssid", bssidString, sizeof(bssidString) - 1));
    NN_RESULT_DO(AddPostData("csnum", serialNumber.string, serialNumberLen));
    NN_RESULT_DO(AddPostData("region", regionCodeString, sizeof(regionCodeString) - 1));
    NN_RESULT_DO(AddPostData("ssid", reinterpret_cast<const char*>(m_Ssid.hex), m_Ssid.length));

    NIFM_DETAIL_CURL_EASY_SETOPT(m_pCurl, CURLOPT_POSTFIELDS, m_pHttpPostData->GetPointer());
    NIFM_DETAIL_CURL_EASY_SETOPT(m_pCurl, CURLOPT_POSTFIELDSIZE, m_pHttpPostData->GetLength());

    NIFM_DETAIL_CURL_EASY_SETOPT(m_pCurl, CURLOPT_WRITEFUNCTION, ResponseBodyWriteCallbackStatic);
    NIFM_DETAIL_CURL_EASY_SETOPT(m_pCurl, CURLOPT_WRITEDATA, this);
    NIFM_DETAIL_CURL_EASY_SETOPT(m_pCurl, CURLOPT_SSL_CTX_FUNCTION, CurlSslContextCallbackStatic);
    NIFM_DETAIL_CURL_EASY_SETOPT(m_pCurl, CURLOPT_SSL_CTX_DATA, this);

    NIFM_DETAIL_CURL_EASY_SETOPT(m_pCurl, CURLOPT_USERAGENT, GetUserAgent()); // NAS に送る User-Agent は上書き

#if defined(NN_DETAIL_NIFM_LOG_ENABLED)
    NIFM_DETAIL_CURL_EASY_SETOPT(m_pCurl, CURLOPT_VERBOSE, 1L);
#endif

    struct curl_slist *chunk = NULL;
    chunk = curl_slist_append(chunk, "Expect:");

    NN_UTIL_SCOPE_EXIT
    {
        if (chunk)
        {
            curl_slist_free_all(chunk);
        }
    };

    NIFM_DETAIL_CURL_EASY_SETOPT(m_pCurl, CURLOPT_HTTPHEADER, chunk);

    NN_RESULT_DO(Connect());

    switch (m_StatusCode)
    {
    case 200:
        break;
    case 301:
        NN_RESULT_THROW(ResultNintendoHotspotAuthenticationHttpStatusMovedPermanently());
    case 302:
        NN_RESULT_THROW(ResultNintendoHotspotAuthenticationHttpStatusFound());
    case 400:
        NN_RESULT_THROW(ResultNintendoHotspotAuthenticationHttpStatusError());
    case 407:
        NN_RESULT_THROW(ResultNintendoHotspotAuthenticationHttpStatusProxyAuthenticationRequired());
    default:
        NN_DETAIL_NIFM_WARN_V3("AuthenticateNas: StatusCode=%d\n", m_StatusCode);
        NN_RESULT_THROW(ResultNintendoHotspotAuthenticationHttpStatusError());
    }

    NN_RESULT_SUCCESS;
}

nn::Result AuthenticationClient::AuthenticateHotspot(const char* pUrl) NN_NOEXCEPT
{
    m_pHttpResponse->Clear();

    // 共通の設定
    SetCurlCommonOpt();

    // 独自の設定
    SetClientCertificateUsed(false);
    SetRequestUrl(pUrl);

    SetRequesetMethod(HttpRequestMethod::Post);

    NIFM_DETAIL_CURL_EASY_SETOPT(m_pCurl, CURLOPT_POSTFIELDS, m_pHttpPostData->GetPointer());
    NIFM_DETAIL_CURL_EASY_SETOPT(m_pCurl, CURLOPT_POSTFIELDSIZE, m_pHttpPostData->GetLength());

    NIFM_DETAIL_CURL_EASY_SETOPT(m_pCurl, CURLOPT_SSL_CTX_FUNCTION, CurlSslContextCallbackStatic);
    NIFM_DETAIL_CURL_EASY_SETOPT(m_pCurl, CURLOPT_SSL_CTX_DATA, this);

#if defined(NN_DETAIL_NIFM_LOG_ENABLED)
    NIFM_DETAIL_CURL_EASY_SETOPT(m_pCurl, CURLOPT_VERBOSE, 1L);
#endif

    NN_RESULT_DO(Connect());

    // 通常 200/302 が返るはずだが，CTR の仕様に従い StatusCode は無視し，
    // 再度の疎通確認により成功，失敗を確認する
    NN_DETAIL_NIFM_TRACE_V3("AuthenticateHotspot: StatusCode=%d\n", m_StatusCode);

    NN_RESULT_SUCCESS;
}

void AuthenticationClient::Clear() NN_NOEXCEPT
{
    m_ContentSize = 0;
    HttpClientBase::Clear();
}

nn::Result AuthenticationClient::AddPostData(const char* pKey, const char* pValue, int size) NN_NOEXCEPT
{
    return m_pHttpPostData->EncodeAdd(pKey, pValue, size, Base64::Mode_NmtokenPad);
}

nn::Result AuthenticationClient::RetrieveDataFromNasResponse(char* pOutUrl, size_t urlSize) NN_NOEXCEPT
{
    Reset();
    m_pHttpPostData->Clear();

    char* str = const_cast<char*>(m_pHttpResponse->GetPointer());

    char returnCodeStr[4] = {};

    char* pch = strtok(str, "=&");
    while (pch != nullptr)
    {
        if (nn::util::Strncmp("data", pch, sizeof("data")) == 0)
        {
            pch = strtok(nullptr, "=&");
            NN_RESULT_DO(m_pHttpPostData->DecodeAdd(pch, Base64::Mode_NmtokenPad));
        }
        else if (nn::util::Strncmp("interval", pch, sizeof("interval")) == 0)
        {
        }
        else if (nn::util::Strncmp("retry", pch, sizeof("retry")) == 0)
        {
        }
        else if (nn::util::Strncmp("returncd", pch, sizeof("returncd")) == 0)
        {
            pch = strtok(nullptr, "=&");
            size_t count;
            auto status = Base64::FromBase64String(&count, &returnCodeStr, sizeof(returnCodeStr), pch, Base64::Mode_NmtokenPad);
            NN_RESULT_THROW_UNLESS(status == Base64::Status_Success, ResultNintendoHotspotAuthenticationResponseError());
        }
        else if (nn::util::Strncmp("url", pch, sizeof("url")) == 0)
        {
            pch = strtok(nullptr, "=&");
            size_t count;
            auto status = Base64::FromBase64String(&count, pOutUrl, urlSize, pch, Base64::Mode_NmtokenPad);
            NN_RESULT_THROW_UNLESS(status == Base64::Status_Success, ResultNintendoHotspotAuthenticationResponseError());
        }
        else if (nn::util::Strncmp("datetime", pch, sizeof("datetime")) == 0)
        {
        }

        pch = strtok(nullptr, "=&");
    }

    NN_RESULT_THROW(ConvertToResultFromReturnCode(returnCodeStr));
}

nn::Result AuthenticationClient::Authenticate(const char* pUrl) NN_NOEXCEPT
{
    SetSequencePeriod(SequencePeriod::AuthenticationNas);

    // 任天堂認証サーバー
    NN_RESULT_DO(AuthenticateNas(pUrl));

    char url[256] = {};    // サーバー仕様では data:max char(100)

    // 応答解析
    nn::Result result = RetrieveDataFromNasResponse(url, sizeof(url));
    if (result <= ResultNintendoHotspotAuthenticationResponseParseSuccess())
    {
        SetSequencePeriod(SequencePeriod::AuthenticationHotspot);

        // HOTSPOT 認証サーバー
        result = AuthenticateHotspot(url);
    }
    else if (result <= ResultNintendoHotspotAuthenticationResponseAlreadyAuthenticated())
    {
        // 認証済み
        result = ResultSuccess();
    }

    NN_RESULT_THROW(result);
}

void AuthenticationClient::SetMacAddress(const MacAddress& macAddress) NN_NOEXCEPT
{
    m_MacAddress = macAddress;
}

void AuthenticationClient::SetBssid(const MacAddress& bssid) NN_NOEXCEPT
{
    m_Bssid = bssid;
}

void AuthenticationClient::SetSsid(const Ssid& ssid) NN_NOEXCEPT
{
    m_Ssid = ssid;
}

}
}
}
