﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <cstring>
#include <algorithm>
#include <iterator>
#include <memory>

#include "../detail/ngc_WorkBufAllocator.h"
#include "./ngc_ErrnoT.h"

namespace nn { namespace ngc { namespace detail {

// PODを要素に持つベクタをreallocベースで実装しています。
template <class T>
class ReallocVec
{
public:
    typedef void* (*ReallocFunc)(void*, size_t);
    typedef T value_type;
    typedef size_t size_type;
    typedef ptrdiff_t difference_type;
    typedef T& reference;
    typedef const T& const_reference;
    typedef T* pointer;
    typedef const T* const_pointer;
    typedef T* iterator;
    typedef const T* const_iterator;
    typedef std::reverse_iterator<iterator> reverse_iterator;
    typedef std::reverse_iterator<const_iterator> const_reverse_iterator;
    ReallocVec() NN_NOEXCEPT : m_Vec(NULL),
                               m_Cur(0),
                               m_Size(0),
                               m_Realloc(ReallocateMemoryNgc),
                               m_pAllocator(NULL){}
    explicit ReallocVec(nn::ngc::detail::WorkBufAllocator* pAllocator) NN_NOEXCEPT : m_Vec(NULL),
                                                                                     m_Cur(0),
                                                                                     m_Size(0),
                                                                                     m_Realloc(ReallocateMemoryNgc),
                                                                                     m_pAllocator(pAllocator){}
    explicit ReallocVec(ReallocFunc func) NN_NOEXCEPT;
    ~ReallocVec() NN_NOEXCEPT
    {
        if (m_pAllocator)
        {
            NN_SDK_LOG("ERROR: Please call ReallocVec::ReleaseAllocator() before calling ReallocVec::~ReallocVec()\n");
            NN_ABORT();
        }
        else if (m_Vec)
        {
            m_Realloc(m_Vec, 0);
        }
        NN_STATIC_ASSERT(std::is_pod<T>::value);
    }
    inline ReallocVec(ReallocVec&& rhs) : m_Vec(), m_Cur(), m_Size(), m_Realloc(), m_pAllocator() { this->swap(rhs); }
    inline ReallocVec& operator=(ReallocVec&& rhs) NN_NOEXCEPT
    {
        ReallocVec tmp(std::move(rhs));
        this->swap(tmp);
        return *this;
    }

    ReallocFunc GetRealloc() const NN_NOEXCEPT
    {
        return m_Realloc;
    }
    bool PushBack(const T& v) NN_NOEXCEPT
    {
        if (m_Cur == m_Size)
        {
            size_t newsize = (m_Size == 0) ? 1 : m_Size * 2;
            if (newsize < m_Size)
            {
                return false;
            }
            if (!this->Expand(newsize))
            {
                return false;
            }
        }
        m_Vec[m_Cur++] = v;
        return true;
    }
    bool PopBack() NN_NOEXCEPT
    {
        if (m_Cur == 0)
        {
            return false;
        }
        --m_Cur;
        return true;
    }
    T& operator[](size_t idx)
    {
        return m_Vec[idx];
    }
    const T& operator[](size_t idx) const
    {
        return m_Vec[idx];
    }
    T& Front()
    {
        return *Begin();
    }
    const T& Front() const
    {
        return *Begin();
    }
    T& Back()
    {
        return *(Begin() + m_Cur - 1);
    }
    const T& Back() const
    {
        return *(Begin() + m_Cur - 1);
    }

    bool SetAllocator(WorkBufAllocator* pAllocator) NN_NOEXCEPT
    {
        if (!pAllocator)
        {
            return false;
        }
        m_pAllocator = pAllocator;
        return true;
    }
    void ReleaseAllocator() NN_NOEXCEPT
    {
        if (!m_pAllocator)
        {
            return;
        }
        if (m_Vec)
        {
            m_pAllocator->Free(m_Vec);
            m_Vec = NULL;
        }
        m_pAllocator = NULL;
    }

    iterator Begin() NN_NOEXCEPT
    {
        return m_Vec;
    }
    const_iterator Begin() const NN_NOEXCEPT
    {
        return m_Vec;
    }
    const_iterator Cbegin() const NN_NOEXCEPT
    {
        return m_Vec;
    }
    iterator End() NN_NOEXCEPT
    {
        return Begin() + m_Cur;
    }
    const_iterator End() const NN_NOEXCEPT
    {
        return Begin() + m_Cur;
    }
    const_iterator Cend() const NN_NOEXCEPT
    {
        return Begin() + m_Cur;
    }
    reverse_iterator Rbegin() NN_NOEXCEPT
    {
        return reverse_iterator(End());
    }
    const_reverse_iterator Rbegin() const NN_NOEXCEPT
    {
        return const_reverse_iterator(End());
    }
    reverse_iterator Rend() NN_NOEXCEPT
    {
        return reverse_iterator(Begin());
    }
    const_reverse_iterator Rend() const NN_NOEXCEPT
    {
        return const_reverse_iterator(Begin());
    }
    const_reverse_iterator Crbegin() const NN_NOEXCEPT
    {
        return const_reverse_iterator(End());
    }
    const_reverse_iterator Crend() const NN_NOEXCEPT
    {
        return const_reverse_iterator(Begin());
    }
    void Swap(ReallocVec& rhs) NN_NOEXCEPT
    {
        using std::swap;
        swap(m_Vec, rhs.m_Vec);
        swap(m_Cur, rhs.m_Cur);
        swap(m_Size, rhs.m_Size);
        swap(m_Realloc, rhs.m_Realloc);
        nn::ngc::detail::WorkBufAllocator* pAllocatorTmp = rhs.m_pAllocator;
        rhs.m_pAllocator = m_pAllocator;
        m_pAllocator = pAllocatorTmp;
    }
    size_t Size() const NN_NOEXCEPT
    {
        return m_Cur;
    }
    size_t Capacity() const NN_NOEXCEPT
    {
        return m_Size;
    }
    bool Empty() const NN_NOEXCEPT
    {
        return m_Cur == 0;
    }
    bool Reserve(size_t n) NN_NOEXCEPT
    {
        return (n > m_Size) ? this->Expand(n) : true;
    }
    bool Resize(size_t n) NN_NOEXCEPT
    {
        if (!this->Reserve(n))
        {
            return false;
        }
        m_Cur = n;
        return true;
    }
    void Clear() NN_NOEXCEPT
    {
        m_Cur = 0;
    }
    void ShrinkToFit() NN_NOEXCEPT
    {
        if (m_Cur != m_Size)
        {
            if (m_Cur > 0)
            {
                this->Expand(m_Cur);
            }
            else
            {
                if (m_pAllocator)
                {
                    m_pAllocator->Reallocate(m_Vec, 0);
                }
                else
                {
                    m_Realloc(m_Vec, 0);
                }
                m_Vec = NULL;
                m_Size = 0;
            }
        }
    }
    iterator Insert(iterator pos, const T& val)
    {
        ptrdiff_t idx = pos - Begin();
        if (!this->push_back(val))
        {
            return NULL;
        }
        std::rotate(Begin() + idx, &Back(), End());
        return Begin() + idx;
    }

private:
    bool Expand(size_t newSize) NN_NOEXCEPT;

    T* m_Vec;
    size_t m_Cur;
    size_t m_Size;
    ReallocFunc m_Realloc;
    nn::ngc::detail::WorkBufAllocator* m_pAllocator;

    ReallocVec(const ReallocVec&) = delete;
    void operator=(const ReallocVec&) = delete;
};

template <class T>
inline ReallocVec<T>::ReallocVec(ReallocFunc func) NN_NOEXCEPT : m_Vec(NULL),
                                                                 m_Cur(0),
                                                                 m_Size(0),
                                                                 m_Realloc(func),
                                                                 m_pAllocator(NULL) {}

template <class T>
bool ReallocVec<T>::Expand(size_t newSize) NN_NOEXCEPT
{
    T* ptr = m_Vec;
    void* new_ptr;
    if (m_pAllocator)
    {
        new_ptr = m_pAllocator->Reallocate(ptr, newSize * sizeof(T));
    }
    else
    {
        new_ptr = m_Realloc(ptr, newSize * sizeof(T));
    }
    if (!new_ptr)
    {
        return false;
    }
    m_Vec = reinterpret_cast<T*>(new_ptr);
    m_Size = newSize;
    return true;
}

template<class AL>
inline void swap(nn::ngc::detail::ReallocVec<AL>& lhs, nn::ngc::detail::ReallocVec<AL>& rhs)
{
    lhs.Swap(rhs);
}

}}} // nn::ngc::detail
