﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "./ngc_BinaryWriter.h"
#include "./ngc_OutputStream.h"

namespace nn { namespace ngc { namespace detail {

#define IS_WRITING_LITTLE_ENDIAN !m_SwapEndian

errno_t BinaryWriter::Init(EndianSetting endian) NN_NOEXCEPT
{
    if (m_Errno != -1)
    {
        return EALREADY;
    }
    m_SwapEndian = (endian == EndianSetting_EndianBig);
    m_Errno = 0;
    return 0;
}

errno_t BinaryWriter::Open(OutputStream* stream) NN_NOEXCEPT
{
    if (m_Errno != 0)
    {
        if (m_Errno != -1)
        {
            return EINVAL;
        }
        errno_t e = Init();
        if (e != 0)
        {
            return e;
        }
    }
    if (!stream)
    {
        return EINVAL;
    }
    if (m_Stream)
    {
        return EALREADY;
    }
    m_Stream = stream;
    return 0;
}

bool BinaryWriter::WriteStream() NN_NOEXCEPT
{
    if (!m_MiniBuf.Flush(m_Stream))
    {
        this->SetError(EIO);
        return false;
    }
    return true;
}

bool BinaryWriter::Write_(uint8_t x) NN_NOEXCEPT
{
    if (!m_MiniBuf.CheckAndFlush(m_Stream, sizeof(x)))
    {
        this->SetError(EIO);
        return false;
    }
    m_MiniBuf[0] = x;
    m_MiniBuf.Advance(sizeof(x));
    return true;
}

bool BinaryWriter::Write_(uint16_t x) NN_NOEXCEPT
{
    if (!m_MiniBuf.CheckAndFlush(m_Stream, sizeof(x)))
    {
        this->SetError(EIO);
        return false;
    }
    if (IS_WRITING_LITTLE_ENDIAN)
    {
        m_MiniBuf[0] = static_cast<unsigned char>(x & 0xFF);
        m_MiniBuf[1] = static_cast<unsigned char>(x >> 8);
    }
    else
    {
        m_MiniBuf[0] = static_cast<unsigned char>(x >> 8);
        m_MiniBuf[1] = static_cast<unsigned char>(x & 0xFF);
    }
    m_MiniBuf.Advance(sizeof(x));
    return true;
}

bool BinaryWriter::Write_(uint32_t x) NN_NOEXCEPT
{
    if (!m_MiniBuf.CheckAndFlush(m_Stream, sizeof(x)))
    {
        this->SetError(EIO);
        return false;
    }
    if (IS_WRITING_LITTLE_ENDIAN)
    {
        m_MiniBuf[0] = static_cast<unsigned char>(x & 0xFFU);
        m_MiniBuf[1] = static_cast<unsigned char>((x >> 8) & 0xFFU);
        m_MiniBuf[2] = static_cast<unsigned char>((x >> 16) & 0xFFU);
        m_MiniBuf[3] = static_cast<unsigned char>((x >> 24) & 0xFFU);
    }
    else
    {
        m_MiniBuf[0] = static_cast<unsigned char>((x >> 24) & 0xFFU);
        m_MiniBuf[1] = static_cast<unsigned char>((x >> 16) & 0xFFU);
        m_MiniBuf[2] = static_cast<unsigned char>((x >> 8) & 0xFFU);
        m_MiniBuf[3] = static_cast<unsigned char>(x & 0xFFU);
    }
    m_MiniBuf.Advance(sizeof(x));
    return true;
}

bool BinaryWriter::Write_(uint64_t x) NN_NOEXCEPT
{
    if (!m_MiniBuf.CheckAndFlush(m_Stream, sizeof(x)))
    {
        this->SetError(EIO);
        return false;
    }
    if (IS_WRITING_LITTLE_ENDIAN)
    {
        m_MiniBuf[0] = static_cast<uint8_t>(x & 0xFFU);
        m_MiniBuf[1] = static_cast<uint8_t>((x >> 8) & 0xFFU);
        m_MiniBuf[2] = static_cast<uint8_t>((x >> 16) & 0xFFU);
        m_MiniBuf[3] = static_cast<uint8_t>((x >> 24) & 0xFFU);

        m_MiniBuf[4] = static_cast<uint8_t>((x >> 32) & 0xFFU);
        m_MiniBuf[5] = static_cast<uint8_t>((x >> 40) & 0xFFU);
        m_MiniBuf[6] = static_cast<uint8_t>((x >> 48) & 0xFFU);
        m_MiniBuf[7] = static_cast<uint8_t>((x >> 56) & 0xFFU);
    }
    else
    {
        m_MiniBuf[0] = static_cast<uint8_t>((x >> 56) & 0xFFU);
        m_MiniBuf[1] = static_cast<uint8_t>((x >> 48) & 0xFFU);
        m_MiniBuf[2] = static_cast<uint8_t>((x >> 40) & 0xFFU);
        m_MiniBuf[3] = static_cast<uint8_t>((x >> 32) & 0xFFU);

        m_MiniBuf[4] = static_cast<uint8_t>((x >> 24) & 0xFFU);
        m_MiniBuf[5] = static_cast<uint8_t>((x >> 16) & 0xFFU);
        m_MiniBuf[6] = static_cast<uint8_t>((x >> 8) & 0xFFU);
        m_MiniBuf[7] = static_cast<uint8_t>(x & 0xFFU);
    }
    m_MiniBuf.Advance(sizeof(x));
    return true;
}

bool BinaryWriter::Write_(float x) NN_NOEXCEPT
{
    union {
        float x;
        uint32_t u;
    } tmp;
    tmp.x = x;
    return this->Write_(tmp.u);
}

bool BinaryWriter::Write_(double x) NN_NOEXCEPT
{
    union {
        double x;
        uint64_t u;
    } tmp;
    tmp.x = x;
    return this->Write_(tmp.u);
}

bool BinaryWriter::WriteArray(const unsigned char* x, size_t n) NN_NOEXCEPT
{
    if (n > RSIZE_MAX)
    {
        this->SetError(EINVAL);
        return false;
    }
    if (n <= m_MiniBuf.BufSize / 2)
    {
        if (m_MiniBuf.WriteAndAdvance(m_Stream, x, n))
        {
            return true;
        }
    }
    else
    {
        if (!this->WriteStream())
        {
            return false;
        }
        if (m_Stream->Write(x, n))
        {
            return true;
        }
    }
    this->SetError(EIO);
    return false;
}

bool BinaryWriter::WriteArray(const unsigned short* x, size_t n) NN_NOEXCEPT
{  // NOLINT
    if (!m_SwapEndian)
    {
        return this->WriteArray(reinterpret_cast<const uint8_t*>(x), n * sizeof(*x));
    }
    if (n > RSIZE_MAX)
    {
        this->SetError(EINVAL);
        return false;
    }
    if (!this->WriteStream())
    {
        return false;
    }
    const size_t elem_size = m_MiniBuf.BufSize / sizeof(*x);
    while (n > 0)
    {
        size_t nn = (n >= elem_size) ? elem_size : n;
        MemCpy(&m_MiniBuf[0], m_MiniBuf.BufSize, x, nn * sizeof(*x));
        SwapEndian16(reinterpret_cast<uint16_t*>(&m_MiniBuf[0]), nn);
        if (!m_Stream->Write(&m_MiniBuf[0], nn * sizeof(*x)))
        {
            this->SetError(EIO);
            return false;
        }
        x += nn;
        n -= nn;
    }
    return true;
}

bool BinaryWriter::WriteArray(const unsigned int* x, size_t n) NN_NOEXCEPT
{
    if (!m_SwapEndian)
    {
        return this->WriteArray(reinterpret_cast<const uint8_t*>(x), n * sizeof(*x));
    }
    if (n > RSIZE_MAX)
    {
        this->SetError(EINVAL);
        return false;
    }
    if (!this->WriteStream())
    {
        return false;
    }
    const size_t elem_size = m_MiniBuf.BufSize / sizeof(*x);
    while (n > 0)
    {
        size_t nn = (n >= elem_size) ? elem_size : n;
        MemCpy(&m_MiniBuf[0], m_MiniBuf.BufSize, x, nn * sizeof(*x));
        SwapEndian32(reinterpret_cast<uint32_t*>(&m_MiniBuf[0]), nn);
        if (!m_Stream->Write(&m_MiniBuf[0], nn * sizeof(*x)))
        {
            this->SetError(EIO);
            return false;
        }
        x += nn;
        n -= nn;
    }
    return true;
}

bool BinaryWriter::WriteArray(const unsigned long long* x, size_t n) NN_NOEXCEPT
{  // NOLINT
    if (!m_SwapEndian)
    {
        return this->WriteArray(reinterpret_cast<const uint8_t*>(x), n * sizeof(*x));
    }
    if (n > RSIZE_MAX)
    {
        this->SetError(EINVAL);
        return false;
    }
    if (!this->WriteStream())
    {
        return false;
    }
    const size_t elem_size = m_MiniBuf.BufSize / sizeof(*x);
    while (n > 0)
    {
        size_t nn = (n >= elem_size) ? elem_size : n;
        MemCpy(&m_MiniBuf[0], m_MiniBuf.BufSize, x, nn * sizeof(*x));
        SwapEndian64(reinterpret_cast<uint64_t*>(&m_MiniBuf[0]), nn);
        if (!m_Stream->Write(&m_MiniBuf[0], nn * sizeof(*x)))
        {
            this->SetError(EIO);
            return false;
        }
        x += nn;
        n -= nn;
    }
    return true;
}

bool BinaryWriter::WriteArray(const float* x, size_t n) NN_NOEXCEPT
{
    // GHSコンパイラを黙らせるためにキャストをかます。
    const uint32_t* tmp = (const uint32_t*)(const void*)x;  // NOLINT
    return this->WriteArray(tmp, n);
}

bool BinaryWriter::WriteArray(const double* x, size_t n) NN_NOEXCEPT
{
    // GHSコンパイラを黙らせるためにキャストをかます。
    const uint64_t* tmp = (const uint64_t*)(const void*)x;  // NOLINT
    return this->WriteArray(tmp, n);
}

bool BinaryWriter::Flush() NN_NOEXCEPT
{
    if (!this->WriteStream()) return false;
    if (!m_Stream->Flush())
    {
        SetError(EIO);
        return false;
    }
    return true;
}

bool BinaryWriter::Close() NN_NOEXCEPT
{
    OutputStream* stream = m_Stream;
    m_Stream = NULL;  // NOTE: 失敗してもベースストリームは切り離される
                      // NOTE: ベースストリームはFlushしない
    return !stream || m_MiniBuf.Flush(stream);
}

}}} // nn::ngc::detail
