﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <typeinfo>
#include <type_traits>

// Programs/Iris/Resources/NgWord2 の生成ツールからは Chris のライブラリはリンク
// しないため分岐する
#if !defined(NGC_PCTOOL)
#  include <nn/lmem/lmem_ExpHeap.h>
#else
#  include <nn/nn_Common.h>
#endif

namespace nn { namespace ngc { namespace detail {

/**
 * @brief   ワークバッファの確保／解放のためのアロケータです
 * @details 内部実装は nn::lmem::ExpHeap です
 */
class WorkBufAllocator {
public:
#if !defined(NGC_PCTOOL)
    WorkBufAllocator() NN_NOEXCEPT : m_Handle(NULL) {}
#else
    WorkBufAllocator() NN_NOEXCEPT {}
#endif
    ~WorkBufAllocator() NN_NOEXCEPT
    {
        Finalize();
    }

    /**
     * @brief   アロケータを初期化します
     */
    bool Initialize(void* ptr, size_t size) NN_NOEXCEPT ;

    /**
     * @brief   アロケータを破棄します
     */
    void Finalize() NN_NOEXCEPT ;

    /**
     * @brief   メモリを確保します
     */
    void* Allocate(size_t size) NN_NOEXCEPT ;

    void* Allocate(size_t size, int alignment) NN_NOEXCEPT ;

    /**
     * @brief   メモリを解放します
     */
    void Free(void* ptr) const NN_NOEXCEPT ;

    /**
     * @brief   メモリを新たなサイズで再確保します
     */
    void* Reallocate(void* ptr, size_t newSize) NN_NOEXCEPT ;

private:
#if !defined(NGC_PCTOOL)
    nn::lmem::HeapHandle m_Handle;
#endif
};

/**
 * @brief   POD 配列型を WorkBufAllocator で管理するためのラップクラスです
 * @details std::unique_ptr (+カスタムデリータ)で管理されるポインタを
 *          各プラットフォームで扱えるよう汎用化したもの。
 *          インターフェイスは unique_ptr に合わせる。
 *          ポインタは基本 unique_ptr で管理するが、解放時に内部で使っている
 *          WorkBufAllocator を呼び出す。
 *          VisualStudio2013 の unique_ptr にて独自定義の Deleter クラスを渡すための
 *          定義が削除されているため代わりに作成。
 *          (VisualStudio2013 がサポート外になったらこのクラスを削除することが可能)
 */
template <typename T>
class ArrayManager {
public:
    // 何もしない Deleter
    // 確保・解放は自前で行う
    struct NullDeleter
    {
        void operator ()(void* ptr)
        {
            NN_UNUSED(ptr);
        }
    };
public:
    ArrayManager() NN_NOEXCEPT : m_pAllocator(NULL), uniquePtr()
    {
        NN_STATIC_ASSERT(std::is_pod<T>::value);
    }
    explicit ArrayManager(void* ptr) NN_NOEXCEPT : m_pAllocator(NULL), uniquePtr()
    {
        NN_STATIC_ASSERT(std::is_pod<T>::value);
        uniquePtr.reset(ptr);
    }
    ArrayManager(void* ptr, WorkBufAllocator* pAllocator) NN_NOEXCEPT :
        m_pAllocator(pAllocator), uniquePtr()
    {
        NN_STATIC_ASSERT(std::is_pod<T>::value);
        uniquePtr.reset(ptr);
    }
    ArrayManager(ArrayManager&& ptrManager) NN_NOEXCEPT : m_pAllocator(), uniquePtr()
    {
        m_pAllocator = ptrManager.m_pAllocator;
        uniquePtr = std::move(ptrManager.uniquePtr);
        ptrManager.m_pAllocator = NULL;
    }
    ~ArrayManager() NN_NOEXCEPT
    {
        Reset();
    }

    void Swap(ArrayManager& arrayManager) NN_NOEXCEPT
    {
        std::swap(uniquePtr, arrayManager.uniquePtr);
        WorkBufAllocator* pAllocatorTmp = arrayManager.m_pAllocator;
        arrayManager.m_pAllocator = m_pAllocator;
        m_pAllocator = pAllocatorTmp;
    }

    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        if (uniquePtr)
        {
            return true;
        }
        return false;
    }
    ArrayManager& operator=(ArrayManager&& ptrManager) NN_NOEXCEPT
    {
        Free();
        m_pAllocator = ptrManager.m_pAllocator;
        uniquePtr = std::move(ptrManager.uniquePtr);
        ptrManager.m_pAllocator = NULL;
    }
    bool operator!() NN_NOEXCEPT {
        return !uniquePtr;
    }
    T& operator[](size_t index) const NN_NOEXCEPT {
        return uniquePtr[index];
    }

    T* Get() const NN_NOEXCEPT
    {
        return uniquePtr.get();
    }
    T* Release() NN_NOEXCEPT
    {
        T* ptr = uniquePtr.release();
        Reset();
        return ptr;
    }
    void Reset() NN_NOEXCEPT
    {
        Free();
        uniquePtr.reset();
        m_pAllocator = NULL;
    }
    void Reset(T* ptr) NN_NOEXCEPT
    {
        Reset();
        uniquePtr.reset(ptr);
    }
    void Reset(T* ptr, WorkBufAllocator* pAllocator) NN_NOEXCEPT
    {
        Reset();
        uniquePtr.reset(ptr);
        m_pAllocator = pAllocator;
    }
private:
    void Free() NN_NOEXCEPT
    {
        if (m_pAllocator)
        {
            m_pAllocator->Free(uniquePtr.get());
        }
        else if(uniquePtr.get())
        {
            delete[] uniquePtr.get();
        }
    }
private:
    WorkBufAllocator* m_pAllocator;                 // ポインタの確保に使われるアロケータ
    std::unique_ptr<T[], NullDeleter> uniquePtr;    // ポインタ管理用 unique_ptr
private:
    // コピーの禁止
    ArrayManager(const ArrayManager&) = delete;
    void operator=(const ArrayManager&) = delete;
};

/**
 * @brief   POD 型を WorkBufAllocator で管理するためのラップクラスです
 * @details std::unique_ptr (+カスタムデリータ)で管理されるポインタを
 *          各プラットフォームで扱えるよう汎用化したもの。
 *          インターフェイスは unique_ptr に合わせる。
 *          ポインタは基本 unique_ptr で管理するが、解放時に内部で使っている
 *          WorkBufAllocator を呼び出す。
 *          VisualStudio2013 の unique_ptr にて独自定義の Deleter クラスを渡すための
 *          定義が削除されているため代わりに作成。
 *          (VisualStudio2013 がサポート外になったらこのクラスを削除することが可能)
 */
template <typename T>
class PodManager {
public:
    // 何もしない Deleter
    // 確保・解放は自前で行う
    struct NullDeleter
    {
        void operator ()(void* ptr)
        {
            NN_UNUSED(ptr);
        }
    };
public:
    PodManager() NN_NOEXCEPT : m_pAllocator(NULL), uniquePtr()
    {
        NN_STATIC_ASSERT(std::is_pod<T>::value);
    }
    explicit PodManager(T* ptr) NN_NOEXCEPT : m_pAllocator(NULL), uniquePtr()
    {
        NN_STATIC_ASSERT(std::is_pod<T>::value);
        uniquePtr.reset(ptr);
    }
    PodManager(T* ptr, WorkBufAllocator* pAllocator) NN_NOEXCEPT :
        m_pAllocator(pAllocator), uniquePtr()
    {
        NN_STATIC_ASSERT(std::is_pod<T>::value);
        uniquePtr.reset(ptr);
    }
    PodManager(PodManager&& ptrManager) NN_NOEXCEPT : m_pAllocator(), uniquePtr()
    {
        m_pAllocator = ptrManager.m_pAllocator;
        uniquePtr = std::move(ptrManager.uniquePtr);
        ptrManager.m_pAllocator = NULL;
    }
    ~PodManager() NN_NOEXCEPT
    {
        Reset();
    }

    void Swap(PodManager& arrayManager) NN_NOEXCEPT
    {
        std::swap(uniquePtr, arrayManager.uniquePtr);
        WorkBufAllocator* pAllocatorTmp = arrayManager.m_pAllocator;
        arrayManager.m_pAllocator = m_pAllocator;
        m_pAllocator = pAllocatorTmp;
    }

    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        if (uniquePtr)
        {
            return true;
        }
        return false;
    }
    PodManager& operator=(PodManager&& ptrManager) NN_NOEXCEPT
    {
        Free();
        m_pAllocator = ptrManager.m_pAllocator;
        uniquePtr = std::move(ptrManager.uniquePtr);
        ptrManager.m_pAllocator = NULL;
    }
    bool operator!() NN_NOEXCEPT {
        return !uniquePtr;
    }

    T* Get() const NN_NOEXCEPT
    {
        return uniquePtr.get();
    }
    T* Release() NN_NOEXCEPT
    {
        T* ptr = uniquePtr.release();
        Reset();
        return ptr;
    }
    void Reset() NN_NOEXCEPT
    {
        Free();
        uniquePtr.reset();
        m_pAllocator = NULL;
    }
    void Reset(T* ptr) NN_NOEXCEPT
    {
        Reset();
        uniquePtr.reset(ptr);
    }
    void Reset(T* ptr, WorkBufAllocator* pAllocator) NN_NOEXCEPT
    {
        Reset();
        uniquePtr.reset(ptr);
        m_pAllocator = pAllocator;
    }
private:
    void Free() NN_NOEXCEPT
    {
        if (m_pAllocator)
        {
            m_pAllocator->Free(uniquePtr.get());
        }
        else
        {
            delete uniquePtr.get();
        }
    }
private:
    WorkBufAllocator* m_pAllocator;                 // ポインタの確保に使われるアロケータ
    std::unique_ptr<T, NullDeleter> uniquePtr;    // ポインタ管理用 unique_ptr
private:
    // コピーの禁止
    PodManager(const PodManager&) = delete;
    void operator=(const PodManager&) = delete;
};
}}} // nn::ngc::detail
