﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nfc/server/core/nfc_CoreManager.h>
#include "nfp_Ntag.h"
#include "nfp_Util.h"
#include <nn/nfc/server/core/nfc_CoreUtil.h>
#include <nn/nfc/server/util/nfc_ScopedMutexLock.h>

namespace
{
nn::Result CheckReadData(const nn::nfc::server::core::NtagReadParameter& parameter, const nn::nfc::server::core::NtagData& ntagData)
{
    if(parameter.blockCount != ntagData.blockCount)
    {
        return nn::nfc::ResultNeedRetry();
    }

    std::unique_ptr<bool[]> checked(new bool[ntagData.blockCount]);
    for(int32_t i = 0; i < ntagData.blockCount; i++)
    {
        checked[i] = false;
    }

    for(int32_t i = 0; i < parameter.blockCount; i++)
    {
        bool find = false;
        for(int32_t j = 0; j < ntagData.blockCount; j++)
        {
            if(!checked[j]
               && parameter.addresses[i].startPage == ntagData.readDataBlocks[j].address.startPage
               && parameter.addresses[i].endPage == ntagData.readDataBlocks[j].address.endPage)
            {
                checked[j] = true;
                find = true;
                break;
            }
        }
        if(!find)
        {
            return nn::nfc::ResultNeedRetry();
        }
    }

    NN_RESULT_SUCCESS;
}

}

namespace nn { namespace nfp {namespace server {

Ntag::Ntag(const nn::nfc::DeviceHandle& deviceHandle, const nn::nfc::TagId& id, nn::os::SystemEventType* accessFinishEvent, nn::os::SystemEventType* accessResetEvent) NN_NOEXCEPT : Tag(deviceHandle, id, nn::nfc::NfcProtocol_TypeA, nn::nfc::TagType_Type2, accessFinishEvent, accessResetEvent)
{
}

Ntag::~Ntag() NN_NOEXCEPT
{
};


nn::Result Ntag::Read(nn::nfc::server::core::NtagData* ntagData, nn::nfc::server::core::Service* service, const nn::nfc::server::core::NtagReadParameter& ntagReadParameter) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    ClearEvent();
    NN_RESULT_DO(nn::nfc::server::core::Manager::GetInstance().StartNtagRead(service, m_Handle, m_Id, ntagReadParameter));

    std::unique_ptr<nn::nfc::server::core::Info> info(new nn::nfc::server::core::Info());
    NN_RESULT_DO(WaitInfo(info.get(), service, m_Handle, ntagReadParameter.timeoutMsec + 1000));
    switch(info->reason)
    {
    case nn::xcd::NfcEventReason_ReadFinish:
        {
            NN_RESULT_DO(CheckReadData(ntagReadParameter, info->ntagData));
            std::memcpy(ntagData, &info->ntagData, sizeof(nn::nfc::server::core::NtagData));
            m_Type2TagVersion = ntagData->type2TagVersion;
            NN_RESULT_SUCCESS;
        }
    case nn::xcd::NfcEventReason_Error:
        {
            return nn::nfc::server::core::ConvertToNfcResult(info->errorInfo.resultCode);
        }
    default:
        {
            return nn::nfc::ResultNeedRetry();
        }
    }
}

nn::Result Ntag::Write(nn::nfc::server::core::Service* service, const nn::nfc::server::core::NtagWriteParameter& ntagWriteParameter) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    ClearEvent();
    NN_RESULT_DO(nn::nfc::server::core::Manager::GetInstance().StartNtagWrite(service, m_Handle, m_Id, ntagWriteParameter));

    std::unique_ptr<nn::nfc::server::core::Info> info(new nn::nfc::server::core::Info());
    NN_RESULT_DO(WaitInfo(info.get(), service, m_Handle, ntagWriteParameter.timeoutMsec + 1000));
    switch(info->reason)
    {
    case nn::xcd::NfcEventReason_WriteFinish:
        {
            NN_RESULT_SUCCESS;
        }
    case nn::xcd::NfcEventReason_Error:
        {
            return nn::nfc::server::core::ConvertToNfcResult(info->errorInfo.resultCode);
        }
    default:
        {
            return nn::nfc::ResultNeedRetry();
        }
    }
}

nn::Result Ntag::ReadyToWrite(nn::nfc::server::core::Service* service, bool isPasswordRequired) NN_NOEXCEPT
{
    nn::nfc::server::core::NtagReadParameter ntagReadParameter;

    ntagReadParameter.timeoutMsec = 2000;
    ntagReadParameter.isPasswordRequired = isPasswordRequired;
    ntagReadParameter.tagId = m_Id;
    ntagReadParameter.blockCount = 1;

    auto* addresses = ntagReadParameter.addresses;
    addresses[0].startPage = 0x03;
    addresses[0].endPage   = 0x03;

    std::unique_ptr<nn::nfc::server::core::NtagData> ntagData(new nn::nfc::server::core::NtagData);
    NN_RESULT_DO(Read(ntagData.get(), service, ntagReadParameter));
    NN_RESULT_SUCCESS;
}

bool Ntag::IsNtag(nn::nfc::NfcProtocol protocol, nn::nfc::TagType type) NN_NOEXCEPT
{
    return (protocol == nn::nfc::NfcProtocol_TypeA && type == nn::nfc::TagType_Type2);
}

nn::Result Ntag::SendFastReadCommand(void* pOutBuffer, nn::nfc::server::core::Service* service, uint8_t startPage, uint8_t endPage, size_t bufferSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(bufferSize == static_cast<size_t>((endPage - startPage + 1) * 4));

    nn::Bit8 data[3];

    int loop = static_cast<int>(bufferSize / /*nn::nfc::server::core::PassThruDataSizeMax*/128);
    size_t mod = bufferSize % /*nn::nfc::server::core::PassThruDataSizeMax*/128;
    size_t maxPage = /*nn::nfc::server::core::PassThruDataSizeMax*/128 / 4;
    nn::Bit8* buffer = reinterpret_cast<nn::Bit8*>(pOutBuffer);

    if(mod != 0)
    {
        ++loop;
    }

    data[0] = 0x3A;

    for(int i = 0; i < loop; ++i)
    {
        size_t size;
        size_t responseSize;
        if(mod != 0 && i == loop - 1)
        {
            size = mod;
            data[1] = static_cast<nn::Bit8>(startPage + (i * maxPage));
            data[2] = static_cast<nn::Bit8>(endPage);
        }
        else
        {
            size = /*nn::nfc::server::core::PassThruDataSizeMax*/128;
            data[1] = static_cast<nn::Bit8>(startPage + (i * maxPage));
            data[2] = static_cast<nn::Bit8>(data[1] + maxPage - 1);
        }

        NN_RESULT_DO(SendCommand(buffer, &responseSize, service, data, sizeof(data), size, nn::TimeSpan::FromMilliSeconds(2000)));
        if(responseSize != size)
        {
            return nn::nfc::ResultNotSupported();
        }

        buffer += /*nn::nfc::server::core::PassThruDataSizeMax*/128;
    }

    NN_RESULT_SUCCESS;
}


nn::Result Ntag::SendWriteCommand(nn::nfc::server::core::Service* service, uint8_t page, const void* pData, size_t dataSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(dataSize == 4);

    nn::Bit8 data[6];
    data[0] = 0xA2;
    data[1] = static_cast<nn::Bit8>(page);
    std::memcpy(&data[2], pData, 4);

    nn::Bit8 response;
    size_t responseSize;

    NN_RESULT_DO(SendCommand(&response, &responseSize, service, data, sizeof(data), sizeof(response), nn::TimeSpan::FromMilliSeconds(2000)));
    if(responseSize != sizeof(response)
       || response != 0x0a)
    {
        return nn::nfc::ResultNotSupported();
    }

    NN_RESULT_SUCCESS;
}

nn::Result Ntag::SendWriteCommandAndVerify(nn::nfc::server::core::Service* service, uint8_t page, const void* pData, size_t dataSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(dataSize == 4);
    nn::Bit8 response[4];

    NN_RESULT_DO(SendWriteCommand(service, page, pData, dataSize));
    NN_RESULT_DO(SendFastReadCommand(response, service, page, page, sizeof(response)));

    if( std::memcmp(response, pData, sizeof(response)) != 0)
    {
        return nn::nfc::ResultNeedRetry();
    }

    NN_RESULT_SUCCESS;
}

nn::Result Ntag::Read(void* pOutBuffer, nn::nfc::server::core::Service* service, uint16_t  timeoutMsec, bool isPasswordRequired, uint8_t startPage, uint8_t endPage, size_t bufferSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(bufferSize == static_cast<size_t>((endPage - startPage + 1) * 4));

    nn::nfc::server::core::NtagReadParameter ntagReadParameter;
    ntagReadParameter.timeoutMsec = timeoutMsec;
    ntagReadParameter.isPasswordRequired = isPasswordRequired;
    ntagReadParameter.tagId = m_Id;

    int totalPageCount = (endPage - startPage + 1);
    int totalBlockCount = (totalPageCount + (nn::xcd::NtagReadBlockPageCountMax - 1)) / nn::xcd::NtagReadBlockPageCountMax;
    int readCount = (totalBlockCount + (nn::xcd::NtagReadBlockCountMax - 1)) / nn::xcd::NtagReadBlockCountMax;
    int lastBlockCount = totalBlockCount % nn::xcd::NtagReadBlockCountMax;
    int lastPageCount = totalPageCount % nn::xcd::NtagReadBlockPageCountMax;

    nn::Bit8* targetData = reinterpret_cast<nn::Bit8*>(pOutBuffer);
    uint8_t targetPage = startPage;

    auto* addresses = ntagReadParameter.addresses;

    for(int i = 0; i < readCount; ++i)
    {
        if(i == readCount - 1 && lastBlockCount != 0)
        {
            ntagReadParameter.blockCount = lastBlockCount;
        }
        else
        {
            ntagReadParameter.blockCount = nn::xcd::NtagReadBlockCountMax;
        }

        for(int j = 0; j < ntagReadParameter.blockCount; ++j)
        {
            int readPageCount;
            if(i == readCount - 1 && j == ntagReadParameter.blockCount - 1 && lastPageCount != 0)
            {
                readPageCount = lastPageCount;
            }
            else
            {
                readPageCount = nn::xcd::NtagReadBlockPageCountMax;
            }

            addresses[j].startPage = targetPage;
            addresses[j].endPage = static_cast<uint8_t>(addresses[j].startPage + readPageCount - 1);

            targetPage += static_cast<uint8_t>(readPageCount);
        }

        std::unique_ptr<nn::nfc::server::core::NtagData> ntagData(new nn::nfc::server::core::NtagData);
        NN_RESULT_DO(Read(ntagData.get(), service, ntagReadParameter));

        for(int j = 0; j < ntagData->blockCount; ++j)
        {
            const auto& block = ntagData->readDataBlocks[j];
            auto dataSize = (block.address.endPage - block.address.startPage + 1) * nn::xcd::Type2TagPageSize;
            std::memcpy(targetData, block.data, dataSize);
            targetData += dataSize;
        }
    }
    NN_RESULT_SUCCESS;
}


nn::Result Ntag::Write(nn::nfc::server::core::Service* service, uint16_t  timeoutMsec, bool isPasswordRequired, bool isActivationNeeded, const void* clearData, const void* activationData, size_t activationDataSize, uint8_t startPage, uint8_t endPage, const void* pData, size_t dataSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(dataSize == static_cast<size_t>((endPage - startPage + 1) * 4));

    std::unique_ptr<nn::nfc::server::core::NtagWriteParameter> ntagWriteParameter(new nn::nfc::server::core::NtagWriteParameter);

    ntagWriteParameter->timeoutMsec = timeoutMsec;
    ntagWriteParameter->isPasswordRequired = isPasswordRequired;
    ntagWriteParameter->tagId = m_Id;
    ntagWriteParameter->type2TagVersion = m_Type2TagVersion;
    ntagWriteParameter->ntagWriteData.isActivationNeeded = isActivationNeeded;
    if(ntagWriteParameter->ntagWriteData.isActivationNeeded)
    {
        NN_ABORT_UNLESS(activationDataSize == nn::xcd::NtagActivationAreaSize);

        //Activation クリア時設定
        std::memcpy(ntagWriteParameter->ntagWriteData.clearData, clearData, nn::xcd::NtagActivationAreaSize);
        //Activation 時設定
        std::memcpy(&ntagWriteParameter->ntagWriteData.activationData, activationData, nn::xcd::NtagActivationAreaSize);
    }

    int totalPageCount = (endPage - startPage + 1);
    int totalBlockCount = (totalPageCount + (nn::xcd::NtagWriteBlockPageCountMax - 1)) / nn::xcd::NtagWriteBlockPageCountMax;
    int writeCount = (totalBlockCount + (nn::xcd::NtagWriteBlockCountMax - 1)) / nn::xcd::NtagWriteBlockCountMax;
    int lastBlockCount = totalBlockCount % nn::xcd::NtagWriteBlockCountMax;
    int lastPageCount = totalPageCount % nn::xcd::NtagWriteBlockPageCountMax;

    const nn::Bit8* targetData = reinterpret_cast<const nn::Bit8*>(pData);
    uint8_t targetPage = startPage;

    for(int i = 0; i < writeCount; ++i)
    {
        auto* dataBlocks = ntagWriteParameter->ntagWriteData.dataBlocks;

        if(i == writeCount - 1 && lastBlockCount != 0)
        {
            ntagWriteParameter->ntagWriteData.blockCount = lastBlockCount;
        }
        else
        {
            ntagWriteParameter->ntagWriteData.blockCount = nn::xcd::NtagWriteBlockCountMax;
        }

        for(int j = 0; j < ntagWriteParameter->ntagWriteData.blockCount; ++j)
        {
            int writePageCount;

            if(i == writeCount - 1 && j == ntagWriteParameter->ntagWriteData.blockCount - 1 && lastPageCount != 0)
            {
                writePageCount = lastPageCount;
            }
            else
            {
                writePageCount = nn::xcd::NtagWriteBlockPageCountMax;
            }

            dataBlocks[j].startPageAddress = targetPage;
            dataBlocks[j].dataSize = static_cast<uint8_t>(nn::xcd::Type2TagPageSize * writePageCount);
            std::memcpy(dataBlocks[j].data, targetData, dataBlocks[j].dataSize);

            targetPage += static_cast<uint8_t>(writePageCount);
            targetData += dataBlocks[j].dataSize;
        }

        NN_RESULT_DO(Write(service, *ntagWriteParameter));
    }

    NN_RESULT_SUCCESS;
}

}}}  // namespace nn::nfp::server
