﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_SdkLog.h>
#include <nn/crypto.h>
#include <nn/result/result_HandlingUtility.h>
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/spl/spl_Api.h>
#endif
#include "nfp_Crypto.h"
#include "nfp_Util.h"

namespace nn {
namespace nfp {
namespace server {


namespace {
    const size_t NoftSize = 480;// 456 以上で 32 の倍数の最小の値
    const size_t RandomSize = 32;
    const size_t BlockSize = 64;

    struct NfcRamKeySetForPrf
    {
        char label[14];
        nn::Bit8 fixedValue[14];
        nn::Bit8 secret[16];
    };

    struct NfcRomKeySetForPrf
    {
        char label[14];
        nn::Bit8 fixedValue[16];
        nn::Bit8 secret[16];
    };

    struct NfcKeySetForRnd
    {
        nn::Bit8 key[16];
        nn::Bit8 iv[16];
    };

    struct NfcKeys
    {
        NfcRamKeySetForPrf ramKeySet;
        NfcRomKeySetForPrf romKeySet;
        NfcKeySetForRnd rndKeySet;
        nn::Bit8 padding[6];
    };

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    const nn::Bit8 KeyForSpl[] = {
        //暗号化済み鍵
        0x83, 0xF6, 0xEF, 0xD8, 0x13, 0x26, 0x49, 0xAB, 0x97, 0x5F, 0xEA, 0xBA, 0x65, 0x71, 0xCA, 0xCA
    };

    const nn::Bit8 Iv[16] = {
        0xB9, 0x1D, 0xC1, 0xCF, 0x33, 0x5F, 0xA6, 0x13, 0x2A, 0xEF, 0x90, 0x99, 0xAA, 0xCA, 0x93, 0xC8
    };

    const nn::Bit8 EncryptedNfcKeysForDev[] = {
        //開発用暗号化済み鍵
        0x13, 0xB0, 0xFB, 0xC2, 0x91, 0x6D, 0x6E, 0x5A, 0x10, 0x31, 0x40, 0xB7, 0xDF, 0xCF, 0x69, 0x69,
        0xB0, 0xFA, 0xAE, 0x7F, 0xB2, 0x4D, 0x27, 0xC9, 0xE9, 0x3F, 0x5B, 0x38, 0x39, 0x24, 0x98, 0xCE,
        0xED, 0xD2, 0xA9, 0x6C, 0x6F, 0xA7, 0x72, 0xD7, 0x11, 0x31, 0x17, 0x93, 0x12, 0x49, 0x32, 0x85,
        0x21, 0xE5, 0xE1, 0x88, 0x0F, 0x08, 0xF2, 0x30, 0x5C, 0xC3, 0xAA, 0xFF, 0xC0, 0xAB, 0x21, 0x96,
        0x74, 0x39, 0xED, 0xE0, 0x5A, 0xB6, 0x75, 0xC2, 0x3B, 0x08, 0x61, 0xE4, 0xA7, 0xD6, 0xED, 0x8C,
        0xA9, 0x02, 0x12, 0xA6, 0xCC, 0x27, 0x4C, 0x1C, 0x41, 0x9C, 0xD8, 0x4C, 0x00, 0xC7, 0x5B, 0x5D,
        0xED, 0xC2, 0x3D, 0x5E, 0x00, 0xF5, 0x49, 0xFA, 0x6C, 0x75, 0x67, 0xCF, 0x1F, 0x73, 0x1A, 0xE8,
        0x47, 0xD4, 0x3D, 0x9B, 0x83, 0x5B, 0x18, 0x2F, 0x95, 0xA9, 0x04, 0xBC, 0x2E, 0xBB, 0x64, 0x4A
    };

    const nn::Bit8 EncryptedNfcKeysForProd[] = {
        //製品用暗号化済み鍵
        0x76, 0x50, 0x87, 0x02, 0x40, 0xA6, 0x5A, 0x98, 0xCE, 0x39, 0x2F, 0xC8, 0x83, 0xAF, 0x54, 0x76,
        0x28, 0xFF, 0x50, 0xFC, 0xC1, 0xFB, 0x26, 0x14, 0xA2, 0x4A, 0xA6, 0x74, 0x90, 0xA4, 0x37, 0x06,
        0x03, 0x63, 0xC2, 0xB1, 0xAF, 0x9F, 0xF7, 0x07, 0xFC, 0x8A, 0xB9, 0xCA, 0x28, 0x68, 0x6E, 0xF7,
        0x42, 0xCD, 0x68, 0x13, 0xCD, 0x7B, 0x3A, 0x60, 0x3E, 0x8B, 0xAB, 0x3A, 0xCC, 0xED, 0xE0, 0xDD,
        0x71, 0x1F, 0xA5, 0xDE, 0xB8, 0xB1, 0xF5, 0x1D, 0x14, 0x73, 0xBE, 0x27, 0xCC, 0xA1, 0x9B, 0x23,
        0x06, 0x91, 0x89, 0x05, 0xED, 0xD6, 0x92, 0x76, 0x3F, 0x42, 0xFB, 0xD1, 0x8F, 0x2D, 0x6D, 0x72,
        0xC8, 0x9E, 0x48, 0xE8, 0x03, 0x64, 0xF0, 0x3C, 0x0E, 0x2A, 0xF1, 0x26, 0x83, 0x02, 0x4F, 0xE2,
        0x41, 0xAA, 0xC8, 0x33, 0x68, 0x84, 0x3A, 0xFB, 0x87, 0x18, 0xEA, 0xF7, 0x36, 0xA2, 0x4E, 0xA9
    };
#else
    const nn::Bit8 NfcKeysForDev[] = {
        //開発用生鍵
        0x75, 0x6e, 0x66, 0x69, 0x78, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x66, 0x6f, 0x73, 0x00, 0x52, 0xc5,
        0xdf, 0x99, 0xd7, 0x12, 0x2b, 0x0d, 0x9e, 0x88, 0xd7, 0xc6, 0x66, 0xa9, 0xed, 0x29, 0xf8, 0x9a,
        0x78, 0xe0, 0x75, 0x2a, 0xae, 0x05, 0x01, 0x23, 0xa3, 0xfd, 0x8a, 0xc0, 0x6c, 0x6f, 0x63, 0x6b,
        0x65, 0x64, 0x20, 0x73, 0x65, 0x63, 0x72, 0x65, 0x74, 0x00, 0x4c, 0xdb, 0xba, 0x43, 0xe6, 0xa3,
        0x71, 0x35, 0x3c, 0x3d, 0x0c, 0xe6, 0xf2, 0xcf, 0xd2, 0x9b, 0x83, 0x0e, 0x75, 0xfd, 0x6d, 0x88,
        0xd8, 0x9b, 0x0d, 0x31, 0x9c, 0x87, 0x0b, 0x88, 0xb2, 0x40, 0x84, 0xe7, 0x17, 0xe2, 0x92, 0x58,
        0x69, 0xd7, 0xcf, 0x83, 0x5c, 0x9a, 0x5c, 0x30, 0xbe, 0x67, 0x16, 0x6d, 0x1a, 0xb3, 0xa2, 0x95,
        0x80, 0x92, 0xee, 0x6f, 0xac, 0x9a, 0xcb, 0x4c, 0x2e, 0x51, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
    };
#endif
}

void Crypto::CalculateSha256(void* pOutHash, size_t hashSize, const void* pData, size_t dataSize) NN_NOEXCEPT
{
    nn::crypto::Sha256Generator generator;
    generator.Initialize();
    generator.Update(pData, dataSize);
    generator.GetHash(pOutHash, hashSize);
}

void Crypto::CalculateHmacSha256(void* pOutHmac, size_t hmacSize, const void* pData, size_t dataSize, const void* pKey, size_t keySize) NN_NOEXCEPT
{

    // 鍵がブロック長よりも長い場合、ハッシュ値を鍵とする.
    nn::Bit8 key[BlockSize];
    if( keySize >= BlockSize )
    {
        std::memcpy(key, pKey, BlockSize);
    }
    else
    {
        std::memcpy(key, pKey, keySize);
        // 鍵データの残りを 0 でパディング
        std::memset(key + keySize, 0, BlockSize - keySize);
    }

    // 鍵とipadのXOR
    nn::Bit8 padKey[BlockSize];
    for (size_t i = 0 ; i < BlockSize ; ++i )
    {
        padKey[i] = key[i] ^ 0x36;
    }

    // データとipadの結合
    std::unique_ptr<nn::Bit8[]> buffer(new nn::Bit8[NoftSize + BlockSize]);
    nn::Bit8* pBuffer = buffer.get();
    std::memcpy(pBuffer, padKey, BlockSize);
    std::memcpy(pBuffer + BlockSize, pData, dataSize);

    // ハッシュ値計算
    nn::Bit8 hash[HashSize];
    CalculateSha256(hash, sizeof(hash), pBuffer, dataSize + BlockSize);
    // 鍵とopadのXOR
    for (size_t i = 0 ; i < BlockSize ; ++i )
    {
        padKey[i] = key[i] ^ 0x5c;
    }
    //ハッシュ値とopadの結合
    std::memset(pBuffer,0,sizeof(buffer));
    std::memcpy(pBuffer, padKey, BlockSize);
    std::memcpy(pBuffer + BlockSize, hash, sizeof(hash));

    // ハッシュ値計算
    std::memset(hash,0,sizeof(hash));
    CalculateSha256(hash, sizeof(hash), pBuffer, BlockSize + sizeof(hash));
    std::memcpy(pOutHmac, hash, hmacSize);
}

void Crypto::CalculateHmacSha256Vector(void* pOutHmac, size_t hmacSize,
                                       const void* pKey, size_t keySize,
                                       size_t elementNum, const void* element[], const size_t elementLen[]) NN_NOEXCEPT
{
    size_t allElementLen = 0;
    for (size_t i = 0; i < elementNum; i++)
    {
        allElementLen += elementLen[i];
    }

    std::unique_ptr<nn::Bit8[]> allElement(new nn::Bit8[allElementLen]);
    size_t len = 0;
    for (size_t i = 0; i < elementNum; i++)
    {
        std::memcpy(allElement.get() + len, element[i], elementLen[i]);
        len += elementLen[i];
    }
    CalculateHmacSha256(pOutHmac, hmacSize, allElement.get(), allElementLen, pKey, keySize);
}

void Crypto::CalculateSha256Prf(Keys* pOutKeys, const void* pKey, size_t keySize, const char* pLabel, const void* pSeed, size_t seedSize) NN_NOEXCEPT
{
    const void* element[3];
    size_t elementLen[3];
    nn::Bit8 counterArray[2];
    nn::Bit8* keys = reinterpret_cast<nn::Bit8*>(pOutKeys);

    element[0] = counterArray;
    elementLen[0] = sizeof(counterArray);
    element[1] = pLabel;
    elementLen[1] = std::strlen(pLabel) + 1;
    element[2] = pSeed;
    elementLen[2] = seedSize;

    size_t pos = 0;
    uint16_t counter = 0;
    while (pos < sizeof(Keys))
    {
        size_t rest = sizeof(Keys) - pos;
        counterArray[0] = static_cast<nn::Bit8>((counter >> 8) & 0xFF);
        counterArray[1] = static_cast<nn::Bit8>(counter & 0xFF);
        if (rest >= HashSize)
        {
            CalculateHmacSha256Vector(&keys[pos], HmacSize, pKey, keySize, 3, element, elementLen);
            pos += HashSize;
        }
        else
        {
            nn::Bit8 hash[HashSize];
            CalculateHmacSha256Vector(hash, sizeof(hash), pKey, keySize, 3, element, elementLen);
            std::memcpy(&keys[pos], hash, rest);
            break;
        }
        ++counter;
    }
}

void Crypto::SignHmacSha256(void* pOutHmac, size_t hmacSize, const void* pData, size_t dataSize, void* pKey, size_t keySize) NN_NOEXCEPT
{
    CalculateHmacSha256(pOutHmac, hmacSize, pData, dataSize, pKey, keySize);
}

void Crypto::CalculateKey(Keys* pOutKeys, const void* pSeed, size_t seedSize, bool isWrite) NN_NOEXCEPT
{
    std::unique_ptr<nn::Bit8> seed(new nn::Bit8[seedSize]);
    std::memcpy(seed.get(), pSeed, seedSize);

    std::unique_ptr<NfcKeys> nfcKeys(new NfcKeys);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    {
        nn::spl::InitializeForCrypto();
        const nn::Bit8* encryptedNfcKeys = nn::spl::IsDevelopment() ? EncryptedNfcKeysForDev : EncryptedNfcKeysForProd;

        //暗号化された鍵を復号化
        nn::Bit8 decryptedKeyForSpl[16];
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::DecryptAesKey(decryptedKeyForSpl, sizeof(decryptedKeyForSpl), KeyForSpl, sizeof(KeyForSpl), 0, 0));

        //復号化された鍵を使って NfcKey を復号化
        nn::crypto::Aes128CtrDecryptor aes;
        aes.Initialize(decryptedKeyForSpl, sizeof(decryptedKeyForSpl), Iv, sizeof(Iv));
        aes.Update(nfcKeys.get(), sizeof(NfcKeys), encryptedNfcKeys, sizeof(NfcKeys));
        std::memset(decryptedKeyForSpl, 0, sizeof(decryptedKeyForSpl));

        nn::spl::Finalize();
    }
#else
    std::memcpy(nfcKeys.get(), NfcKeysForDev, sizeof(NfcKeysForDev));
#endif

    const size_t randomStartAddr = 32;

    nn::crypto::Aes128CtrEncryptor aes;
    aes.Initialize(nfcKeys->rndKeySet.key, sizeof(nfcKeys->rndKeySet.key), nfcKeys->rndKeySet.iv, sizeof(nfcKeys->rndKeySet.iv));
    aes.Update(seed.get() + randomStartAddr, RandomSize, reinterpret_cast<const nn::Bit8*>(pSeed) + randomStartAddr, RandomSize);
    if(isWrite)
    {
        std::memcpy(seed.get() + 2, nfcKeys->ramKeySet.fixedValue, sizeof(nfcKeys->ramKeySet.fixedValue));
        CalculateSha256Prf(pOutKeys,
                           nfcKeys->ramKeySet.secret, sizeof(nfcKeys->ramKeySet.secret),
                           nfcKeys->ramKeySet.label,
                           seed.get(), seedSize);
    }
    else
    {
        std::memcpy(seed.get() + 0, nfcKeys->romKeySet.fixedValue, sizeof(nfcKeys->romKeySet.fixedValue));
        CalculateSha256Prf(pOutKeys,
                           nfcKeys->romKeySet.secret, sizeof(nfcKeys->romKeySet.fixedValue),
                           nfcKeys->romKeySet.label,
                           seed.get(), seedSize);
    }
}

void Crypto::EncryptAndDecryptAesCtr(void* pOutBuffer, size_t bufferSize, const void* pData, size_t dataSize, void* pKey, size_t keySize, void* pIv, size_t ivSize) NN_NOEXCEPT
{
    nn::crypto::Aes128CtrEncryptor aes;
    aes.Initialize(pKey, keySize, pIv, ivSize);
    aes.Update(pOutBuffer, bufferSize, pData, dataSize);
}

void Crypto::DecryptNtf(void* pOutBuffer, size_t bufferSize, const void* pData, size_t dataSize) NN_NOEXCEPT
{
    nn::Bit8 key[16] = {0x1e,0x95,0x49,0x31,0xbb,0x88,0x77,0x8e,0xac,0x60,0xb4,0x23,0x61,0x0d,0x3d,0x28};
    nn::Bit8 iv[16] = {0x5a,0x0e,0x84,0xfa,0xaa,0x97,0xc6,0x80,0x59,0xed,0x82,0x30,0x50,0xf8,0x9c,0x24};

    nn::crypto::Aes128CbcDecryptor aes;
    aes.Initialize(key, sizeof(key), iv, sizeof(iv));
    aes.Update(pOutBuffer, bufferSize, pData, dataSize);
}

} // end of namespace server
} // end of namespace nfp
} // end of namespace nn
