﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include "nfc_SimpleMessageQueue.h"
#include <nn/nfc/server/util/nfc_ScopedMutexLock.h>

namespace nn { namespace nfc { namespace server { namespace core {

namespace
{
}

SimpleMessageQueue::SimpleMessageQueue(size_t queueCapacity) NN_NOEXCEPT : m_QueueCapacity(queueCapacity)
{
    nn::os::InitializeMutex(&m_Mutex, true, 0);

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    m_QueueBuffer.reset(new uintptr_t[m_QueueCapacity]);
    m_IsEmpty.reset(new bool[m_QueueCapacity]);
    for(size_t i = 0; i < m_QueueCapacity; ++i)
    {
        m_IsEmpty[i] = true;
    }
    m_Data.reset(new nn::Bit8[m_QueueCapacity]);
    m_Queue.reset(new nn::os::MessageQueue(m_QueueBuffer.get(), m_QueueCapacity));
}

SimpleMessageQueue::~SimpleMessageQueue() NN_NOEXCEPT
{
    m_Queue.reset(nullptr);
    m_Data.reset(nullptr);
    m_IsEmpty.reset(nullptr);
    m_QueueBuffer.reset(nullptr);

    nn::os::FinalizeMutex(&m_Mutex);
}

void SimpleMessageQueue::Send(nn::Bit8 data) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    for(size_t i = 0; i < m_QueueCapacity; ++i)
    {
        if(m_IsEmpty[i])
        {
            m_IsEmpty[i] = false;
            m_Data[i] = data;
            lock.Unlock();
            m_Queue->Send(reinterpret_cast<uintptr_t>(&m_Data[i]));
            return;
        }
    }

    NN_SDK_LOG("[NFC] message queue over\n");
}

void SimpleMessageQueue::Receive(nn::Bit8* pOutData) NN_NOEXCEPT
{
    uintptr_t outData;
    m_Queue->Receive(&outData);
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    Empty(outData);
    *pOutData = *(reinterpret_cast<nn::Bit8*>(outData));
}

bool SimpleMessageQueue::TimedReceive(nn::Bit8* pOutData, nn::TimeSpan timeout) NN_NOEXCEPT
{
    uintptr_t outData;
    if(m_Queue->TimedReceive(&outData, timeout))
    {
        nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
        Empty(outData);
        *pOutData = *(reinterpret_cast<nn::Bit8*>(outData));
        return true;
    }

    return false;
}

void SimpleMessageQueue::Peek(nn::Bit8* pOutData) NN_NOEXCEPT
{
    uintptr_t outData;
    m_Queue->Peek(&outData);
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    *pOutData = *(reinterpret_cast<nn::Bit8*>(outData));
}

void SimpleMessageQueue::Clear() NN_NOEXCEPT
{
    bool IsReceived;
    do
    {
        uintptr_t outData;
        IsReceived = m_Queue->TryReceive(&outData);
        if(IsReceived)
        {
            nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
            Empty(outData);
        }
    }while(IsReceived);
}

void SimpleMessageQueue::Empty(uintptr_t data)
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    for(size_t i = 0; i < m_QueueCapacity; ++i)
    {
        if(!m_IsEmpty[i] && data == reinterpret_cast<uintptr_t>(&m_Data[i]))
        {
            m_IsEmpty[i] = true;
            break;
        }
    }
}

nn::os::MessageQueueType* SimpleMessageQueue::GetBase() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    return m_Queue->GetBase();
}

}}}}  // namespace nn::nfc::server::core
