﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include "nfc_FileStream.h"

namespace nn {
namespace nfc {
namespace server {
namespace core {

FileStream::FileStream() NN_NOEXCEPT : m_IsInitialized(false), m_Pos(0)
{
}

FileStream::~FileStream() NN_NOEXCEPT
{
    if(m_IsInitialized)
    {
        nn::fs::CloseFile(m_FileHandle);
    }
}

nn::Result FileStream::Initialize(const char* path, int mode) NN_NOEXCEPT
{
    NN_RESULT_DO(nn::fs::OpenFile(&m_FileHandle, path, mode));
    m_Pos = 0;
    m_IsInitialized = true;
    NN_RESULT_SUCCESS;
}

void FileStream::Finalize() NN_NOEXCEPT
{
    if(m_IsInitialized)
    {
        nn::fs::CloseFile(m_FileHandle);
        m_IsInitialized = false;
    }
}

nn::Result FileStream::SetPosition(int64_t position) NN_NOEXCEPT
{
    m_Pos = position;
    NN_RESULT_SUCCESS;
}

nn::Result FileStream::Read(size_t* pOutSize, void* pOutBuffer, size_t bufferSize) NN_NOEXCEPT
{
    int64_t fileSize;
    NN_RESULT_DO(GetFileSize(&fileSize, m_FileHandle));

    size_t remain = static_cast<size_t>(fileSize - m_Pos);
    size_t readSize;

    if(remain < bufferSize)
    {
        readSize = remain;
    }
    else
    {
        readSize = bufferSize;
    }

    NN_RESULT_DO(nn::fs::ReadFile(pOutSize, m_FileHandle, m_Pos, pOutBuffer, readSize));
    m_Pos += *pOutSize;
    NN_RESULT_SUCCESS;
}

nn::Result FileStream::Write(const void* pData, size_t dataSize, bool flush) NN_NOEXCEPT
{
    nn::fs::WriteOption writeOption = nn::fs::WriteOption::MakeValue(flush?nn::fs::WriteOptionFlag_Flush:0);
    NN_RESULT_DO(nn::fs::WriteFile(m_FileHandle, m_Pos, pData, dataSize, writeOption));
    m_Pos += dataSize;
    NN_RESULT_SUCCESS;
}

nn::Result FileStream::GetSize(int64_t* pOutSize) NN_NOEXCEPT
{
    return nn::fs::GetFileSize(pOutSize, m_FileHandle);
}

nn::Result FileStream::Flush() NN_NOEXCEPT
{
    return nn::fs::FlushFile(m_FileHandle);
}

} // end of namespace core
} // end of namespace server
} // end of namespace nfc
} // end of namespace nn
