﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/nfc/server/core/nfc_Result.h>
#include "nfc_Ntag.h"
#include "nfc_Mifare.h"
#include "nfc_SimpleMessageQueue.h"

namespace nn { namespace nfc { namespace server { namespace core {

    class Device
    {
        NN_DISALLOW_COPY(Device);
        NN_DISALLOW_MOVE(Device);

    public:
        enum State
        {
            State_None,
            State_Init,
            State_Search,
            State_Active,
            State_Deactive,
            State_Unexpected,
        };

        enum EventThreadState
        {
            EventThreadState_NotInitialized,       //未 Initialize
            EventThreadState_NotStartedDiscovery,  //未 StartDiscovery
            EventThreadState_NormalStartDiscovery, //タグを発見するまで
            EventThreadState_IntermittentLow,      //タグ発見後の間欠動作Low中
            EventThreadState_IntermittentHigh,     //タグ発見後の間欠動作High中
            EventThreadState_ReadyAccessing,       //タグアクセスするためのタグ検出中
            EventThreadState_Accessing,            //タグアクセス中
        };

        enum EventType : nn::Bit8
        {
            EventType_Terminate,                //イベントスレッド終了用イベント
            EventType_AccessStart,              //タグアクセス開始要求
            EventType_AccessTerminate,          //タグアクセス終了要求
            EventType_NormalDiscoveryStart,     //通常 Discover 開始要求
            EventType_NormalDiscoveryStop,      //通常 Discover 終了要求
            EventType_Reset,                    //イベントスレッドリセットイベント
            EventType_ManageDriverEvent,        //Manager に渡すドライバ用イベントの操作要求
        };

        enum AccessEventType : nn::Bit8
        {
            AccessEventType_Cancel,                   //アクセス対象タグ発見キャンセルイベント
            AccessEventType_Deteceted,                //アクセス対象タグ発見イベント
            AccessEventType_Deactivated,              //アクセス対象タグ喪失イベント
        };

        // ドライバ用イベントの状態
        enum DriverEventState
        {
            DriverEventState_NotInitialized,          //未初期化
            DriverEventState_Initialized,             //初期化済み、Manager には未登録
            DriverEventState_Linked                   //Manager に登録済み
        };

        // ドライバ用イベントに対する処理要求の種類
        enum DriverEventRequestType
        {
            DriverEventRequestType_None,              //要求が出ていない
            DriverEventRequestType_Link,              //Manager への Link 要求
            DriverEventRequestType_Unlink             //Manager からの Unlink 要求
        };

    public:
        Device(uint64_t uniqueId, nn::hid::NpadIdType npadId, nn::os::MultiWaitType* eventMultiWait) NN_NOEXCEPT;
        ~Device() NN_NOEXCEPT;
        void Initialize() NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;
        void Reset(ResetReason reason, Service* service) NN_NOEXCEPT;
        nn::Result StartDiscovery(Service* service, const DiscoveryParameter& discoveryParameter, const McuVersionData& mcuVersionData) NN_NOEXCEPT;
        nn::Result StopDiscovery() NN_NOEXCEPT;
        nn::Result KeepSession() NN_NOEXCEPT;
        nn::Result ReleaseSession() NN_NOEXCEPT;
        void SetEvent(nn::os::SystemEventType* pEvent, nn::os::SystemEventType* pDetectEvent, nn::os::SystemEventType* pResetEvent) NN_NOEXCEPT;
        Info& GetInfo() NN_NOEXCEPT;
        Info& GetDetectInfo() NN_NOEXCEPT;
        ResetInfo& GetResetInfo() NN_NOEXCEPT;
        uint64_t GetUniqueId() NN_NOEXCEPT;
        nn::nfc::DeviceHandle GetHandle() NN_NOEXCEPT;
        nn::Result StartNtagRead(const nn::nfc::TagId& tagId, const NtagReadParameter& ntagReadParameter) NN_NOEXCEPT;
        nn::Result StartNtagWrite(const nn::nfc::TagId& tagId, const NtagWriteParameter& ntagWriteParameter) NN_NOEXCEPT;
        nn::Result SendRawData(const nn::nfc::TagId& tagId, const PassThruParameter& passThruParameter) NN_NOEXCEPT;
        nn::Result GetTagInfo(nn::nfc::TagInfo* tagInfo, const nn::nfc::TagId& tagId) NN_NOEXCEPT;
        nn::Result GetNpadId(nn::hid::NpadIdType* pOutNpadId) NN_NOEXCEPT;
        nn::Result GetXcdDeviceHandle(nn::xcd::DeviceHandle* pOutXcdDeviceHandle) NN_NOEXCEPT;
        bool IsNtag(const nn::nfc::TagId& tagId) NN_NOEXCEPT;
        bool IsMifare(const nn::nfc::TagId& tagId) NN_NOEXCEPT;
        void SetState(Device::State state) NN_NOEXCEPT;
        bool IsInitialized() NN_NOEXCEPT;
        nn::Result StartMifareRead(const nn::nfc::TagId& tagId, const MifareReadParameter& mifareReadParameter) NN_NOEXCEPT;
        nn::Result StartMifareWrite(const nn::nfc::TagId& tagId, const MifareWriteParameter& mifareWriteParameter) NN_NOEXCEPT;
        void LoopEventThreadFunction(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;

    private:
        Device() NN_NOEXCEPT;
        Tag* GetTag(const nn::nfc::TagId& tagId) NN_NOEXCEPT;
        Ntag* GetNtag(const nn::nfc::TagId& tagId) NN_NOEXCEPT;
        nn::Result CheckFunctionError() NN_NOEXCEPT;
        nn::Result CheckActive() NN_NOEXCEPT;
        Mifare* GetMifare(const nn::nfc::TagId& tagId) NN_NOEXCEPT;
        void SetEventThreadState(Device::EventThreadState state) NN_NOEXCEPT;
        nn::Result StartDiscoveryInEventThread(uint16_t activationTimeout) NN_NOEXCEPT;
        nn::Result StopDiscoveryInEventThread() NN_NOEXCEPT;
        void DeactivateTag() NN_NOEXCEPT;
        void ProcessIntermittentLow() NN_NOEXCEPT;
        void ProcessIntermittentHigh() NN_NOEXCEPT;
        void PrintEvent(nn::os::MultiWaitHolderType* pHolder, Device::EventType eventType, const Info& info, bool unexpected) NN_NOEXCEPT;
        void DispatchEventInNotInitialized(nn::os::MultiWaitHolderType* pHolder, Device::EventType eventType, const Info& info) NN_NOEXCEPT;
        void DispatchEventInNotStartedDiscovery(nn::os::MultiWaitHolderType* pHolder, Device::EventType eventType, const Info& info) NN_NOEXCEPT;
        void DispatchEventInNormalStartDiscovery(nn::os::MultiWaitHolderType* pHolder, Device::EventType eventType, const Info& info) NN_NOEXCEPT;
        void DispatchEventInIntermittentLow(nn::os::MultiWaitHolderType* pHolder, Device::EventType eventType, const Info& info) NN_NOEXCEPT;
        void DispatchEventInIntermittentHigh(nn::os::MultiWaitHolderType* pHolder, Device::EventType eventType, const Info& info) NN_NOEXCEPT;
        void DispatchEventInReadyAccessing(nn::os::MultiWaitHolderType* pHolder, Device::EventType eventType, const Info& info) NN_NOEXCEPT;
        void DispatchEventInAccessing(nn::os::MultiWaitHolderType* pHolder, Device::EventType eventType, const Info& info) NN_NOEXCEPT;
        bool PreProcessEvent(Device::EventType* pOutEventType, Info* pOutInfo, nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;
        void DispatchEventInEventThread(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;
        nn::Result StartNtagReadCommon(const nn::nfc::TagId& tagId, const NtagReadParameter& ntagReadParameter) NN_NOEXCEPT;
        nn::Result StartNtagWriteCommon(const nn::nfc::TagId& tagId, const NtagWriteParameter& ntagWriteParameter) NN_NOEXCEPT;
        nn::Result SendRawDataCommon(const nn::nfc::TagId& tagId, const PassThruParameter& passThruParameter) NN_NOEXCEPT;
        nn::Result InitializeNpad() NN_NOEXCEPT;
        void FinalizeNpad() NN_NOEXCEPT;
        nn::Result InitializeDriver(const McuVersionData& mcuVersionData) NN_NOEXCEPT;
        void FinalizeDriver() NN_NOEXCEPT;
        bool IsDriverEventInitialized() const NN_NOEXCEPT;
        void RequestLinkDriverEvent() NN_NOEXCEPT;
        void RequestUnlinkDriverEvent() NN_NOEXCEPT;
        void ProcessDriverEventRequest() NN_NOEXCEPT;
        nn::Result CheckXcdDeviceHandle() NN_NOEXCEPT;
        void SignalEvent() NN_NOEXCEPT;
        void SignalDetectEvent() NN_NOEXCEPT;
        void SignalResetEvent() NN_NOEXCEPT;
        nn::Result StartMifareReadCommon(const nn::nfc::TagId& tagId, const MifareReadParameter& mifareReadParameter) NN_NOEXCEPT;
        nn::Result StartMifareWriteCommon(const nn::nfc::TagId& tagId, const MifareWriteParameter& mifareWriteParameter) NN_NOEXCEPT;

    private:
        Device::State m_State;
        uint64_t m_UniqueId;
        nn::nfc::DeviceHandle m_Handle;
        nn::xcd::DeviceHandle m_XcdDeviceHandle;

        //イベント多重待ち
        nn::os::MultiWaitType* m_pEventMultiWait;

        //イベントメッセージキュー
        SimpleMessageQueue m_EventMessageQueue;
        nn::os::MultiWaitHolderType m_EventMessageQueueHolder;

        //ドライバからのタグ検出・喪失イベント
        nn::os::SystemEventType m_DetectEvent;
        nn::os::MultiWaitHolderType m_DetectEventHolder;
        nn::os::SystemEventType* m_pDetectEventForNfcUser;

        //タグ検出・喪失以外のドライバからのイベント
        nn::os::SystemEventType m_Event;
        nn::os::MultiWaitHolderType m_EventHolder;
        nn::os::SystemEventType* m_pEventForNfcUser;

        //間欠動作 Low 期間終了イベント
        nn::os::TimerEventType m_IntermittentLowTerminateEvent;
        nn::os::MultiWaitHolderType m_IntermittentLowTerminateHolder;

        //アクセス対象イベントメッセージキュー
        SimpleMessageQueue m_AccessEventMessageQueue;

        //通常 Discover 開始・終了受領イベント
        nn::os::SystemEventType m_NormalDiscoveryEvent;
        nn::Result m_NormalDiscoveryResult;

        //Activate Nfc イベント
        nn::os::SystemEventType m_NfcActivateEvent;
        nn::os::SystemEventType m_NfcActivateCancelEvent;

        std::unique_ptr<Tag> m_DetectedTag;
        mutable nn::os::MutexType m_Mutex;
        mutable nn::os::SdkRecursiveMutexType m_DriverEventMutex;
        Info m_Info;
        Info m_DetectInfo;
        ResetInfo m_ResetInfo;
        bool m_IsRunningThread;
        EventThreadState m_EventThreadState;
        DriverEventState m_DriverEventState;
        DriverEventRequestType m_DriverEventRequest;
        DiscoveryParameter m_DiscoveryParameter;
        McuVersionData m_McuVersionData;
        bool m_IsEnableKeepSession;
        bool m_IsFunctionError;
        int m_InitializedCount;
        bool m_IsInitializedDriver;
        nn::os::SystemEventType* m_pResetEventForNfcUser;
        Service* m_ActiveService;
    };

}}}}  // namespace nn::nfc::server::core
