﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nfc/server/core/nfc_CoreManager.h>
#include <nn/nfc/pt/server/nfc_PtDevice.h>
#include <nn/nfc/server/nfc_Tag.h>
#include <nn/nfc/server/util/nfc_UtilUtil.h>
#include <nn/nfc/server/util/nfc_ScopedMutexLock.h>
#include <nn/nfc/server/core/nfc_ScopedSession.h>
#include <nn/nfc/server/nfc_Device.h>

namespace nn { namespace nfc { namespace pt { namespace server {

namespace
{
nn::Result ConvertToNfcPtResultForSendCommand(nn::Result nfcResult) NN_NOEXCEPT
{
    NN_RESULT_TRY(nfcResult)
        NN_RESULT_CATCH(nn::nfc::ResultTimeOutError)
        {
            NN_RESULT_THROW(nn::nfc::ResultAccessTimeOutError());
        }
        NN_RESULT_CATCH(nn::nfc::ResultOperationFailed)
        {
            NN_RESULT_THROW(nn::nfc::ResultAccessOperationFailed());
        }
        NN_RESULT_CATCH(nn::nfc::ResultNeedRetry)
        {
            NN_RESULT_THROW(nn::nfc::ResultAccessError());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}
}

Device::Device(nn::nfc::server::Device* device) NN_NOEXCEPT
: m_Device(device)
{
}

Device::~Device() NN_NOEXCEPT
{
}

nn::Result Device::SendCommand(void* pOutBuffer, size_t* pOutSize, nn::nfc::server::core::Service* service, const void* pData, size_t dataSize, size_t bufferSize, nn::TimeSpan timeout) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    if(m_Device->m_State == nn::nfc::server::Device::State_Keep)
    {
        nn::Result result = ConvertToNfcPtResultForSendCommand(m_KeptTag->SendCommand(pOutBuffer, pOutSize, service, pData, dataSize, bufferSize, timeout));
        if(result.IsFailure())
        {
            ReleaseSession();
            nn::nfc::server::core::ScopedSession session(service, m_Device->m_Handle);
            return m_Device->AfterAccessTag(service, &session, result);
        }
        NN_RESULT_SUCCESS;
    }
    else if(m_Device->m_State == nn::nfc::server::Device::State_Active)
    {
        nn::nfc::TagInfo tagInfo;
        NN_RESULT_DO(m_Device->GetTagInfo(&tagInfo, service));

        nn::nfc::server::core::ScopedSession session(service, m_Device->m_Handle);
        NN_RESULT_DO(session.Keep());

        std::unique_ptr<nn::nfc::server::Tag> targetTag(new nn::nfc::server::Tag(m_Device->m_Handle, m_Device->m_SelectedTagId, static_cast<nn::nfc::NfcProtocol>(tagInfo.protocol), static_cast<nn::nfc::TagType>(tagInfo.type), &m_Device->m_AccessFinishEvent, &m_Device->m_AccessResetEvent));

        return m_Device->AfterAccessTag(service, &session, ConvertToNfcPtResultForSendCommand(targetTag->SendCommand(pOutBuffer, pOutSize, service, pData, dataSize, bufferSize, timeout)));
    }
    else
    {
        if(m_Device->m_State == nn::nfc::server::Device::State_Deactive)
        {
            return nn::nfc::ResultTagNotFound();
        }
        return nn::nfc::ResultInvalidDeviceState();
    }
}

nn::Result Device::CheckKeep() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    if(m_Device->m_State != nn::nfc::server::Device::State_Keep)
    {
        if(m_Device->m_State == nn::nfc::server::Device::State_Deactive)
        {
            return nn::nfc::ResultTagNotFound();
        }
        return nn::nfc::ResultInvalidDeviceState();
    }

    NN_RESULT_SUCCESS;
}

nn::Result Device::KeepSession(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(m_Device->CheckActive());

    nn::nfc::TagInfo tagInfo;
    NN_RESULT_DO(m_Device->GetTagInfo(&tagInfo, service));

    std::unique_ptr<nn::nfc::server::core::ScopedSession> session(new nn::nfc::server::core::ScopedSession(service, m_Device->m_Handle));
    NN_RESULT_DO(session->Keep());

    std::unique_ptr<nn::nfc::server::Tag> targetTag(new nn::nfc::server::Tag(m_Device->m_Handle, m_Device->m_SelectedTagId, static_cast<nn::nfc::NfcProtocol>(tagInfo.protocol), static_cast<nn::nfc::TagType>(tagInfo.type), &m_Device->m_AccessFinishEvent, &m_Device->m_AccessResetEvent));
    m_KeptTag = std::move(targetTag);
    m_Session = std::move(session);
    m_Device->SetState(nn::nfc::server::Device::State_Keep);
    NN_RESULT_SUCCESS;
}

nn::Result Device::ReleaseSession() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckKeep());

    m_Session.reset(nullptr);
    m_KeptTag.reset(nullptr);
    m_Device->SetState(nn::nfc::server::Device::State_Active);

    NN_RESULT_SUCCESS;
}

}}}}  // namespace nn::nfc::pt::server
