﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/core/news_NewsMetadataReader.h>
#include <nn/news/detail/service/core/news_IncrementalId.h>
#include <nn/news/detail/service/msgpack/news_MsgpackReader.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

NewsMetadataReader::NewsMetadataReader() NN_NOEXCEPT :
    m_Bits(),
    m_RawNewsId(0),
    m_Version(),
    m_Record(nullptr),
    m_EssentialPickupInfo()
{
}

nn::Result NewsMetadataReader::Read(NewsDatabase::InsertRecord* outRecord, nne::nlib::InputStream& stream, bool isLocal) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outRecord);

    m_Record = outRecord;

    detail::service::msgpack::MsgpackReader reader;

    NN_RESULT_TRY(reader.Parse(stream, *this, sizeof (TopicId)))
        NN_RESULT_CATCH(ResultAbortedByCallback)
        {
            NN_RESULT_THROW(ResultInvalidFormat());
        }
    NN_RESULT_END_TRY;

    if (!m_Bits.newsId)
    {
        NN_RESULT_DO(IncrementalId::GetInstance().Issue(&m_RawNewsId));
    }

    nn::util::SNPrintf(m_Record->newsId.value, sizeof (m_Record->newsId.value),
        "%c%c%020llu", isLocal ? 'L' : 'N', m_Bits.newsId ? 'S' : 'A', m_RawNewsId);

    // 必須項目の存在確認。
    if (m_Bits.publishedAt && m_Bits.pickupLimit && m_Bits.priority && m_Bits.deletionPriority &&
        m_Bits.topicId && m_Bits.surprise && m_Bits.bashotorya)
    {
        if (m_Bits.essentialPickupPickupLimit && m_Bits.essentialPickupPriorityAfter)
        {
            nn::time::PosixTime now;

            if (nn::time::StandardNetworkSystemClock::GetCurrentTime(&now).IsSuccess())
            {
                if (now > m_EssentialPickupInfo.pickupLimit)
                {
                    NN_DETAIL_NEWS_INFO("[news] The essential pickup period has expired. (priority: %d -> %d)\n",
                        m_Record->priority, m_EssentialPickupInfo.priorityAfter);

                    m_Record->priority = m_EssentialPickupInfo.priorityAfter;
                }
            }
            else
            {
                // ネットワーク時刻が取得できない場合、期限切れとみなす。
                m_Record->priority = m_EssentialPickupInfo.priorityAfter;
            }
        }

        NN_RESULT_SUCCESS;
    }
    else
    {
        NN_RESULT_THROW(ResultInvalidFormat());
    }
}

bool NewsMetadataReader::OnStartArray(const detail::service::msgpack::JsonPath& jsonPath) NN_NOEXCEPT
{
    if (!m_Bits.applicationIds &&
        jsonPath.Match("$.application_ids[]"))
    {
        std::memset(m_Record->applicationIds, 0, sizeof (m_Record->applicationIds));
    }

    return true;
}

bool NewsMetadataReader::OnEndArray(const detail::service::msgpack::JsonPath& jsonPath) NN_NOEXCEPT
{
    if (!m_Bits.applicationIds &&
        jsonPath.Match("$.application_ids[]"))
    {
        m_Bits.applicationIds = true;
    }

    return true;
}

bool NewsMetadataReader::OnBoolean(const detail::service::msgpack::JsonPath& jsonPath, bool value) NN_NOEXCEPT
{
    detail::service::msgpack::AnyInteger integer;

    integer.s = value ? 1 : 0;
    integer.isSigned = true;

    // bool を 0/1 にマッピングする。
    return OnInteger(jsonPath, integer);
}

bool NewsMetadataReader::OnInteger(const detail::service::msgpack::JsonPath& jsonPath, const detail::service::msgpack::AnyInteger& value) NN_NOEXCEPT
{
    if (!m_Bits.formatVersion &&
        jsonPath.Compare("$.version.format"))
    {
        m_Version.format = static_cast<int8_t>(value.s);
        m_Bits.formatVersion = true;
    }
    else if (!m_Bits.semanticsVersion &&
        jsonPath.Compare("$.version.semantics"))
    {
        m_Version.semantics = static_cast<int8_t>(value.s);
        m_Bits.semanticsVersion = true;
    }
    else if (!m_Bits.newsId &&
        jsonPath.Compare("$.news_id"))
    {
        m_RawNewsId = value.u;
        m_Bits.newsId = true;
    }
    else if (!m_Bits.publishedAt &&
        jsonPath.Compare("$.published_at"))
    {
        m_Record->publishedAt.value = value.s;
        m_Bits.publishedAt = true;
    }
    else if (!m_Bits.expireAt &&
        jsonPath.Compare("$.expire_at"))
    {
        m_Record->expireAt.value = value.s;
        m_Bits.expireAt = true;
    }
    else if (!m_Bits.pickupLimit &&
        jsonPath.Compare("$.pickup_limit"))
    {
        m_Record->pickupLimit.value = value.s;
        m_Bits.pickupLimit = true;
    }
    else if (!m_Bits.priority &&
        jsonPath.Compare("$.priority"))
    {
        if (value.s < 0)
        {
            return false;
        }
        m_Record->priority = static_cast<int32_t>(value.s);
        m_Bits.priority = true;
    }
    else if (!m_Bits.deletionPriority &&
        jsonPath.Compare("$.deletion_priority"))
    {
        if (value.s < 0)
        {
            return false;
        }
        m_Record->deletionPriority = static_cast<int32_t>(value.s);
        m_Bits.deletionPriority = true;
    }
    else if (!m_Bits.ageLimit &&
        jsonPath.Compare("$.age_limit"))
    {
        if (value.s < 0)
        {
            return false;
        }
        m_Record->ageLimit = static_cast<int32_t>(value.s);
        m_Bits.ageLimit = true;
    }
    else if (!m_Bits.applicationIds &&
        jsonPath.Match("$.application_ids[*]"))
    {
        for (int i = 0; i < NewsDatabase::ApplicationIdCountMax; i++)
        {
            if (m_Record->applicationIds[i].value == 0)
            {
                m_Record->applicationIds[i].value = value.u;
                break;
            }
        }
    }
    else if (!m_Bits.surprise &&
        jsonPath.Compare("$.surprise"))
    {
        m_Record->surprise = static_cast<int32_t>(value.s);
        m_Bits.surprise = true;
    }
    else if (!m_Bits.bashotorya &&
        jsonPath.Compare("$.bashotorya"))
    {
        m_Record->bashotorya = static_cast<int32_t>(value.s);
        m_Bits.bashotorya = true;
    }
    else if (!m_Bits.essentialPickupPickupLimit &&
        jsonPath.Compare("$.essential_pickup.pickup_limit"))
    {
        m_EssentialPickupInfo.pickupLimit.value = value.s;
        m_Bits.essentialPickupPickupLimit = true;
    }
    else if (!m_Bits.essentialPickupPriorityAfter &&
        jsonPath.Compare("$.essential_pickup.priority_after"))
    {
        m_EssentialPickupInfo.priorityAfter = static_cast<int32_t>(value.s);
        m_Bits.essentialPickupPriorityAfter = true;
    }

    return true;
} // NOLINT(impl/function_size)

bool NewsMetadataReader::OnString(const detail::service::msgpack::JsonPath& jsonPath, const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT
{
    NN_UNUSED(length);
    NN_UNUSED(isOverflowed);

    if (!m_Bits.topicId &&
        jsonPath.Compare("$.topic_id"))
    {
        if (!StringVerifier::VerifyTopicId(value))
        {
            return false;
        }
        nn::util::Strlcpy(m_Record->topicId.value, value, sizeof (m_Record->topicId.value));
        m_Bits.topicId = true;
    }

    return true;
}

}}}}}
