﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/core/news_AccountCountryChecker.h>
#include <nn/news/detail/service/core/news_NewsTaskManager.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

AccountCountryChecker::AccountCountryChecker() NN_NOEXCEPT :
    m_Mutex(true),
    m_CurrentQueryString()
{
}

nn::Result AccountCountryChecker::Initialize() NN_NOEXCEPT
{
    UpdateQueryString();

    NN_RESULT_SUCCESS;
}

bool AccountCountryChecker::UpdateQueryString() NN_NOEXCEPT
{
    Account::Country countries[nn::account::UserCountMax] = {};
    int count;

    Account::GetCountryList(&count, countries, nn::account::UserCountMax);

    QueryString queryString = {};
    ConvertCountryListToCountryQueryString(&queryString, countries, count);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (nn::util::Strncmp(queryString.value, m_CurrentQueryString.value, sizeof (queryString.value)) == 0)
    {
        return false;
    }

    m_CurrentQueryString = queryString;

    return true;
}

void AccountCountryChecker::GetQueryString(QueryString* out) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(out);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    std::memcpy(out, &m_CurrentQueryString, sizeof (m_CurrentQueryString));
}

void AccountCountryChecker::ConvertCountryListToCountryQueryString(QueryString* out, const Account::Country* countries, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(out);
    NN_SDK_REQUIRES(count <= nn::account::UserCountMax);

    Account::Country normalized[nn::account::UserCountMax] = {};
    int actualCount = 0;

    for (int i = 0; i < count; i++)
    {
        NN_SDK_REQUIRES(nn::util::Strnlen(countries[i].value, 3) == 2);

        bool isExists = false;

        for (int c = 0; c < actualCount; c++)
        {
            if (nn::util::Strncmp(normalized[c].value, countries[i].value, sizeof (countries[i].value)) == 0)
            {
                isExists = true;
                break;
            }
        }

        if (!isExists)
        {
            nn::util::Strlcpy(normalized[actualCount++].value, countries[i].value, sizeof (countries[i].value));
        }
    }

    if (actualCount == 0)
    {
        out->value[0] = '\0';
        return;
    }

    // 昇順に並べ替える。
    std::sort(normalized, normalized + actualCount,
        [](const Account::Country& lhs, const Account::Country& rhs)
        {
            return nn::util::Strncmp(lhs.value, rhs.value, sizeof (rhs.value)) < 0;
        });

    int length = 0;

    for (int i = 0; i < actualCount; i++)
    {
        // c[]=%s
        length += nn::util::SNPrintf(&out->value[length], sizeof (out->value) - length, "c%%5b%%5d=%s", normalized[i].value);

        if (i < actualCount - 1)
        {
            length += nn::util::SNPrintf(&out->value[length], sizeof (out->value) - length, "&");
        }
    }
}

}}}}}
