﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
//#include <cstdlib>
#include <atomic>

#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/socket.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/netdiag/netdiag_Result.h>
#include <nn/netdiag/netdiag_ResultPrivate.h>
#include <nn/netdiag/detail/netdiag_CurlMulti.h>
#include <nn/netdiag/detail/netdiag_Log.h>

#include "netdiag_Util.h"

namespace nn { namespace netdiag { namespace detail {

namespace {
    // 中断指示
    std::atomic<bool> g_IsInterrupted;
}

//----------------------------------------------------------------
// CURL 実行(curl_multi による中断を考慮した実行)
//
// https://curl.haxx.se/libcurl/c/multi-app.html や
// PR-23500 が参考になる。
//
nn::Result PerformCurlMulti( CURL* curl, CURLcode* pCurlCode ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( curl );
    NN_ABORT_UNLESS_NOT_NULL( pCurlCode );

    // curl マルチハンドラを用意する
    CURLM* curlMultiHandle;
    NN_RESULT_DO(AcquireCurlMultiHandle(&curlMultiHandle));
    curl_multi_add_handle( curlMultiHandle, curl );
    NN_UTIL_SCOPE_EXIT
    {
        curl_multi_remove_handle( curlMultiHandle, curl );
        curl_multi_cleanup(curlMultiHandle);
    };

    int runningCount;
    curl_multi_perform( curlMultiHandle, &runningCount );

    while( runningCount > 0 && ! g_IsInterrupted )
    {
        nn::socket::FdSet fdSetRead;
        nn::socket::FdSet fdSetWrite;
        nn::socket::FdSet fdSetException;
        int fdMax = -1;

        nn::socket::FdSetZero( &fdSetRead );
        nn::socket::FdSetZero( &fdSetWrite );
        nn::socket::FdSetZero( &fdSetException );

        CURLMcode fdsetCode = curl_multi_fdset( curlMultiHandle, &fdSetRead, &fdSetWrite, &fdSetException, &fdMax );

        if ( fdsetCode != CURLM_OK )
        {
            NN_DETAIL_NETDIAG_INFO("[netdiag] curl_multi error (%d).", fdsetCode);
            break;
        }

        int selectCode;
        if ( fdMax == -1 )
        {
            // 準備できている fdset がないの 100ms 待つ
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100) );
            selectCode = 0; // perform を実行するために 0 とする
        }
        else
        {
            // ソケット動作までにどれだけ待つかを決定
            nn::socket::TimeVal socketTimeout = { 1, 0 };
            long curlNextTimeout = -1;
            curl_multi_timeout( curlMultiHandle, &curlNextTimeout );
            if ( curlNextTimeout >= 0 )
            {
                socketTimeout.tv_sec = curlNextTimeout / 1000;
                if ( socketTimeout.tv_sec > 1 )
                {
                    socketTimeout.tv_sec = 1;
                }
                else
                {
                    socketTimeout.tv_usec = (curlNextTimeout % 1000) * 1000;
                }
            }
            selectCode = nn::socket::Select( fdMax + 1, &fdSetRead, &fdSetWrite, &fdSetException, &socketTimeout );
        }

        switch( selectCode )
        {
            case -1: // select error
                NN_DETAIL_NETDIAG_INFO("[netdiag] nn::socket::Select returned %d, last errno is %d.\n", selectCode, nn::socket::GetLastError());
                runningCount = 0;
                break;

            case 0: // timeout
            default: // action
                curl_multi_perform( curlMultiHandle, &runningCount );
                break;
        }
    }

    // 中断されているか
    NN_RESULT_THROW_UNLESS( ! g_IsInterrupted, nn::netdiag::ResultInterrupted() );

    // curl multi スタック情報から結果を取得
    CURLcode code = CURLE_OK;
    for(;;)
    {
        int messageCount; // 残りメッセージ数(未使用)
        CURLMsg* pMessage = curl_multi_info_read( curlMultiHandle, &messageCount );

        if ( pMessage == nullptr )
        {
            break;
        }
        if ( pMessage->msg == CURLMSG_DONE )
        {
            if ( pMessage->easy_handle == curl )
            {
                code = pMessage->data.result;
            }
        }
    }

    *pCurlCode = code;
    NN_RESULT_SUCCESS;
}

//----------------------------------------------------------------
// curl_multi 処理に対し中断指示を出す
//
void InterruptCurlMulti() NN_NOEXCEPT
{
    g_IsInterrupted = true;
}
//----------------------------------------------------------------
// curl_multi 処理に対する中断指示をクリアする
//
void ClearInterruptCurlMulti() NN_NOEXCEPT
{
    g_IsInterrupted = false;
}

}}} // nn::netdiag::detail
