﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/mmnv/mmnv_Api.h>
#include "mmnv_CreateRequest.h"
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/sf/sf_Types.h>
#include <nn/mmnv/sfdl/mmnv.sfdl.h>
#include <nn/os/os_Mutex.h>
#include <mutex>

namespace nn   {
namespace mmnv {

namespace {

nn::sf::SharedPointer<IRequest> g_Request = nullptr;
uint32_t g_RefCount = 0;
nn::os::MutexType g_StaticMutex = NN_OS_MUTEX_INITIALIZER(false);
const int RequestNotInitialized = -1;
}

nn::Result Request::Initialize(Module id, Priority priority, nn::os::EventClearMode eventClearMode)
NN_NOEXCEPT
{
    int requestId;
    nn::Result result;

    if (m_RequestId != RequestNotInitialized)
    {
        // return by the judgement that this instance is already initialized.
        return ResultSuccess();
    }

    nn::os::LockMutex(&g_StaticMutex);
    if (g_Request == nullptr)
    {
        g_Request = nn::mmnv::CreateRequestByHipc();
    }
    g_RefCount++;
    if ((result = g_Request->InitializeById(&requestId, id, priority, eventClearMode)).IsSuccess())
    {
        m_RequestId = requestId;
    }
    nn::os::UnlockMutex(&g_StaticMutex);

    return result;
}

nn::Result Request::Initialize(Module id, Priority priority)
NN_NOEXCEPT
{
    return Initialize(id, priority, nn::os::EventClearMode_ManualClear);
}

nn::Result Request::Finalize()
NN_NOEXCEPT
{
    if (m_RequestId == RequestNotInitialized)
    {
        return ResultSuccess();
    }

    nn::os::LockMutex(&g_StaticMutex);
    NN_SDK_ASSERT_NOT_NULL(g_Request);
    nn::Result result = g_Request->FinalizeById(m_RequestId);
    m_RequestId = RequestNotInitialized;
    if ( g_RefCount != 0 )
    {
        g_RefCount--;
    }
    nn::os::UnlockMutex(&g_StaticMutex);
    return result;
}

nn::Result Request::SetAndWait(Setting min, Setting max)
NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Request);
    NN_SDK_ASSERT(m_RequestId != RequestNotInitialized);
    return g_Request->SetAndWaitById(m_RequestId, min, max);
}

nn::Result Request::Get(Setting* pCurrent)
NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Request);
    NN_SDK_ASSERT(m_RequestId != RequestNotInitialized);
    return g_Request->GetById(pCurrent, m_RequestId);
}

Request::Request()
NN_NOEXCEPT : m_RequestId(RequestNotInitialized)
{
}

Request::~Request()
NN_NOEXCEPT
{
    Finalize();

    nn::os::LockMutex(&g_StaticMutex);
    if (g_Request != nullptr && g_RefCount == 0)
    {
        g_Request = nullptr;
    }
    nn::os::UnlockMutex(&g_StaticMutex);
}

}}

