﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <mutex>
#include <nn/htcs.h>
#include <nn/htcs/htcs_LibraryPrivate.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/util/util_ScopeExit.h>
#include "lm_LogServerProxy.h"
#include "../detail/lm_Log.h"

namespace nn { namespace lm { namespace impl {

namespace
{
    const char* PortName = "iywys@$LogManager";
    const nn::TimeSpan PollingInterval = nn::TimeSpan::FromSeconds(1);
    const int HtcsSessionCountMax = 2;

    //!< htcs 用ヒープのメモリ領域
    char g_HeapBuffer[2 * 1024];
}

LogServerProxy::LogServerProxy() NN_NOEXCEPT
    : m_StopEvent(nn::os::EventClearMode_ManualClear)
    , m_ConnectionMutex(false)
    , m_ObserverMutex(false)
    , m_ServerSocket(-1)
    , m_ClientSocket(-1)
    , m_ConnectionObserver(nullptr)
{}

void LogServerProxy::LoopAuto() NN_NOEXCEPT
{
    const auto htcsWorkingMemorySize = nn::htcs::GetWorkingMemorySize(HtcsSessionCountMax);
    NN_ABORT_UNLESS(htcsWorkingMemorySize <= sizeof(g_HeapBuffer));

    nn::htcs::InitializeForDisableDisconnectionEmulation(g_HeapBuffer, htcsWorkingMemorySize, HtcsSessionCountMax);

    NN_UTIL_SCOPE_EXIT
    {
        nn::htcs::Finalize();
    };

    nn::htcs::SockAddrHtcs serverAddress;
    serverAddress.family = nn::htcs::HTCS_AF_HTCS;
    serverAddress.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(serverAddress.portName.name, PortName);

    do
    {
        auto serverSocket = nn::htcs::Socket();
        if (!(serverSocket >= 0))
        {
            NN_DETAIL_LM_ERROR("Failed to get the server socket. error = %d\n", nn::htcs::GetLastError());
            continue;
        }
        m_ServerSocket = serverSocket;

        NN_UTIL_SCOPE_EXIT
        {
            if (!(nn::htcs::Close(serverSocket) == 0))
            {
                NN_DETAIL_LM_ERROR("Failed to close the server socket. error = %d\n", nn::htcs::GetLastError());
            }
            m_ServerSocket = -1;
        };

        if (!(nn::htcs::Bind(serverSocket, &serverAddress) == 0))
        {
            NN_DETAIL_LM_ERROR("Failed to bind. error = %d\n", nn::htcs::GetLastError());
            continue;
        }

        if (!(nn::htcs::Listen(serverSocket, 0) == 0))
        {
            NN_DETAIL_LM_ERROR("Failed to listen. error = %d\n", nn::htcs::GetLastError());
            continue;
        }

        while (!m_StopEvent.TryWait())
        {
            NN_DETAIL_LM_INFO("Waiting for connection from Target Manager.\n");

            auto clientSocket = nn::htcs::Accept(serverSocket, nullptr);
            if (!(clientSocket >= 0))
            {
                NN_DETAIL_LM_ERROR("Failed to accept. error = %d\n", nn::htcs::GetLastError());
                break;
            }
            m_ClientSocket = clientSocket;

            NN_DETAIL_LM_INFO("Connected to Target Manager.\n");
            InvokeConnectionObserver(true);
            SignalConnection();

            NN_UTIL_SCOPE_EXIT
            {
                if (!(nn::htcs::Close(clientSocket) == 0))
                {
                    NN_DETAIL_LM_ERROR("Failed to close the client socket. error = %d\n", nn::htcs::GetLastError());
                }
                m_ClientSocket = -1;

                NN_DETAIL_LM_INFO("Disconnected to Target Manager.\n");
                InvokeConnectionObserver(false);
            };

            // 切断するまでブロックする。
            uint8_t c;
            while (nn::htcs::Recv(clientSocket, &c, sizeof(c), 0) == sizeof(c));
            NN_DETAIL_LM_ERROR("Failed to receive. error = %d\n", nn::htcs::GetLastError());
        }
    } while (!m_StopEvent.TimedWait(PollingInterval));
}

void LogServerProxy::Start() NN_NOEXCEPT
{
    auto threadFunction = [](void* argument)
    {
        reinterpret_cast<LogServerProxy*>(argument)->LoopAuto();
    };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_Thread, threadFunction, this, &m_ThreadStack, sizeof(m_ThreadStack), NN_SYSTEM_THREAD_PRIORITY(lm, HtcsConnection)));
    nn::os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(lm, HtcsConnection));
    m_StopEvent.Clear();
    nn::os::StartThread(&m_Thread);
}

void LogServerProxy::Stop() NN_NOEXCEPT
{
    m_StopEvent.Signal();

    int clientSocket = m_ClientSocket;
    if (clientSocket >= 0)
    {
        if (!(nn::htcs::Close(clientSocket) == 0))
        {
            NN_DETAIL_LM_ERROR("Failed to close the client socket. error = %d\n", nn::htcs::GetLastError());
        }
    }

    int serverSocket = m_ServerSocket;
    if (serverSocket >= 0)
    {
        if (!(nn::htcs::Close(serverSocket) == 0))
        {
            NN_DETAIL_LM_ERROR("Failed to close the server socket. error = %d\n", nn::htcs::GetLastError());
        }
    }

    nn::os::WaitThread(&m_Thread);
    nn::os::DestroyThread(&m_Thread);
}

bool LogServerProxy::IsConnected() NN_NOEXCEPT
{
    return (m_ClientSocket >= 0);
}

void LogServerProxy::WaitToConnect() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ConnectionMutex);

    while (!IsConnected())
    {
        m_ConditionConnected.Wait(m_ConnectionMutex);
    }
}

void LogServerProxy::SignalConnection() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ConnectionMutex);

    m_ConditionConnected.Broadcast();
}

void LogServerProxy::SetConnectionObserver(ConnectionObserver observer) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ObserverMutex);

    m_ConnectionObserver = observer;
}

void LogServerProxy::InvokeConnectionObserver(bool connected) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ObserverMutex);

    if (m_ConnectionObserver)
    {
        m_ConnectionObserver(connected);
    }
}

bool LogServerProxy::Send(const uint8_t* data, size_t size) NN_NOEXCEPT
{
    size_t offset = 0;
    while (IsConnected() && offset < size)
    {
        auto result = nn::htcs::Send(m_ClientSocket, data + offset, size - offset, 0);
        if (result >= 0)
        {
            offset += static_cast<size_t>(result);
            if (static_cast<size_t>(result) < size) // 送信が分割されたときのみログ出力する。
            {
                NN_DETAIL_LM_INFO("Sending %zu/%zu bytes.\n", offset, size);
            }
        }
        else
        {
            NN_DETAIL_LM_ERROR("Failed to send. error = %d\n", nn::htcs::GetLastError());

            if (!(nn::htcs::Shutdown(m_ClientSocket, nn::htcs::HTCS_SHUT_RDWR) == 0))
            {
                NN_DETAIL_LM_ERROR("Failed to shutdown the client socket. error = %d\n", nn::htcs::GetLastError());
            }

            // 切断検知させるために、スリープして、実行権を委譲する。
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

            return false;
        }
    }
    NN_SDK_ASSERT_LESS_EQUAL(offset, size);
    return offset == size;
}

}}} // nn::lm::impl
