﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/os.h>
#include <nn/time.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include "lm_EventLogTransmitter.h"
#include "lm_LogBuffer.h"
#include "lm_LogServerProxy.h"
#include "lm_SdCardLogger.h"

namespace nn { namespace lm { namespace impl {

bool IsSleeping() NN_NOEXCEPT;

namespace {
    std::aligned_storage<32 * 1024>::type g_FsHeapStorage;
    nn::lmem::HeapHandle g_FsHeapHandle;

    void* Allocate(size_t size)
    {
        return nn::lmem::AllocateFromExpHeap(g_FsHeapHandle, size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        return nn::lmem::FreeToExpHeap(g_FsHeapHandle, p);
    }
}

namespace {
    std::aligned_storage<8 * 1024, os::ThreadStackAlignment>::type g_ThreadStack;
    os::ThreadType g_Thread;

    os::Event g_StopEvent(nn::os::EventClearMode_ManualClear);
    os::Event g_SdLoggingEvent(nn::os::EventClearMode_ManualClear);
#if !defined(NN_DETAIL_LM_WITHOUT_TMA)
    os::Event g_HostConnectionEvent(nn::os::EventClearMode_ManualClear);
#endif
    std::unique_ptr<fs::IEventNotifier> g_SdCardDetectionEventNotifier;
    os::SystemEvent g_SdCardDetectionEvent;

    bool WaitForFlush() NN_NOEXCEPT
    {
        for (;;)
        {
            nn::os::WaitAny(
                g_StopEvent.GetBase(),
    #if !defined(NN_DETAIL_LM_WITHOUT_TMA)
                g_HostConnectionEvent.GetBase(),
    #endif
                g_SdLoggingEvent.GetBase(),
                g_SdCardDetectionEvent.GetBase());

            if (g_StopEvent.TryWait())
            {
                return false;
            }

    #if !defined(NN_DETAIL_LM_WITHOUT_TMA)
            if (g_HostConnectionEvent.TryWait())
            {
                return true;
            }
    #endif

            if (g_SdLoggingEvent.TryWait())
            {
                return true;
            }

            if (g_SdCardDetectionEvent.TryWait())
            {
                g_SdCardDetectionEvent.Clear();
                if (nn::fs::IsSdCardInserted())
                {
                    return true;
                }
            }
        }
    }

#if !defined(NN_DETAIL_LM_WITHOUT_TMA)
    void HostConnectionObserver(bool isConnected) NN_NOEXCEPT
    {
        if (isConnected)
        {
            g_HostConnectionEvent.Signal();
        }
        else
        {
            g_HostConnectionEvent.Clear();

            if (!g_SdLoggingEvent.TryWait())
            {
                LogBuffer::GetDefaultInstance().CancelPush();
            }
        }
    }
#endif

    void SdLoggingObserver(bool isAvailable) NN_NOEXCEPT
    {
        if (isAvailable)
        {
            g_SdLoggingEvent.Signal();
        }
        else
        {
            g_SdLoggingEvent.Clear();

#if !defined(NN_DETAIL_LM_WITHOUT_TMA)
            if (!g_HostConnectionEvent.TryWait())
            {
                LogBuffer::GetDefaultInstance().CancelPush();
            }
#endif
        }
    }

    void ThreadFunction(void*) NN_NOEXCEPT
    {
        nn::fs::SetEnabledAutoAbort(false);

        g_FsHeapHandle = nn::lmem::CreateExpHeap(&g_FsHeapStorage, sizeof(g_FsHeapStorage), nn::lmem::CreationOption_NoOption);
        NN_ABORT_UNLESS(g_FsHeapHandle != nullptr);
        nn::fs::SetAllocator(Allocate, Deallocate);

        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenSdCardDetectionEventNotifier(&g_SdCardDetectionEventNotifier));
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_SdCardDetectionEventNotifier->BindEvent(g_SdCardDetectionEvent.GetBase(), nn::os::EventClearMode_ManualClear));

        nn::time::Initialize();

        SdCardLogger::GetInstance().SetLoggingObserver(SdLoggingObserver);

#if !defined(NN_DETAIL_LM_WITHOUT_TMA)
        LogServerProxy::GetInstance().SetConnectionObserver(HostConnectionObserver);
#endif

        do
        {
            if (LogBuffer::GetDefaultInstance().Flush())
            {
                EventLogTransmitter::GetDefaultInstance().PushLogPacketDropCountIfExists();
            }
        }
        while (WaitForFlush());

#if !defined(NN_DETAIL_LM_WITHOUT_TMA)
        LogServerProxy::GetInstance().SetConnectionObserver(nullptr);
#endif

        nn::lm::impl::SdCardLogger::GetInstance().~SdCardLogger();

        SdCardLogger::GetInstance().SetLoggingObserver(nullptr);

        nn::time::Finalize();

        nn::lmem::DestroyExpHeap(g_FsHeapHandle);
    }
}

bool IsFlushAvailable() NN_NOEXCEPT
{
    if (IsSleeping())
    {
        return false;
    }

    const int stopEventIndex = 0;

#if defined(NN_DETAIL_LM_WITHOUT_TMA)
    return nn::os::TryWaitAny(
        g_StopEvent.GetBase(),
        g_SdLoggingEvent.GetBase()) > stopEventIndex;
#else
    return nn::os::TryWaitAny(
        g_StopEvent.GetBase(),
        g_HostConnectionEvent.GetBase(),
        g_SdLoggingEvent.GetBase()) > stopEventIndex;
#endif
}

void InitializeFlushThread() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&g_Thread, ThreadFunction, nullptr, &g_ThreadStack, sizeof(g_ThreadStack), NN_SYSTEM_THREAD_PRIORITY(lm, Flush)));
    os::SetThreadNamePointer(&g_Thread, NN_SYSTEM_THREAD_NAME(lm, Flush));
    g_StopEvent.Clear();
    os::StartThread(&g_Thread);
}

void FinalizeFlushThread() NN_NOEXCEPT
{
    g_StopEvent.Signal();
    os::WaitThread(&g_Thread);
    os::DestroyThread(&g_Thread);
}

}}} // nn::lm::impl
