﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/os.h>
#include <nn/lbl/detail/lbl_Log.h>
#include <nn/lbl/impl/lbl_AutoControlServer.h>

#include "lbl_AutoControlThread.h"

namespace {

bool g_IsTerminatingAutoControlThread = false;
bool g_IsThreadStarted = false;

// 自動輝度調整スレッド関連
const size_t LblControlStackSize = 4096;
NN_ALIGNAS(4096) char g_LblControlStack[LblControlStackSize];
nn::os::ThreadType g_LblControlThread;

void LblBrightnessControlFunction(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    NN_DETAIL_LBL_TRACE("Start brightness control listener.\n");

    const nn::TimeSpan BrightnessControlInterval = nn::TimeSpan::FromMilliSeconds(16);  // 60fps
    while ( !g_IsTerminatingAutoControlThread )
    {
        nn::lbl::impl::UpdateBrightness();
        nn::os::SleepThread(BrightnessControlInterval);
    }
}

} // namespace

namespace nn {
namespace lbl {
namespace server {

void StartAutoControlThread() NN_NOEXCEPT
{
    if ( !g_IsThreadStarted )
    {
        // 輝度制御スレッドの作成。
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_LblControlThread, &LblBrightnessControlFunction,
            nullptr, g_LblControlStack, sizeof(g_LblControlStack), NN_SYSTEM_THREAD_PRIORITY(lbl, AutoControl)));
        nn::os::SetThreadNamePointer(&g_LblControlThread, NN_SYSTEM_THREAD_NAME(lbl, AutoControl));
        nn::os::StartThread(&g_LblControlThread);
        g_IsThreadStarted = true;
    }
}

void StopAutoControlThread() NN_NOEXCEPT
{
    if ( g_IsThreadStarted )
    {
        g_IsTerminatingAutoControlThread = true;

        // 輝度制御スレッド終了待機と破棄。
        nn::os::WaitThread(&g_LblControlThread);
        nn::os::DestroyThread(&g_LblControlThread);

        g_IsTerminatingAutoControlThread = false;
        g_IsThreadStarted = false;
    }
}

} // server
} // lbl
} // nn
