﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/lbl/detail/lbl_Log.h>

#include "lbl_DeviceAccessorBacklight.h"

namespace nn { namespace lbl { namespace impl { namespace detail {

namespace {

    const int PwmBaseFrequency = static_cast<int>(29.5 * 1000); // PWMの周波数(29.5kHz)

}

DeviceAccessorBacklight::DeviceAccessorBacklight() NN_NOEXCEPT
{
}

void DeviceAccessorBacklight::Initialize() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);

    ++m_InitializeCount;
    if ( m_InitializeCount == 1 )
    {
        nn::gpio::Initialize();
        nn::gpio::OpenSession(&m_GpioSession, nn::gpio::GpioPadName_PowLcdBlEn);
        nn::gpio::SetDirection(&m_GpioSession, nn::gpio::Direction_Output);
        nn::gpio::SetValue(&m_GpioSession, nn::gpio::GpioValue_Low); // 消灯
        m_IsBacklightEnergyOn = false;

        nn::pwm::Initialize();
        nn::pwm::OpenSession(&m_PwmSession, nn::pwm::ChannelName::ChannelName_LcdBacklight);
        const auto TargetPeriod = nn::TimeSpan::FromNanoSeconds((1000 * 1000 * 1000) / PwmBaseFrequency);
        nn::pwm::SetPeriod(&m_PwmSession, TargetPeriod);
        nn::pwm::SetDuty(&m_PwmSession, 0);
        nn::pwm::SetEnabled(&m_PwmSession, true);
    }
}

void DeviceAccessorBacklight::Finalize() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);

    --m_InitializeCount;
    if ( m_InitializeCount == 0 )
    {
        nn::pwm::SetEnabled(&m_PwmSession, false);
        nn::pwm::CloseSession(&m_PwmSession);
        nn::pwm::Finalize();

        m_IsBacklightEnergyOn = false;
        nn::gpio::SetValue(&m_GpioSession, nn::gpio::GpioValue_Low); // 消灯
        nn::gpio::CloseSession(&m_GpioSession);
        nn::gpio::Finalize();
    }
}

void DeviceAccessorBacklight::SetDuty(int duty) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(duty, 0, 256);
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);

    nn::pwm::SetDuty(&m_PwmSession, duty);
}

void DeviceAccessorBacklight::SetPower(bool enable) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);

    if (!m_IsBacklightEnergyOn && enable)
    {
        nn::gpio::SetValue(&m_GpioSession, nn::gpio::GpioValue_High);
    }
    else if (m_IsBacklightEnergyOn && !enable)
    {
        nn::gpio::SetValue(&m_GpioSession, nn::gpio::GpioValue_Low);
    }
    m_IsBacklightEnergyOn = enable;
}

}}}} // namespace nn::lbl::impl::detail
