﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/erpt.h>
#include <nn/err.h>
#include <nn/err/detail/err_ErrorCodeConvert.h>
#include <nn/i2c/i2c_Result.h>
#include <nnd/bh1730fvc/bh1730fvc.h>
#include <nn/lbl/lbl_ResultPrivate.h>
#include <nn/lbl/detail/lbl_Log.h>

#include "lbl_DeviceAccessorAls.h"

#define NN_DETAIL_LBL_RESULT_DO_WITH_ERRORMSG(expression) \
    do \
    { \
        auto _nn_detail_lbl_result = (expression); \
        NN_RESULT_TRY(_nn_detail_lbl_result) \
            NN_RESULT_CATCH_ALL \
            { \
                NN_DETAIL_LBL_ERROR("Failed: %s\n  Module: %d\n  Description: %d\n  InnerValue: 0x%08x\n", \
                    NN_MACRO_STRINGIZE(expression), _nn_detail_lbl_result.GetModule(), \
                    _nn_detail_lbl_result.GetDescription(), _nn_detail_lbl_result.GetInnerValueForDebug()); \
                NN_RESULT_RETHROW; \
            } \
        NN_RESULT_END_TRY \
    } while ( NN_STATIC_CONDITION(false) )


namespace nn { namespace lbl { namespace impl { namespace detail {

namespace {

    class ErrorReporter
    {
    public:
        void SubmitError(nn::Result errorCodeResult, nn::Result detailResult) NN_NOEXCEPT
        {
            // 同じエラーが繰り返し起きる場合、エラーレポートを再送しない
            // このステートはアクセサの再初期化（再起動 or スリープ復帰）によりクリアされる
            bool* pHasSubmitted = nullptr;
            if ( errorCodeResult <= nn::lbl::ResultAlsDeviceAccessFailed() )
            {
                pHasSubmitted = &m_HasSubmittedAlsDeviceAccessFailed;
            }
            if ( pHasSubmitted && *pHasSubmitted )
            {
                return;
            }

            auto submitResult = SubmitErrorCore(errorCodeResult, detailResult);
            if ( submitResult.IsSuccess() && pHasSubmitted )
            {
                *pHasSubmitted = true;
            }
        }

        void ResetState() NN_NOEXCEPT
        {
            m_HasSubmittedAlsDeviceAccessFailed = false;
        }

    private:
        nn::Result SubmitErrorCore(nn::Result errorCodeResult, nn::Result detailResult) NN_NOEXCEPT
        {
            nn::erpt::Context context(nn::erpt::CategoryId::ErrorInfo);

            // 解析用に生の result 値を埋めておく
            uint32_t resultArray[1];
            resultArray[0] = detailResult.GetInnerValueForDebug();
            NN_DETAIL_LBL_RESULT_DO_WITH_ERRORMSG(context.Add(nn::erpt::FieldId::ResultBacktrace, resultArray, 1));

            // 集計上は照度センサの誤動作を意味するコードに抽象化する
            auto errorCode = nn::err::detail::ConvertResultToErrorCode(errorCodeResult);
            char errorCodeString[nn::err::ErrorCode::StringLengthMax];
            util::SNPrintf(errorCodeString, sizeof(errorCodeString), "%04d-%04d", errorCode.category, errorCode.number);
            NN_DETAIL_LBL_RESULT_DO_WITH_ERRORMSG(context.Add(nn::erpt::FieldId::ErrorCode, errorCodeString, static_cast<uint32_t>(sizeof(errorCodeString))));
            NN_DETAIL_LBL_RESULT_DO_WITH_ERRORMSG(context.CreateReport(nn::erpt::ReportType_Invisible));

            NN_DETAIL_LBL_ERROR("Error Report %s (ResultBacktrace = %d:%d) has been sent\n",
                errorCodeString, detailResult.GetModule(), detailResult.GetDescription());
            NN_RESULT_SUCCESS;
        }

    private:
        bool m_HasSubmittedAlsDeviceAccessFailed = false;
    } g_ErrorReporter;

    // 照度センサの設定パラメータ
    const nnd::bh1730fvc::Gain ConfigGain = nnd::bh1730fvc::Gain::X64;

}

DeviceAccessorAls::DeviceAccessorAls(const SettingsHolder* pSettingsHolder) NN_NOEXCEPT
    : m_pSettingsHolder(pSettingsHolder)
{
}

void DeviceAccessorAls::Initialize() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);

    ++m_InitializeCount;
    if ( m_InitializeCount == 1 )
    {
        g_ErrorReporter.ResetState();
        m_IsDeviceInError = false; // すでにデバイスエラーに落ちていても、スリープ復帰時には復活のチャンスを与える
        if ( m_pSettingsHolder->IsAlsEnabled() )
        {
            const int MaxInitializeRetryCount = 5;
            const nn::TimeSpan InitializeRetryInterval = nn::TimeSpan::FromMilliSeconds(50);

            int retryCount = 0;

            while ( NN_STATIC_CONDITION(true) )
            {
                if ( nnd::bh1730fvc::Initialize() )
                {
                    break;
                }
                else
                {
                    retryCount++;
                    if ( retryCount >= MaxInitializeRetryCount )
                    {
                        NN_DETAIL_LBL_ERROR("Ambient Light Sensor was not detected\n");
                        return;
                    }
                    nn::os::SleepThread(InitializeRetryInterval);
                }
            }

            // ゲイン設定を変更。
            if ( HandleResultFromDriver(nnd::bh1730fvc::SetGain(ConfigGain)).IsFailure() )
            {
                NN_DETAIL_LBL_ERROR("SetGain failed\n");
                nnd::bh1730fvc::Finalize();
                return;
            }

            // 連続して値を取得。
            if ( HandleResultFromDriver(nnd::bh1730fvc::SetMeasurementMode(nnd::bh1730fvc::MeasurementMode::Continuous)).IsFailure() )
            {
                NN_DETAIL_LBL_ERROR("SetMeasurementMode failed\n");
                nnd::bh1730fvc::Finalize();
                return;
            }
            m_IsDeviceOpen = true;
        }
    }
}

void DeviceAccessorAls::Finalize() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);

    --m_InitializeCount;
    if ( m_InitializeCount == 0 )
    {
        if ( m_IsDeviceOpen )
        {
            m_IsDeviceOpen = false;

            // スタンバイモードに遷移。
            HandleResultFromDriver(nnd::bh1730fvc::SetMeasurementMode(nnd::bh1730fvc::MeasurementMode::Standby));

            // 照度センサードライバライブラリの Finalize
            nnd::bh1730fvc::Finalize();
        }
    }
}

void DeviceAccessorAls::UpdateIlluminance() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);

    if ( !IsDeviceAccessible() )
    {
        return;
    }

    nnd::bh1730fvc::MeasurementValue value = { 0.0f, 0, 0, false };
    auto result = HandleResultFromDriver(nnd::bh1730fvc::GetMeasurementValue(&value));
    if ( result.IsSuccess() )
    {
        // 照度値が負値を返す場合は 0 に丸めて良い
        if ( value.lux < 0 )
        {
            value.lux = 0;
        }

        m_CurrentIlluminance = value.lux;
        m_IsCurrentIlluminanceOverflown = value.isOverflown;
    }
    else
    {
        NN_DETAIL_LBL_WARN(
            "bh1730fvc::GetMeasurementValue() returned 0x%x, skipping sensor value update\n",
            result.GetInnerValueForDebug());
    }
}

nn::Result DeviceAccessorAls::HandleResultFromDriver(nn::Result result) NN_NOEXCEPT
{
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(nn::i2c::ResultBusBusy)
        {
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH_ALL
        {
            g_ErrorReporter.SubmitError(nn::lbl::ResultAlsDeviceAccessFailed(), result);
            m_IsDeviceInError = true; // ライブラリの再初期化までアクセスを試行しない
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY
    NN_RESULT_THROW(result);
}

}}}} // namespace nn::lbl::impl::detail
