﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "lbl_AutoBrightnessAdjuster.h"
#include <cmath>
#include <limits>

namespace {

float g_PreApplyValue;                      // 前フレームで画面に適用させた明るさ。ただしclampさせる前の値を保存する

// 定数
const float BrightnessMinValue = 0.15f;     // 自動輝度調整の明るさの最低値
const float AutoBrightFrame = 60.f;         // 暗い方から明るい方への遅延レベル
const float AutoDarkFrame = 170.f;          // 明るい方から暗い方への遅延レベル

const float LowerCurveCoefficient = 2.f;        // 暗い側曲線の曲がり具合
const float LowerCenterLux        = 700.f;      // 暗い側曲線における環境の明るさの基準値
const float LowerBrightnessMin    = 0.15f;      // 暗い側曲線の最低値
const float LowerBrightnessMax    = 0.4776f;    // 暗い側曲線の最高値

const float UpperCurveCoefficient = 1.f;        // 明るい側曲線の曲がり具合
const float UpperCenterLux        = 400.f;      // 明るい側曲線における環境の明るさの基準値
const float UpperBrightnessMin    = 0.5f;       // 明るい側曲線の最低値
const float UpperBrightnessMax    = 1.f;        // 明るい側曲線の最高値

// 最適化のための定数
const float DenominatorLower = std::log10f( LowerCurveCoefficient + 1.f );
const float DenominatorUpper = std::log10f( UpperCurveCoefficient + 1.f );

const float ManualBrightnessMinValue = 0.063f;  // 手動輝度調整の明るさの最低値。最終的に duty の値が 1 になるような値

float CalcCurveValueLower( float sensor )
{
    // 暗い側曲線
    const float ccur = LowerCurveCoefficient;
    const float clux = LowerCenterLux;
    const float bmin = LowerBrightnessMin;
    const float bmax = LowerBrightnessMax;

    double ccurMultiSensor = static_cast<double>(ccur) * sensor;
    double lower = (std::log10((ccurMultiSensor / clux) + 1.f) / DenominatorLower) * (bmax - bmin) + bmin;
    return static_cast<float>(lower);
}

float CalcCurveValueUpper( float sensor )
{
    // 明るい側曲線
    const float ccur = UpperCurveCoefficient;
    const float clux = UpperCenterLux;
    const float bmin = UpperBrightnessMin;
    const float bmax = UpperBrightnessMax;

    return ( std::log10f( ( ( ccur * sensor ) / clux ) + 1.f ) / DenominatorUpper ) * ( bmax - bmin ) + bmin;
}

float Clamp( float low, float value, float high )
{
    float ret = value < low ? low : value;
    return ret > high ? high : ret;
}

float ClampMax( float value, float high )
{
    // 上端のみclamp
    return value > high ? high : value;
}

float GetLowPassFilteredValue( float value )
{
    // 滑らかに値を遷移させる。Auto***Frameが大きいほどゆっくりと遷移

    float lowpassedValue = 0.f;
    if ( value > g_PreApplyValue )
    {
        // 暗→明方向
        lowpassedValue = value * 1.f / AutoBrightFrame + g_PreApplyValue * ( AutoBrightFrame - 1.f ) / AutoBrightFrame;
    }
    else
    {
        // 明→暗方向
        lowpassedValue = value * 1.f / AutoDarkFrame + g_PreApplyValue * ( AutoDarkFrame - 1.f ) / AutoDarkFrame;
    }

    g_PreApplyValue = lowpassedValue;
    return lowpassedValue;
}

float ApplyNow( float value )
{
    // 即座に画面に反映させる明るさを計算
    g_PreApplyValue = Clamp( BrightnessMinValue, value, 1.f );
    return g_PreApplyValue;
}

float UpdateWithSensorValue( float sensor, float brightness )
{
    // 照度センサーと輝度設定値の値に応じた値を計算
    float upperValue = CalcCurveValueUpper( sensor );
    float lowerValue = CalcCurveValueLower( sensor );
    double brightnessTmp = static_cast<double>(brightness);
    double calcValueTmp = brightnessTmp * upperValue + (1.0 - brightnessTmp) * lowerValue;
    float calcValue = (calcValueTmp < std::numeric_limits<float>::max()) ?
        static_cast<float>(calcValueTmp) : std::numeric_limits<float>::max();
    float upperLimit = brightness * UpperBrightnessMax + ( 1.f - brightness ) * LowerBrightnessMax;
    return ClampMax( calcValue, upperLimit );
}

} // namespace

namespace nn { namespace lbl { namespace impl { namespace detail {

float AutoBrightnessAdjuster::UpdateForAutoBrightness( float sensor, float brightness, bool applyNow )
{
    float appliedValue = UpdateWithSensorValue( sensor, brightness );

    if ( applyNow )
    {
        appliedValue = ApplyNow( appliedValue );
    }
    else
    {
        appliedValue = GetLowPassFilteredValue( appliedValue );
        appliedValue = Clamp( BrightnessMinValue, appliedValue, 1.f );
    }

    return appliedValue;
}

float AutoBrightnessAdjuster::UpdateForManualBrightness(float brightness)
{
    // [ ManualBrightnessMinValue, 1.0f ] に写像
    return brightness * (1.f - ManualBrightnessMinValue) + ManualBrightnessMinValue;
}

float AutoBrightnessAdjuster::GetMinManualBrightness() const
{
    return ManualBrightnessMinValue;
}

}}}} // nn::lbl::impl::detail
