﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ライブラリアプレット起動に関する非公開ヘッダ
 */

#include <nn/la/la_Api.h>

#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/os.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>

#include <nn/la/la_Configs.h>
#include <nn/la/la_Result.h>
#include <nn/applet/applet_LibraryApplet.h>
#include <algorithm>

namespace nn { namespace la {

namespace {
    LibraryAppletCreateHook g_CreateHook = nullptr;
    bool                    g_UseJumpLibrayApplet = false;

    //---------------------------------------------------------------------------
    // LibraryAppletExitReason_Abnormal なら abort する
    //---------------------------------------------------------------------------
    void abortIfAbnormal( nn::applet::LibraryAppletExitReason reason )
    {
        if( reason == nn::applet::LibraryAppletExitReason_Abnormal )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::la::ResultLibraryAppletExitedAbnormally() );
        }
    }

    //---------------------------------------------------------------------------
    // LibraryAppletExitReason_Unexpected なら abort する
    //---------------------------------------------------------------------------
    void abortIfUnexpected( nn::applet::LibraryAppletExitReason reason )
    {
        if( reason == nn::applet::LibraryAppletExitReason_Unexpected )
        {
            // Unexpectedはデバッグで撲滅する必要がある
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::la::ResultLibraryAppletExitedUnexpectedly() );
        }
    }

    //---------------------------------------------------------------------------
    // LibraryAppletExitReason_Abnormal または LibraryAppletExitReason_Unexpected なら abort する
    //---------------------------------------------------------------------------
    void abortIfAbnormalOrUnexpected( nn::applet::LibraryAppletExitReason reason )
    {
        // SA が LA を強制終了させたとき LibraryAppletExitReason_Canceled になる
        // それは許容する必要があるので abort しない
        // 異常終了のときは ABORT でいいはず。
        // ポリシー：http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=164702960
        abortIfAbnormal( reason );
        abortIfUnexpected( reason );
    }
}

//---------------------------------------------------------------------------
void LibraryAppletCreateHookArg::Initialize(nn::applet::AppletId appletId_, nn::applet::LibraryAppletMode libraryAppletMode_, bool isExtremity_ ) NN_NOEXCEPT
{
    appletId = appletId_;
    libraryAppletMode = libraryAppletMode_;
    isExtremity = isExtremity_;
}

//---------------------------------------------------------------------------
void SetLibraryAppletCreateHook(LibraryAppletCreateHook hook) NN_NOEXCEPT
{
    g_CreateHook = hook;
}

//---------------------------------------------------------------------------
LibraryAppletCreateHookResult InvokeLibraryAppletCreateHook(const LibraryAppletCreateHookArg& arg) NN_NOEXCEPT
{
    if ( ! g_CreateHook ) { return LibraryAppletCreateHookResult_Accept; }

    return g_CreateHook( arg );
}

//---------------------------------------------------------------------------
void PushToInChannel( nn::applet::LibraryAppletHandle laHandle, const void* pBuffer, size_t bufferSize ) NN_NOEXCEPT
{
    nn::applet::StorageHandle storage_handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateStorage( &storage_handle, bufferSize) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::WriteToStorage( storage_handle, 0, pBuffer, bufferSize ) );
    nn::applet::PushToInChannel( laHandle, storage_handle );
}

//---------------------------------------------------------------------------
void PopFromOutChannel( nn::applet::LibraryAppletHandle laHandle, void* pBuffer, size_t bufferSize, size_t* pReadSize ) NN_NOEXCEPT
{
    nn::applet::StorageHandle storage_handle;
    size_t read_size = 0;
    // ストレージの取得を試みる
    if ( nn::applet::TryPopFromOutChannel( &storage_handle, laHandle ) )
    {
        read_size = std::min( nn::applet::GetStorageSize(storage_handle), bufferSize );

        if ( pBuffer )
        {
            // データがあるなら読み込む
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::ReadFromStorage( storage_handle, 0, pBuffer, read_size ) );
        }

        nn::applet::ReleaseStorage( storage_handle );
    }

    if ( pReadSize )
    {
        *pReadSize = read_size;
    }
}

//---------------------------------------------------------------------------
nn::Result CreateLibraryAppletEasy(
    nn::applet::LibraryAppletHandle* pOut,
    nn::applet::AppletId appletId,
    nn::applet::LibraryAppletMode    libraryAppletMode,
    bool isExtremity ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pOut != NULL);

    LibraryAppletCreateHookArg hookArg;
    hookArg.Initialize(appletId, libraryAppletMode, isExtremity);
    LibraryAppletCreateHookResult hookResult = InvokeLibraryAppletCreateHook( hookArg );

    if ( hookResult == LibraryAppletCreateHookResult_Reject )
    {
        // nn::la::ResultRejected() を返すのは、アプレットに対してのみ。アプリにこのエラーが返る事はない。
        return nn::la::ResultLibraryAppletRejected();
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateLibraryApplet(
        pOut,
        appletId,
        libraryAppletMode));

    return nn::ResultSuccess();
}

//---------------------------------------------------------------------------
void UseJumpLibraryApplet(bool use) NN_NOEXCEPT
{
    g_UseJumpLibrayApplet = use;
}

//---------------------------------------------------------------------------
nn::Result StartLibraryApplet( nn::applet::LibraryAppletHandle laHandle, bool isExtremity) NN_NOEXCEPT
{
    LibraryAppletStartHookUserArg callbackArg;
    callbackArg.isExtremity = isExtremity;

    // ライブラリアプレットの起動
    if ( g_UseJumpLibrayApplet )
    {
        // nn::ae::SetLibraryAppletStartHook で設定したコールバックは呼ばれない。 Result も返さない。
        // ここで自プロセスは死ぬのでこの API から帰ってくることもない。
        nn::applet::JumpLibraryApplet( laHandle );
        NN_ABORT("should not be reached");
        return nn::ResultSuccess();
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::StartLibraryApplet( laHandle, &callbackArg ) );

    // ライブラリアプレットの終了確認
    nn::applet::JoinLibraryApplet(laHandle);
    nn::applet::LibraryAppletExitReason exit_reason = nn::applet::GetLibraryAppletExitReason(laHandle);

    abortIfAbnormalOrUnexpected( exit_reason );

    if( exit_reason == nn::applet::LibraryAppletExitReason_Canceled )
    {
        return nn::la::ResultLibraryAppletCanceled();
    }

    // ここまでで問題なければ、成功として結果を返す
    return nn::ResultSuccess();
}

//---------------------------------------------------------------------------
nn::Result StartLibraryAppletEasy(
    nn::applet::AppletId appletId,
    const nn::la::CommonArgumentsWriter& commonArg,
    const void* laParam, size_t laParamSize,
    void* pResult, size_t resultBufferSize, size_t* pResultSize,
    bool isExtremity ) NN_NOEXCEPT
{
    nn::applet::LibraryAppletHandle laHandle;
    nn::Result result = nn::la::CreateLibraryAppletEasy(
        &laHandle,
        appletId,
        nn::applet::LibraryAppletMode_AllForeground,
        isExtremity);

    if ( result.IsFailure() )
    {
        return result;
    }

    commonArg.PushToInChannel( laHandle );
    nn::la::PushToInChannel(laHandle, laParam, laParamSize);

    result = nn::la::StartLibraryApplet(laHandle, isExtremity );

    if (result.IsSuccess())
    {
        nn::la::PopFromOutChannel(laHandle, pResult, resultBufferSize, pResultSize);
    }

    nn::applet::CloseLibraryApplet(laHandle);

    return result;
}

//---------------------------------------------------------------------------
//
void RequestHomeMenu() NN_NOEXCEPT
{
    // SAはlsys::SystemAppletMessage::MessageReceiverでこれを解釈している。
    // lsysを使えるLAであれば、送信もこの関数ではなく、lsys::SystemAppletMessageで行える。
    // ここにデータ直書きもどうかとおもいつつ、簡単なものなのでひとまずこれで。
    static const uint8_t data[] = {
        0x53, 0x41, 0x4d, 0x53, // Magic code = SAMS
        0x01, 0x00, 0x00, 0x00, // Base version
        0x02, 0x00, 0x00, 0x00, // Type = RequestHomeMenu
        0x01, 0x00, 0x00, 0x00  // RequestHomeMenu version
    };
    static const size_t size = sizeof(data);
    nn::applet::StorageHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateStorage(&handle, size));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::WriteToStorage(handle, 0, data, size));
    nn::applet::PushToSystemGeneralChannel(handle);
}

//---------------------------------------------------------------------------
//
void RequestJumpToSystemUpdate() NN_NOEXCEPT
{
    // SAはlsys::SystemAppletMessage::MessageReceiverでこれを解釈している。
    // lsysを使えるLAであれば、送信もこの関数ではなく、lsys::SystemAppletMessageで行える。
    // ここにデータ直書きもどうかとおもいつつ、簡単なものなのでひとまずこれで。
    static const uint8_t data[] = {
        0x53, 0x41, 0x4d, 0x53, // Magic code = SAMS
        0x01, 0x00, 0x00, 0x00, // Base version
        0x0b, 0x00, 0x00, 0x00, // Type = RequestJumpToSystemUpdate
        0x01, 0x00, 0x00, 0x00  // RequestJumpToSystemUpdate version
    };
    static const size_t size = sizeof(data);
    nn::applet::StorageHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateStorage(&handle, size));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::WriteToStorage(handle, 0, data, size));
    nn::applet::PushToSystemGeneralChannel(handle);
}

//---------------------------------------------------------------------------

}} // namespace nn::la

