﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/jit/jit_Api.h>
#include "jit_Library.h"

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <cstring>
#include <utility>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/jitsrv/jitsrv_JitServices.sfdl.h>
#include <nn/svc/svc_Base.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_TransferMemory.h>

namespace nn { namespace jit {

namespace {

Result CreateCodeMemoryImpl(sf::NativeHandle* pOut, uintptr_t address, size_t size) NN_NOEXCEPT
{
    if (size == 0)
    {
        pOut->Reset();
        NN_RESULT_SUCCESS;
    }
    svc::Handle svcHandle;
    NN_RESULT_DO(svc::CreateCodeMemory(&svcHandle, address, size));
    *pOut = sf::NativeHandle(svcHandle.operator nnHandle().value, true);
    NN_RESULT_SUCCESS;
}

Result CreateJitEnvironmentImpl(sf::SharedPointer<jitsrv::IJitEnvironment>* pOut, jitsrv::IJitService* pService, const JitEnvironmentConfiguration& configuration, const JitPluginInfo& pluginInfo, void* workMemory, size_t workMemorySize) NN_NOEXCEPT
{
    const size_t MemoryAlignSize = 2 * 1024 * 1024;
    NN_ABORT_UNLESS_ALIGNED(configuration.rxCodeSize, MemoryAlignSize);
    NN_ABORT_UNLESS_ALIGNED(configuration.roCodeSize, MemoryAlignSize);
    NN_ABORT_UNLESS_ALIGNED(pluginInfo.pluginWorkMemorySize, MemoryAlignSize);
    NN_ABORT_UNLESS_ALIGNED(workMemory, MemoryAlignSize);
    NN_ABORT_UNLESS_LESS_EQUAL(configuration.rxCodeSize + configuration.roCodeSize + pluginInfo.pluginWorkMemorySize, workMemorySize);

    auto current = reinterpret_cast<uintptr_t>(workMemory);
    auto rxBuffer = current;
    current += configuration.rxCodeSize;
    auto roBuffer = current;
    current += configuration.roCodeSize;
    auto pluginWorkBuffer = current;
    current += pluginInfo.pluginWorkMemorySize;
    NN_UNUSED(current);

    auto success = false;
    NN_UNUSED(success);
    sf::SharedPointer<jitsrv::IJitEnvironment> pJitEnvironment;

    {
        sf::NativeHandle rxHandle;
        NN_RESULT_DO(CreateCodeMemoryImpl(&rxHandle, rxBuffer, configuration.rxCodeSize));
        sf::NativeHandle roHandle;
        NN_RESULT_DO(CreateCodeMemoryImpl(&roHandle, roBuffer, configuration.roCodeSize));
        NN_RESULT_DO(pService->CreateJitEnvironment(&pJitEnvironment, sf::NativeHandle(svc::PSEUDO_HANDLE_CURRENT_PROCESS.value, false), std::move(rxHandle), configuration.rxCodeSize, std::move(roHandle), configuration.roCodeSize));
    }
    {
        sf::InBuffer nrr{reinterpret_cast<const char*>(pluginInfo.nrr), pluginInfo.nrrSize};
        sf::InBuffer nro{reinterpret_cast<const char*>(pluginInfo.nro), pluginInfo.nroSize};
        os::TransferMemory pluginMemory{reinterpret_cast<void*>(pluginWorkBuffer), pluginInfo.pluginWorkMemorySize, os::MemoryPermission_None};
        sf::NativeHandle pluginMemoryHandle{pluginMemory.Detach(), true};
        NN_RESULT_TRY(pJitEnvironment->LoadPlugin(nrr, nro, std::move(pluginMemoryHandle), pluginInfo.pluginWorkMemorySize))
            NN_RESULT_CATCH(jit::ResultPluginVersionMismatch)
            {
                NN_SDK_LOG("jit error: jit plugin version mismatch [JIT-PluginVersionMismatch]\n");
                NN_RESULT_RETHROW;
            }
            NN_RESULT_CATCH(jit::ResultOutOfMemoryForDllLoad)
            {
                NN_SDK_LOG("jit error: out of memory for DLL loading [JIT-OutOfMemoryForDllLoad]\n");
                NN_RESULT_RETHROW;
            }
            NN_RESULT_CATCH_ALL
            {
                NN_SDK_LOG("jit error: LoadPlugin() failed [JIT-Failed-LoadPlugin]:%08x\n", NN_RESULT_CURRENT_RESULT.GetInnerValueForDebug());
                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY
    }

    success = true;
    *pOut = std::move(pJitEnvironment);
    NN_RESULT_SUCCESS;
}

int ControlImpl(jitsrv::IJitEnvironment* p, uint64_t tag, const void* inData, size_t inDataSize, void* outBuffer, size_t outBufferSize) NN_NOEXCEPT
{
    int ret;
    NN_RESULT_TRY(p->Control(&ret, tag, sf::InBuffer(static_cast<const char*>(inData), inDataSize), sf::OutBuffer(static_cast<char*>(outBuffer), outBufferSize)))
        NN_RESULT_CATCH(jit::ResultPluginUndefinedCall)
        {
            NN_SDK_LOG("jit error: nn::jit::plugin::Control is not defined [JIT-Undefined-Control]\n");
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_SDK_LOG("jit error: Control() failed [JIT-Failed-Control]:%08x\n", NN_RESULT_CURRENT_RESULT.GetInnerValueForDebug());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY
    return ret;
}

int GenerateCodeImpl(CodeRange* pGeneratedRx, CodeRange* pGeneratedRo, jitsrv::IJitEnvironment* p, uint64_t tag, const void* source, size_t sourceSize, const CodeRange& rxBuffer, const CodeRange& roBuffer, const void* inData, size_t inDataSize, void* outBuffer, size_t outBufferSize) NN_NOEXCEPT
{
    jitsrv::Struct32 inStruct = {};
    static_assert(sizeof(inStruct) == 32, "");
    NN_SDK_REQUIRES(inDataSize <= sizeof(inStruct), "inDataSize must be <= 32");
    std::memcpy(&inStruct, inData, inDataSize);
    int ret;
    NN_RESULT_TRY(p->GenerateCode(&ret, pGeneratedRx, pGeneratedRo, sf::InBuffer(static_cast<const char*>(source), sourceSize), tag, rxBuffer, roBuffer, inStruct, inDataSize, sf::OutBuffer(static_cast<char*>(outBuffer), outBufferSize)))
        NN_RESULT_CATCH(jit::ResultPluginInvalidRxOut)
        {
            NN_SDK_LOG("jit error: plugin returns invalid RX range [JIT-PluginInvalidRxOut]\n");
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH(jit::ResultPluginInvalidRoOut)
        {
            NN_SDK_LOG("jit error: plugin returns invalid RO range [JIT-PluginInvalidRoOut]\n");
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH(jit::ResultPluginPostConditionViolation)
        {
            NN_SDK_LOG("jit error: plugin violates post condition [JIT-PluginPostConditionViolation]\n");
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_SDK_LOG("jit error: GenerateCode() failed [JIT-Failed-GenerateCode]:%08x\n", NN_RESULT_CURRENT_RESULT.GetInnerValueForDebug());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY
    return ret;
}

} // unnamed

Result CreateJitEnvironment(JitHandle* pOut, const JitEnvironmentConfiguration& configuration, const JitPluginInfo& pluginInfo, void* workMemory, size_t workMemorySize) NN_NOEXCEPT
{
    auto success = false;
    auto handle = NewJitEnvironment();
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            CloseJitEnvironment(handle);
        }
    };

    sf::SharedPointer<jitsrv::IJitEnvironment> p;
    NN_RESULT_DO(CreateJitEnvironmentImpl(&p, GetJitService().Get(), configuration, pluginInfo, workMemory, workMemorySize));

    uint64_t rxAddress;
    uint64_t roAddress;
    NN_ABORT_UNLESS_RESULT_SUCCESS(p->GetCodeAddress(&rxAddress, &roAddress));

    success = true;
    auto pImpl = GetImpl(handle);
    pImpl->info.rxCodeAddress = rxAddress;
    pImpl->info.rxCodeSize = configuration.rxCodeSize;
    pImpl->info.roCodeAddress = roAddress;
    pImpl->info.roCodeSize = configuration.roCodeSize;
    pImpl->p = p.Detach();
    *pOut = handle;
    NN_RESULT_SUCCESS;
}

void DestroyJitEnvironment(JitHandle handle) NN_NOEXCEPT
{
    auto pImpl = GetImpl(handle);
    sf::ReleaseSharedObject(pImpl->p);
    CloseJitEnvironment(handle);
}

JitEnvironmentInfo GetJitEnvironmentInfo(JitHandle handle) NN_NOEXCEPT
{
    return GetImpl(handle)->info;
}

int Control(JitHandle handle, uint64_t tag, const void* inData, size_t inDataSize, void* outBuffer, size_t outBufferSize) NN_NOEXCEPT
{
    auto p = GetImpl(handle)->p;
    return ControlImpl(p, tag, inData, inDataSize, outBuffer, outBufferSize);
}

namespace {

    template <typename F>
    inline void IterateCacheLine(uintptr_t addr, size_t size, F f) NN_NOEXCEPT
    {
        if (size == 0)
        {
            return;
        }
        uintptr_t cacheTypeRegister;
        asm volatile ("mrs %0, ctr_el0" : "=r" (cacheTypeRegister));
        size_t cacheLineSize = 4 << ((cacheTypeRegister >> 16) & 0xF);
        auto startAddr = reinterpret_cast<uintptr_t>(addr) & ~(cacheLineSize - 1);
        auto endAddr = reinterpret_cast<uintptr_t>(addr) + size;
        for (auto p = startAddr; p < endAddr; p += cacheLineSize)
        {
            f(p);
        }
    }

    void InvalidateInstructionCacheAndMemoryBarrier(uintptr_t addr, size_t size) NN_NOEXCEPT
    {
        // 命令キャッシュ無効化
        IterateCacheLine(addr, size, [](uintptr_t p)
        {
            asm volatile ("ic ivau, %0"::"r"(p):"memory");
        });
        // ish メモリバリア
        asm volatile ("dsb ish":::"memory");
        // パイプラインフラッシュ
        asm volatile ("isb":::"memory");
    }

    void FlushDataCache(uintptr_t addr, size_t size) NN_NOEXCEPT
    {
        IterateCacheLine(addr, size, [](uintptr_t p)
        {
            asm volatile ("dc civac, %0" : : "r" (p));
        });
        asm volatile ("dsb sy" : : : "memory");
    }

}

int GenerateCode(CodeRange* pGeneratedRx, CodeRange* pGeneratedRo, JitHandle handle, uint64_t tag, const void* source, size_t sourceSize, const CodeRange& rxBuffer, const CodeRange& roBuffer, const void* inData, size_t inDataSize, void* outBuffer, size_t outBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(rxBuffer.IsValid());
    NN_SDK_REQUIRES(roBuffer.IsValid());
    auto pImpl = GetImpl(handle);
    auto p = pImpl->p;
    auto ret = GenerateCodeImpl(pGeneratedRx, pGeneratedRo, p, tag, source, sourceSize, rxBuffer, roBuffer, inData, inDataSize, outBuffer, outBufferSize);
    InvalidateInstructionCacheAndMemoryBarrier(pImpl->info.rxCodeAddress + pGeneratedRx->offset, pGeneratedRx->size);
    FlushDataCache(pImpl->info.roCodeAddress + pGeneratedRo->offset, pGeneratedRo->size);
    return ret;
}

}}
