﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>
#include "idle_HandlerImpl.h"

namespace nn { namespace idle { namespace server {

    // 無操作時間カウンタが更新されるごとに適切なタイミングでハンドラの NotifyActivity を呼ぶことを責任とするクラス
    class InactivityCounter
    {
    public:
        InactivityCounter() NN_NOEXCEPT :
            m_LastActiveTime(0),
            m_pHandlerImpl(nullptr)
        {}

        void Initialize(HandlerImpl* pHandlerImpl, const nn::TimeSpan& initialStartTime) NN_NOEXCEPT
        {
            m_pHandlerImpl = pHandlerImpl;
            m_LastActiveTime = initialStartTime;
        }

        // 最終操作時刻が更新されたら true を返す
        bool InputLastActiveTime(const nn::TimeSpan& newLastActiveTime) NN_NOEXCEPT
        {
            if ( 0 < newLastActiveTime - m_LastActiveTime )
            {
                // NN_DETAIL_IDLE_TRACE("UpdateIdleCountAndNotifyActivity newBase=%lld ms\n", newBase.GetMilliSeconds());
                m_LastActiveTime = newLastActiveTime;
                m_pHandlerImpl->NotifyActivity();
                return true;
            }
            return false;
        }

        nn::TimeSpan GetInactiveTime(const nn::TimeSpan& nowTime) NN_NOEXCEPT
        {
            return nowTime - m_LastActiveTime;
        }


    private:
        // 最終アクティブ時刻 = 無操作をカウントするベース時刻
        // 操作検出によって更新されるほか、ハンドリングコンテキスト変化などの要因でも更新される
        // 更新時に無操作ハンドラの NotifyActivity を呼ぶ
        nn::TimeSpan m_LastActiveTime;

        // 無操作状態ハンドラ実装へのアクセサ
        HandlerImpl* m_pHandlerImpl;
    };

}}} // namespace nn::idle::server

