﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/http/stream/http_ClientCertificate.h>
#include <nn/http/http_Result.h>

#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_Mount.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/nn_SdkLog.h>

/* HostFS から証明書を読む */

namespace nn { namespace http { namespace stream {

#if defined(NN_HTTP_HFIO_CERT_ENABLE)

namespace
{
struct Mutex
{
    os::MutexType _lock;
    void lock() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!os::IsMutexLockedByCurrentThread(&_lock));
        os::LockMutex(&_lock);
    }
    void unlock() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(os::IsMutexLockedByCurrentThread(&_lock));
        os::UnlockMutex(&_lock);
    }
};
Mutex g_Lock = {NN_OS_MUTEX_INITIALIZER(false)};
const char MountName[] = "certvol";
const char MountPoint[] = "C:\\siglo_ssl";
const char CertPath[] = "certvol:/client_cert.p12";
const char PasswordPath[] = "certvol:/client_cert.key";
} // ~namespace nn::http::stream::<anonymous>

Result LoadClientCerificate(
    char** pOutCert, size_t* pOutCertSize,
    char** pOutPassword, size_t* pOutPasswordSize,
    const CertIoBuffer& buffer) NN_NOEXCEPT
{
    std::lock_guard<Mutex> lock(g_Lock);

    NN_RESULT_DO(fs::MountHost(MountName, MountPoint));
    NN_UTIL_SCOPE_EXIT
    {
        fs::Unmount(MountName);
    };

    fs::FileHandle certFile;
    int64_t certSize;
    fs::FileHandle passwordFile;
    int64_t passwordSize;

    /* ファイル長の検査 */
    NN_RESULT_DO(fs::OpenFile(&certFile, CertPath, fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT { nn::fs::CloseFile(certFile); };
    NN_RESULT_DO(fs::GetFileSize(&certSize, certFile));

    NN_RESULT_DO(fs::OpenFile(&passwordFile, PasswordPath, fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT { nn::fs::CloseFile(passwordFile); };
    NN_RESULT_DO(fs::GetFileSize(&passwordSize, passwordFile));

    NN_RESULT_THROW_UNLESS(static_cast<size_t>(certSize + passwordSize + 2) <= buffer.size, ResultInsufficientBuffer());

    /* 読み込み */
    char* cert = buffer.pointer;
    char* password = cert + certSize + 1;
    NN_RESULT_DO(fs::ReadFile(certFile, 0, cert, static_cast<size_t>(certSize)));
    NN_RESULT_DO(fs::ReadFile(passwordFile, 0, password, static_cast<size_t>(passwordSize)));
    cert[certSize] = password[passwordSize] = '\0';

    /* 出力 */
    *pOutCert = cert;
    *pOutCertSize = static_cast<size_t>(certSize);
    *pOutPassword = password;
    *pOutPasswordSize = static_cast<size_t>(passwordSize);
    NN_RESULT_SUCCESS;
}

#endif

}}} // ~namespace nn::http::stream
