﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/http.h>

#include <nn/nn_SdkLog.h>
#include <curl/curl.h>

#include "http_Utility.h"

namespace nn {
namespace http {

Progress::Progress()
    : m_bytesTransferredCurrent(0)
    , m_bytesTransferredLast(0)
    , m_bytesTotal(0)
    , m_bytesLastThroughput(0)
    , m_tickStart(0)
    , m_tickLast(0)
    , m_tickEnd(0)
{
}

int8_t Progress::CalculateProgressPercent() const
{
    if (m_bytesTransferredCurrent == 0 || m_bytesTotal == 0)
    {
        return 0;
    }
    else
    {
        return static_cast<uint8_t>((m_bytesTransferredCurrent * 100) / m_bytesTotal);
    }
}

TimeSpan Progress::CalculateEstimatedArraivalTime() const
{
    if (m_bytesLastThroughput <= 0 || m_bytesTotal <= 0)
    {
        return TimeSpan(0);
    }
    uint64_t bytesRemain = m_bytesTotal - m_bytesTransferredCurrent;
    return TimeSpan::FromMilliSeconds(bytesRemain * 1000 / m_bytesLastThroughput);
}

TimeSpan Progress::CalculateElapsedTime() const
{
    if (m_tickStart.GetInt64Value() == 0)
    {
        return TimeSpan(0);
    }
    else
    {
        os::Tick tickCurrent = m_tickEnd.GetInt64Value() > 0 ?
            m_tickEnd : os::GetSystemTick();
        return (tickCurrent - m_tickStart).ToTimeSpan();
    }
}

uint32_t Progress::GetLastTroughput() const
{
    return m_bytesLastThroughput;
}

uint32_t Progress::CalculateAverageTroughput() const
{
    if (m_bytesTransferredCurrent == 0)
    {
        return 0;
    }

    const TimeSpan timeElapsed = CalculateElapsedTime();
    if (timeElapsed > 0)
    {
        return static_cast<uint32_t>((m_bytesTransferredCurrent * 1000) / timeElapsed.GetMilliSeconds());
    }
    else
    {
        return 0;
    }
}

uint64_t Progress::GetTransferredBytes() const
{
    return m_bytesTransferredCurrent;
}

void Progress::PrintProgressBar()
{
    uint8_t percent = CalculateProgressPercent();
    char bar[50 + 1] = {};
    uint8_t len = percent / 2;
    for (int i = 0; i < 50; ++i)
    {
        if (i < len)
        {
            bar[i] = '=';
        }
        else if (i == len)
        {
            bar[i] = '>';
        }
        else
        {
            if (i > 1 && i < 50 && (i % 10 == 0))
            {
                bar[i] = '+';
            }
            else
            {
                bar[i] = '-';
            }
        }
    }
    const TimeSpan timeEta = CalculateEstimatedArraivalTime();
    NN_SDK_LOG("%3u%% [%s] % 9llu/% 9llu % 5uKB/s   ETA %02lld:%02lld\n",
        percent, bar,
        m_bytesTransferredCurrent, m_bytesTotal, m_bytesLastThroughput / 1000,
        timeEta.GetMinutes(), timeEta.GetSeconds() % 60
    );
    NN_UNUSED(timeEta);
}

void Progress::Update(uint64_t bytesTransferred, uint64_t bytesTotal)
{
    m_bytesTotal = bytesTotal;
    m_bytesTransferredCurrent = bytesTransferred;

    const os::Tick tickCurrent = os::GetSystemTick();
    if (m_tickStart.GetInt64Value() == 0)
    {
        m_tickStart = tickCurrent;
    }
    if (bytesTransferred == bytesTotal)
    {
        m_tickEnd = tickCurrent;
    }

    const TimeSpan timePassed = (tickCurrent - m_tickLast).ToTimeSpan();
    if (timePassed > TimeSpan::FromSeconds(1))
    {
        const uint64_t bytesTransferredDelta = m_bytesTransferredCurrent - m_bytesTransferredLast;
        if (bytesTransferredDelta > 0 && timePassed.GetMilliSeconds() > 0)
        {
            m_bytesLastThroughput = static_cast<uint32_t>((bytesTransferredDelta * 1000) / timePassed.GetMilliSeconds());
        }
        else
        {
            m_bytesLastThroughput = 0;
        }

        m_tickLast = tickCurrent;
        m_bytesTransferredLast = m_bytesTransferredCurrent;
    }
}

}
} // ~namespace nn::http
