﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Result.h>
#include <nn/nn_Allocator.h>
#include <nn/htcs/htcs_Types.h>
#include <nn/os.h>

namespace nn { namespace htcs { namespace detail {

struct virtual_socket;

//======================================================================================
// Collection of virtual socket primitives.  This is layer of virtual sockets fixes
// OASIS-371, "Some of htcs APIs should succeed even when TMS is not connected".
class virtual_socket_collection
{
public:
    explicit            virtual_socket_collection();
                        ~virtual_socket_collection();

    static size_t           GetWorkingMemorySize( int socketCountMax )                      NN_NOEXCEPT;

    //==============================================================================
    // Management functionality
    void                    Init        ( void*                         buffer,
                                          size_t                        bufferSize )        NN_NOEXCEPT;

    void                    Clear       ()                                                  NN_NOEXCEPT;

    //===============================================================================
    // These need to always succeed
    int                     Socket      ( int                           &ErrorCode )        NN_NOEXCEPT;

    int                     Close       ( int                           Id,
                                          int                           &ErrorCode )        NN_NOEXCEPT;

    int                     Bind        ( int                           Id,
                                          const nn::htcs::SockAddrHtcs* pAddr,
                                          int                           &ErrorCode )        NN_NOEXCEPT;

    int                     Listen      ( int                           Id,
                                          int                           backlogCount,
                                          int&                          ErrorCode )         NN_NOEXCEPT;

    int                     Accept      ( int                           Id,
                                          nn::htcs::SockAddrHtcs*       address,
                                          int&                          ErrorCode )         NN_NOEXCEPT;

    int                     Fcntl       ( int                           Id,
                                          int                           command,
                                          int                           value,
                                          int&                          ErrorCode )         NN_NOEXCEPT;

    //===============================================================================
    // These can fail if the network is down, but only because they rely on
    // other functionality that can fail if the network is down.
    int                     Shutdown    ( int                           Id,
                                          int                           how,
                                          int&                          ErrorCode )         NN_NOEXCEPT;

    //===============================================================================
    // These can fail if the network is down.
    nn::htcs::ssize_t       Recv        ( int                           Id,
                                          void*                         buffer,
                                          size_t                        bufferByteSize,
                                          int                           flags,
                                          int&                          ErrorCode )         NN_NOEXCEPT;

    nn::htcs::ssize_t       Send        ( int                           Id,
                                          const void*                   buffer,
                                          size_t                        bufferByteSize,
                                          int                           flags,
                                          int&                          ErrorCode )         NN_NOEXCEPT;


    int                     Connect     ( int                           Id,
                                          const nn::htcs::SockAddrHtcs* address,
                                          int&                          ErrorCode )         NN_NOEXCEPT;

    //======================================================================================

private:
    void*                   m_Buffer;       // Memory management
    size_t                  m_BufferSize;   // Memory management

    virtual_socket*         m_pList;        // Entries
    int32_t                 m_ListCount;    // Number of valid entries in the list.
    int32_t                 m_ListSize;     // Size of the list.
    int                     m_NextId;       // Next valid Id
    nn::os::MutexType       m_Mutex;        // Mutex to protect all of our resource lists

    //======================================================================================

    int                     CreateId    ()                                                  NN_NOEXCEPT;

    //======================================================================================
    // List maintenance functionality.
    int                     Add         ( nn::sf::SharedPointer<nn::tma::ISocket> FromSocket ) NN_NOEXCEPT;
    void                    Insert      ( int                           Id,
                                          nn::sf::SharedPointer<nn::tma::ISocket> pSocket ) NN_NOEXCEPT;
    void                    SetSize     ( int32_t                       Size )              NN_NOEXCEPT;
    int                     Find        ( int                           Id,
                                          int*                          pErrorCode = NULL ) NN_NOEXCEPT;
    bool                    HasAddr     ( const nn::htcs::SockAddrHtcs*       pAddress )    NN_NOEXCEPT;

    //======================================================================================
    // Socket accessing

    nn::sf::SharedPointer<nn::tma::ISocket> GetSocket  ( int            Id,
                                                         int*           pErrorCode = NULL ) NN_NOEXCEPT;

    nn::sf::SharedPointer<nn::tma::ISocket> FetchSocket( int            Id,
                                                         int&           ErrorCode )         NN_NOEXCEPT;

    //======================================================================================
    // Memory management functionality
    //======================================================================================

    // Socket creation and destruction
    int                     CreateSocket( nn::sf::SharedPointer<nn::tma::ISocket> FromSocket,
                                          int&                          ErrorCode )         NN_NOEXCEPT;

    enum
    {
        collection_InitialListCount = 32
    };

};

}}}
